#! /usr/bin/python3
# Last edited on 2024-09-18 10:31:58 by stolfi
 
from math import sin, cos, tan, sqrt, pi, inf, floor, sqrt
import rn
import sys
import re

from slicing_lib import make_edge, make_face, check_repeated_edge, prte, prtf

def make_edges_faces_orig(Ns, Nr, Nb, Vind, Vlst, prec, verbose):
  # See {make_vertices} for the description of the object.
  #
  # Builds lists of the faces and edges of the object, given the 
  # vertices in structured form {Vind} and flat list {Vlst}.
  #
  # Returns the records representing the faces as a flatlist {Flst} with
  # indices from 1 to {Nf}, where {nf} is the total number of faces.
  # Element {Flst[0]} is not used. See {make_face} for the face record
  # format.
  #
  # The edges are returned as single flat list {Elst} indexed from 1 to
  # {Ne}, Element {Elst[0]} is not used. See {make_edge} for the edge
  # record format. (This module generates only edges with {etype=0}.)
  # 
  # The label of a face on the side of the barrel has the form "fv.{ks}.{kr}"
  # where {ks} ranges in {0..Ns-1} and {kr} ranges in {0..Nr-1}.
  # 
  # The top and bottom faces of the barrel have the form "fh.{ks}.{kz}"
  # where {kz} is 0 for bottom and 1 for top.

  # Output element counts:
  Nv = Ns*(Nr*(Nb+1) + 1) + 2   # Expected total number of vertices.
  Ne = Ns*(Nr*(Nb+2) + 3)       # Expected total number of edges.
  Nf = Ns*(Nr + 2)              # Expected total number of faces.

  assert Nv == len(Vlst)-1, "{Nv,Vlst} inconsistent"

  sys.stderr.write("make_edges_faces_orig:\n")
  sys.stderr.write("  expecting %d edges\n" % Ne)
  sys.stderr.write("  expecting %d faces\n" % Nf)
  
  # Variables that are global to the nested functions:
  Fiv = []     # Indices of vertices of current face.
  Flab = "???" # Current face label.
  nf = 0;      # Number of faces found so far.
  ne = 0;      # Number of edges found so far.
  debug_face = False
  
  Flst = [None]*(Nf+1)
  Elst = [None]*(Ne+1)
  
  Eset = set()  # Set of edges as pairs, to check for repetitions.
 
  def Sved(kv_org, kv_dst):
    # Saves the edge from {Vlst[kv_org]} to {Vlst[kv_dst]} in {Elst}, if traversed 
    # in increasing vertex index sense. 
    nonlocal Fiv, Flab, nf, ne, debug_face, Elst, Flst
    elab = Vlst[kv_org][4] + "--" + Vlst[kv_dst][4] + ":" + Flab
    etype = 0
    e = make_edge(Vlst[kv_org], Vlst[kv_dst], ne+1, etype, elab, prec)
    if e != None:
      if verbose: prte(e, prec)
      # Check for repeated edges: */
      ne += 1
      Elst[ne] = e
      check_repeated_edge(ne, Elst, Eset, prec)
    return None

  def Ptit(tit, debug = False):
    # Title for a face or set of faces.
    nonlocal Fiv, Flab, nf, ne, debug_face, Elst, Flst
    if (verbose): sys.stderr.write("  defining %s \n" % (tit));
    debug_face = debug
  
  def Bof(flab,debug = False):
    #  Start of a new face.
    nonlocal Fiv, Flab, nf, ne, debug_face, Elst, Flst
    assert len(Fiv) == 0
    Flab = flab
    if (verbose): sys.stderr.write("\n");
    debug_face = debug
  
  def Addv(ki):
    # Adds vertex with OBJ index {ki} to the current face.
    nonlocal Fiv, Flab, nf, ne, debug_face, Elst, Flst
    p = Vlst[ki]
    assert p[3] == ki
    if debug_face:
      sys.stderr.write("    v%4d = ( %9.*f %9.*f %9.*f ) %s\n" % (ki, prec, p[0], prec, p[1], prec, p[2], p[4]));
    # Saves vertex in list {Fiv} of face vertices:
    Fiv.append(ki);
    return None
    
  def Eof(order):
    # Close current face with edge from last to first vertex:
    nonlocal Fiv, Flab, nf, ne, debug_face, Elst, Flst
    deg = len(Fiv)
    assert deg >= 3, "face has only %d corners" % deg
    if order == -1:
      # Reverse vertex list:
      Fiv = [ Fiv[deg-1-k] for k in range(deg) ]
    nf += 1;
    f = make_face(Fiv, nf, Flab, Vlst, prec)
    if verbose or debug_face: prtf(f, Vlst, prec)
    Flst[nf] = f;
    # Save the edges:
    for jv in range(deg):
      jv1 = (jv+1) % deg
      Sved(Fiv[jv], Fiv[jv1])
    Fiv = []
    
  # CREATE THE FACES:

  for ks in range(Ns):
    ksp = (ks+1) % Ns
    for kr in range(Nr):
      krp = kr+1

      Ptit("Barrel side face -- ks = %d  kr = %d" % (ks,kr))
      Bof(f"fv.{ks}.{kr}")
      Addv(Vind['vp'][ks][kr])
      Addv(Vind['vp'][ksp][kr])
      for kb in range(Nb):
        Addv(Vind['ve'][ksp][kr][kb])
      Addv(Vind['vp'][ksp][krp])
      Addv(Vind['vp'][ks][krp])
      for kb in range(Nb):
        Addv(Vind['ve'][ks][kr][Nb-1-kb])
      Eof(+1)

    for kz in range(2):
      zdir = 2*kz - 1
      kr = 0 if kz == 0 else Nr
      Ptit("horizontal face -- ks = %d kz = %d" % (ks, kz))
      Bof(f"fh.{ks}.{kz}")
      Addv(Vind['vp'][ks][kr])
      Addv(Vind['vp'][ksp][kr])
      Addv(Vind['vc'][kz])
      Eof(zdir)

  if nf < Nf: sys.stderr.write("!! missing some faces\n")
  assert nf <= Nf

  if ne < Ne: sys.stderr.write("!! missing some edges\n")
  assert ne <= Ne

  return Elst, Flst
