#include <iostream>
#include "shapetor-options.hpp"

namespace fs	= std::filesystem;
namespace opt	= boost::program_options;
using paths	= shapetor_options::paths;

using shape_representation = shapetor_options::shape_representation;

/*
 * inner classes implementation
 * ----------------------------
 */

shapetor_options::usage_error::usage_error(std::string const& msg)
	: std::runtime_error(msg)
{}

/*
 * auxiliar functions
 * ------------------
 */

static std::string quote(std::string const& str)
{
	return '"' + str + '"';
}

template<class str_iterator, class ext_iterator>
static bool ends_with(
	str_iterator str_begin, str_iterator str_end,
	ext_iterator ext_begin, ext_iterator ext_end)
{
	auto str_len = std::distance(str_begin, str_end);
	auto ext_len = std::distance(ext_begin, ext_end);

	if(str_len < ext_len)
		return false;
	if(ext_len == 0)
		return true;

	auto ends = std::prev(str_end, ext_len);
	return std::equal(ends, str_end, ext_begin, ext_end);
}

static bool ends_with(std::wstring const& str, std::string const& ext)
{
	return ends_with(begin(str), end(str), begin(ext), end(ext));
}

static std::string const& to_string(shapetor_options::cmd cmd)
{
	static std::string const none		= "none";
	static std::string const convert	= "convert";
	static std::string const union_str	= "union";
	static std::string const minus		= "minus";
	static std::string const intersection	= "intersection";

	switch(cmd){
	case shapetor_options::cmd::NONE:
		return none;
		break;
	case shapetor_options::cmd::CONVERT:
		return convert;
		break;
	case shapetor_options::cmd::UNION:
		return union_str;
		break;
	case shapetor_options::cmd::MINUS:
		return minus;
		break;
	case shapetor_options::cmd::INTERSECTION:
		return intersection;
		break;
	}

	return none;
}

static std::string const& to_string(shape_representation rep)
{
	static std::string const none		= "none";
	static std::string const binary		= "binary";
	static std::string const cdt2d		= "cdt2d";
	static std::string const polygon	= "polygon";

	switch(rep){
	case shape_representation::NONE:
		return none;
		break;
	case shape_representation::BINARY:
		return binary;
		break;
	case shape_representation::CDT2D:
		return cdt2d;
		break;
	case shape_representation::POLYGON:
		return polygon;
		break;
	}

	return none;
}

static shapetor_options::cmd to_command(std::string const& cmd)
{
	if(cmd == to_string(shapetor_options::cmd::CONVERT))
		return shapetor_options::cmd::CONVERT;
	if(cmd == to_string(shapetor_options::cmd::UNION))
		return shapetor_options::cmd::UNION;
	if(cmd == to_string(shapetor_options::cmd::MINUS))
		return shapetor_options::cmd::MINUS;
	if(cmd == to_string(shapetor_options::cmd::INTERSECTION))
		return shapetor_options::cmd::INTERSECTION;

	return shapetor_options::cmd::NONE;
}


static shape_representation to_shape_representation(std::string const& rep)
{
	if(rep == to_string(shape_representation::BINARY))
		return shape_representation::BINARY;
	if(rep == to_string(shape_representation::CDT2D))
		return shape_representation::CDT2D;
	if(rep == to_string(shape_representation::POLYGON))
		return shape_representation::POLYGON;

	return shape_representation::NONE;
}

static shape_representation parse_shape_representation(std::string const& rep)
{
	auto tmp = to_shape_representation(rep);

	if(tmp == shape_representation::NONE){
		throw shapetor_options::usage_error(
				"Unknown shape representation " + rep
				);
	}

	return tmp;
}

/**
  * try to deduce the shape representation by the filetype
  */
static shape_representation deduce_shape_representation(
	std::wstring const& filepath)
{
	if(ends_with(filepath, ".polygon"))
		return shape_representation::POLYGON;
	return shape_representation::NONE;
}

/*
 * Description functions
 * ---------------------
 * 
 * Functions with describe_* are functions that returns a
 * opt::options_description. The final goal of this implementation
 * is to parse shapetor command line in the following format
 *
 * shapetor [general-options] cmd [cmd-options] input-paths
 *
 * where cmd can be convert, union, minus, etc.
 *
 * The functions are separated in describe_*_options and
 * describe_*_positional_options because of the intrinsics of
 * the boost::program_options library.
 *
 */

static opt::options_description describe_general_options()
{
	opt::options_description general("General options");
	general.add_options()
		("help,h",	"Show this message.")
		("version,v",	"Print the version of the program.")
		("verbose,V",	"Enable the verbose mode for shapetor.")
		("force,f",	"Force the overwriting of an existing "
		 		"output file.");

	return general;
}

static auto describe_general_positional_options()
{
	opt::options_description general("General positional options");

	general.add_options()
		(
			"cmd",
			opt::value<std::string>()->required(),
			"shapetor command"
		)
		(
			"cmdargs",
			opt::value<std::vector<std::string>>(),
			"shapetor command arguments"
		);

	opt::positional_options_description positional;
	positional
		.add("cmd", 1)
		.add("cmdargs", -1);

	return std::tuple{ general, positional };
}

static opt::options_description describe_convert_options()
{
	opt::options_description convert("convert command options");
	convert.add_options()
		(
			"delta,d",
			opt::value<shapetor_options::delta_type>(),
			"Sets the delta for the cdt2d in milimeters."
		)
		(
			"reduce-by,k",
			opt::value<shapetor_options::k_type>()
				->default_value(1),
			"Proportion to reduce the input image when converting"
			" to cdt2d."
		)
		(
		 	"from",
			opt::value<std::string>(),
			"Convertion-from representation type"
			" arg = { binary, cdt2d, polygon }. The polygon type"
			" is assumed to be in represented in millimeters."
		)
		(
			"to",
			opt::value<std::string>(),
			"Convertion-to representation type"
			" arg = { binary, cdt2d, polygon }. The polygon type"
			" is assumed to be in represented in millimeters."
		)
		(
			"ppcm,p",
			opt::value<shapetor_options::resolution_rep>()
				->default_value(10),
			"Resolution of the convertion from boundary "
			"representation in pixel per centimeter."
		)
		(
			"margin,m",
			opt::value<shapetor_options::margin_type>()
				->default_value(10u),
			"Empty space margin around the object to consider"
			" in image representation when converting"
			" polygon -> some-image-representation. In cdt2d "
			"representation, the margin is consider to be "
			"delta + margin parameter."
		);
	return convert;
}

static auto describe_multiple_path_positional_options()
{
	opt::options_description desc;

	desc.add_options()
		("paths", opt::value<paths>()->required(), "input paths");

	opt::positional_options_description convert_positional;
	convert_positional.add("paths", -1);

	return std::tuple{ desc, convert_positional };
}

/*
 * constructors
 * ------------
 */

shapetor_options::shapetor_options(int const argc, char const* const* argv)
{
	std::copy(argv, argv + argc, std::back_inserter(m_args));

	opt::options_description to_be_parsed;

	auto general = describe_general_options();

	auto [general_positional, general_positional_description] =
		describe_general_positional_options();

	to_be_parsed.add(general).add(general_positional);

	auto parsed = opt::command_line_parser(argc, argv)
		.options(to_be_parsed)
		.positional(general_positional_description)
		.allow_unregistered()
		.run();

	opt::variables_map vm;
	opt::store(parsed, vm);

	if(vm.count("verbose"))
		m_verbose = true;

	if(vm.count("help")){
		m_state = parsing_state::HELP;
	}else if(vm.count("version")){
		m_state = parsing_state::VERSION;
	}else{
		if(vm.count("force"))
			m_force = true;

		// notify if there is a required option missing
		opt::notify(vm);
		std::string cmdstr = vm["cmd"].as<std::string>();
		m_command = to_command(cmdstr);

		// collect all the suboptions
		auto suboptions = opt::collect_unrecognized(
			parsed.options,
			opt::include_positional
		);
		// remove the command string
		suboptions.erase(suboptions.begin());

		// parse options from the commands
		switch(m_command){
		case cmd::NONE:
			throw shapetor_options::usage_error(
				"command " + quote(cmdstr) + " is unknown"
			);
			break;
		case cmd::CONVERT:
			parse_convert_cmd(suboptions);
			break;
		case cmd::UNION:
		case cmd::MINUS:
		case cmd::INTERSECTION:
			throw std::runtime_error(
				"command "
				+ quote(cmdstr)
				+ " is not implemented"
			);
			break;
		}

		m_state = parsing_state::OPTIONS_READY;
	}
}

/*
 * parsing commands
 * ----------------
 */

void shapetor_options::parse_convert_cmd(std::vector<std::string> const& opt)
{
	auto convert_options_description = describe_convert_options();

	auto [convert_positional_description, convert_positional] = 
		describe_multiple_path_positional_options();

	opt::options_description convert_options;
	convert_options
		.add(convert_options_description)
		.add(convert_positional_description);

	auto parsed = opt::command_line_parser(opt)
		.options(convert_options)
		.positional(convert_positional)
		.run();

	opt::variables_map vm;
	opt::store(parsed, vm);
	opt::notify(vm);

	if(vm.count("delta"))
		m_delta = vm["delta"].as<delta_type>();

	if(vm.count("reduce-by"))
		m_k = vm["reduce-by"].as<k_type>();

	if(vm.count("paths"))
		m_input_paths = vm["paths"].as<paths>();

	if(vm.count("margin"))
		m_margin = vm["margin"].as<margin_type>();

	if(vm.count("to"))
		m_to = parse_shape_representation(vm["to"].as<std::string>());

	if(vm.count("from")){
		m_from = parse_shape_representation(
			vm["from"].as<std::string>()
		);
	}

	if(vm.count("ppcm")){
		m_resolution = resolution_type{
			vm["ppcm"].as<resolution_rep>()
		};
	}

	assert_convert_semantics(vm);
}

/*
 * methods
 * -------
 */

void shapetor_options::assert_convert_semantics(opt::variables_map const& vm)
{
	// check if we have exactly two path inputs
	if(m_input_paths.size() != 2){
		throw usage_error(
			"convert command must have exactly 2 paths: "
			"(1) a input file and (2) a output file"
		);
	}

	// check if from == none and we cant deduce the shape representation
	auto input = input_paths().at(0);
	auto output = input_paths().at(1);

	auto input_filepath = input.wstring();
	auto output_filepath = output.wstring();

	auto input_shape_deduced = deduce_shape_representation(input_filepath);
	auto output_shape_deduced = deduce_shape_representation(
		output_filepath
	);

	if(input_shape_deduced == shape_representation::NONE){
		if(from_representation() == shape_representation::NONE){
			throw usage_error(
				"--from shape representation is unknown and "
				" could not be deduced from the file extension"
			);
		}
	}else{
		if(from_representation() == shape_representation::NONE)
			m_from = input_shape_deduced;
	}

	if(output_shape_deduced == shape_representation::NONE){
		if(to_representation() == shape_representation::NONE){
			throw usage_error(
				"--to shape representation is unknown and "
				"the shape representation could not be "
				"deduced by the file extension"
			);
		}
	}else{
		if(to_representation() == shape_representation::NONE)
			m_to = output_shape_deduced;
	}

	//  the first path must exist and be a file
	//  the second path must not exist
	if(!fs::exists(input)){
		throw usage_error(
			"The input file does not exist"
		);
	}

	if(!force() && fs::exists(output)){
		throw usage_error(
			"The output file already exists"
		);
	}

	// if the conversion is to cdt, then must have delta, k
	if(to_representation() == shape_representation::CDT2D){
		if(vm.count("delta") == 0){
			throw usage_error(
				"For conversion to cdt2d you must give a delta"
				" parameter"
			);
		}
	}

	// if from representation is polygon and no resolution
	// was given
	if(from_representation() == shape_representation::POLYGON
		|| to_representation() == shape_representation::POLYGON){
		if(vm.count("ppcm") == 0){
			throw usage_error(
				"When managing boundary representation you "
				"must give a resolution parameter (--ppcm)"
			);
		}
	}
}


std::string shapetor_options::usage(std::string const& argv0)
{
	return	"USAGE:\n\t" + argv0 +
		" [general-options] convert [convert-options]"
		" <input-file> <output-file>\n"
		+ '\t' + argv0 +
		" [general-options] union [operation-options]"
		" <input-file>... <output-file>\n"
		+ '\t' + argv0 +
		" [general-options] minus [operation-options]"
		" <input-file>... <output-file>\n"
		+ '\t' + argv0 +
		" [general-options] intersection [operation-options]"
		" <input-file>... <output-file>\n";
}

std::string shapetor_options::usage() const
{
	return usage(m_args.front());
}

shapetor_options::cmd shapetor_options::command() const
{
	return m_command;
}

shape_representation shapetor_options::from_representation() const
{
	return m_from;
}

shape_representation shapetor_options::to_representation() const
{
	return m_to;
}

shapetor_options::parsing_state shapetor_options::state() const
{
	return m_state;
}

shapetor_options::options_description
shapetor_options::all_options_descriptions() const
{
	opt::options_description all_options("shapetor options");
	all_options
		.add(describe_general_options())
		.add(describe_convert_options());

	return all_options;
}

shapetor_options::delta_type shapetor_options::delta() const
{
	return m_delta;
}

shapetor_options::k_type shapetor_options::k() const
{
	return m_k;
}

shapetor_options::paths const& shapetor_options::input_paths() const
{
	return m_input_paths;
}

std::vector<std::string> const& shapetor_options::args() const
{
	return m_args;
}

shapetor_options::resolution_type shapetor_options::resolution() const
{
	return m_resolution;
}

shapetor_options::margin_type shapetor_options::margin() const
{
	return m_margin;
}

bool shapetor_options::verbose() const
{
	return m_verbose;
}

bool shapetor_options::force() const
{
	return m_force;
}

std::ostream& operator<<(std::ostream& out, shapetor_options::parsing_state s)
{
	switch(s){
	case shapetor_options::parsing_state::NONE:
		out << "none";
		break;
	case shapetor_options::parsing_state::HELP:
		out << "none";
		break;
	case shapetor_options::parsing_state::VERSION:
		out << "version";
		break;
	case shapetor_options::parsing_state::OPTIONS_READY:
		out << "ready";
		break;
	}

	return out;
}

std::ostream& operator<<(std::ostream& out, shapetor_options::cmd cmd)
{
	out << to_string(cmd);
	return out;
}

std::ostream& operator<<(std::ostream& out, shape_representation shape_rep)
{
	out << to_string(shape_rep);
	return out;
}

std::ostream& operator<<(
	std::ostream& out,
	shapetor_options const& options)
{
	out << "args: ";
	auto const& args = options.args();
	std::for_each(begin(args), prev(end(args)),
	[&out](auto const& arg)
	{
		out << arg << ' ';
	});

	out << args.back() << '\n';

	out << "state: "	<< options.state() << '\n';

	out << "command: "	<< options.command() << '\n';

	out << "from: "		<< options.from_representation() << '\n';
	out << "to: "		<< options.to_representation() << '\n';

	out << "delta: "	<< options.delta() << '\n';
	out << "k: "		<< options.k() << '\n';
	out << "resolution: "	<< options.resolution().count() << '\n';
	out << "margin: "	<< options.margin() << '\n';

	out << "paths: ";
	for(auto const& p : options.input_paths())
		std::cout << '\t' << p << '\n';

	return out;
}

