#! /usr/bin/python3
# Last edited on 2024-06-18 09:40:42 by stolfi

from math import sin, cos, tan, atan2, asin, hypot, sqrt, pi, inf, floor, sqrt
import rn, rmxn
import sys
import re
import random

import slicing_hel_povray as povray
 
prec = 4          # Decimal digits    
eps = 0.1**prec   # Coordinate quantum.

random.seed(4615)

# The program takes three parameters from the command line: 
# 
#  {pert} the coordinate perturbation amount, in units of {2*eps}.
#  {xrot} the rotation towards the {X}-axis (degrees).
#  {yrot} the rotation towards the {Y}-axis (degrees).
#
# The object is a cuboid with various protrusions and holes.

def ref_vertices(S):
  # Computes the vertices of one eight of the solid. Parameters:
  
  # {S}        Side of main cube.
  
  # The value of {S} is assumed approximate. The actual dimensions
  # will be rounded to apropriate even multiples of {eps}.

  # Returns a dict with keys 'vopp', 'vpop', 'vppo', 'vppp'.
  # The values are arrays of vertices.
  # Array 'vppp' is the vertices in the positive octant.
  # Array 'vopp' has the vertices on positive quadrant of {YZ} plane.  
  # Array 'vpop' has the vertices on positive quadrant of {XZ} plane.  
  # Array 'vppo' has the vertices on positive quadrant of {XY} plane.  
  #
  # Each element is a triplet of coordinates. 
  # Does not round any coordinate.

  sys.stderr.write("ref_vertices:")
  sys.stderr.write(" S = %.*f \n" % (prec, S))
  
  # Approximate key dimensions:
  HS = S/2        # Half-side of main cube.
  Ds = HS/4       # Depth of main slot in main cube.
  Es = HS/4       # Distance in {X} from slot to edge of main cube.
  Lh = HS/2       # Length in {X} of handle.
  Wh = HS/3       # Half-width in {Y} of handle.
  Th = HS/4       # Half-height in {Z} of handle in 
  Eh = Wh/2       # Thickness of handle all around. 
  
  Xk = [ None ] * 6              
  Xk[0] = 1.00 * HS      # {X} of side of main cube.
  Xk[1] = 0.25 * HS      # Low {X} of slot and holes on main cube.
  Xk[2] = 0.75 * HS      # High {X} of slot and holes on main cube.
  Xk[3] = 1.60 * HS      # High {X} of side handle.
  Xk[4] = 1.20 * HS      # Low {X} of hole in handle.
  Xk[5] = 1.40 * HS      # High {X} of hole in handle. 
 
  Yk = [ None ] * 3
  Yk[0] = 1.00 * HS      # {Y} of side  of main cube.
  Yk[1] = 0.30 * HS      # High {Y} of handle.
  Yk[2] = 0.10 * HS      # High {Y} of hole in handle.
  
  Zk = [ None ] * 4
  Zk[0] = 1.00 * HS      # High {Z} of main cube.
  Zk[1] = 0.80 * HS      # {Z} of bottom of slot on main cube.
  Zk[2] = 0.20 * HS      # {Z} of top of hole on main cube.
  Zk[3] = 0.15 * HS      # {Z} of top of handle.
  
  vopp = [ None ] *  2
  vpop = [ None ] *  0
  vppo = [ None ] *  0
  vppp = [ None ] * 11
  
  vopp[ 0] = (0,     Yk[0], Zk[0])
  vopp[ 1] = (0,     Yk[0], Zk[2])

  vppp[ 0] = (Xk[1], Yk[0], Zk[0])
  vppp[ 1] = (Xk[1], Yk[0], Zk[1])
  vppp[ 2] = (Xk[2], Yk[0], Zk[1])
  vppp[ 3] = (Xk[2], Yk[0], Zk[0])
  vppp[ 4] = (Xk[0], Yk[0], Zk[0])
  vppp[ 5] = (Xk[1], Yk[0], Zk[2])
  vppp[ 6] = (Xk[2], Yk[0], Zk[2])
  vppp[ 7] = (Xk[0], Yk[1], Zk[3])
  vppp[ 8] = (Xk[3], Yk[1], Zk[3])
  vppp[ 9] = (Xk[4], Yk[2], Zk[3])
  vppp[10] = (Xk[5], Yk[2], Zk[3])
  
  return { 'opp': vopp, 'pop': vpop, 'ppo': vppo, 'ppp': vppp }
 
def all_vertices(vref):
  # The dict {vref} must be a {dict} that gives the vertices 
  # on the positive octant (key 'ppp') and on the planes 
  # {YZ} ('opp'), {XZ} ('pop'), and {XY} ('ppo').
  #
  # Returns a dict {vtot} completed with vertices 
  # on other octants and quadrants of those planes,
  # with keys 'mpp', 'mpm', 'omp', etc.  Each vertex is augmented with a 
  # fourth coordinate that is the OBJ index of the vertex (from 1).
  #
  # Also returns the number of vertices created {Nv}.

  vtot = {}
  
  Nv = 0 # Number of vertices generated so far.
  
  for sx in -1, 0, +1:
    for sy in -1, 0, +1:
      for sz in -1, 0, +1:
        key_src = f"{'pop'[sx+1]}{'pop'[sy+1]}{'pop'[sz+1]}"
        v_src = vref.get(key_src, None)
        if v_src != None and len(v_src) != 0:
          Nv_src = len(v_src)
          key_dst = f"{'mop'[sx+1]}{'mop'[sy+1]}{'mop'[sz+1]}"
          v_dst = [ None ] * Nv_src
          for iv in range(Nv_src):
            Nv += 1
            vi_src = v_src[iv]
            vi_dst = ( vi_src[0]*sx, vi_src[1]*sy, vi_src[2]*sz, Nv )
            v_dst[iv] = vi_dst
          vtot[key_dst] = v_dst
  return vtot, Nv
  
def perturb_and_round_vertices(vtot, rmat,pert):
  # Multiplies eack coord vector of each vertex of {vtot} (as row) by the 3x3 matrix {rmat}.
  # Then adds a random amount in {[-pert _ pert]} to each coordinate.
  # Then rounds each coordinate to an even multiple of {eps}.
  
  for key,v in vtot.items():
    for iv in range(len(v)):
      vi = v[iv]
      ci = rmxn.map_row(vi[:3], rmat)
      v[iv] = tuple([ pertround(ci[j],pert) for j in range(3) ] + [ vi[3], ])
  return None

def pertround(c,pert):
  # Adds a random amount in {[-pert _ pert]*2*eps} to {c}, then 
  # rounds to an even multiple of {eps}.
  
  if pert != 0: 
    d = random.uniform(-pert*2*eps, pert*2*eps)
    c = 2*eps * floor(c/(2*eps) + 0.5)
  return c
  
def write_vertices_OBJ(wro, vtot):
  # Writes the vertices of the object to the file {wro} in OBJ format.
  # 
  # Expect {vtot} to be a dict with keys 'mmm', 'mmo', 'mmp', 'mom', ..., 'ppm',
  # 'ppo,' 'ppp' with the vertices in each 'octant' of {\RR^3},
  # Writes the vertices to {wro} in OBJ format.
  # 
  # Assumes that each vertex in {vtot} has three {float}
  # coordinates and a fourth integer element that is the OBJ index.
  # Assumes that each coordinate is rounded to an even multiple of
  # {eps}.
    
  Nv = 0

  def Prtv(p):
    # Writes {p[0..2]} to {wro} as the coordinates of a new vertex.
    # Expects that the fourth element of {p} is the OBJ index of the vertex
    # and vertices are printed in the order of that index.
    nonlocal Nv
    Nv += 1;
    assert Nv == p[3]
    # Writes {p} to {wro}: 
    wro.write("v");
    for i in range(3): wro.write(" %.*f" % (prec, p[i]))
    wro.write("\n");

  for key,v in vtot.items():
    for iv in range(len(v)):
      Prtv(v[iv])

  wro.write("\n")

  return None
  
def write_faces_OBJ(wro, vtot):
  # Writes the faces of the object to OBJ file {wro}, assuming that
  # the vertices are in the dict {vtot}.
  
  # Also returns tables {Fnrm} of normals to the faces and {Fbar} of their
  # barycenters.
  #
  # Also returns tables {E,G,M,T} of edges:
  #
  #   {E} Non-ghost edges of the mesh.
  #   {G} Ghost edges of the mesh.
  #   {T} Diagonal edges (other than {E,G,M}) that triangulate the face.
  #
  # Each element of these arrays is a pair {(iorg,idst)} of OBJ indices
  # of the endpoint vertices. Each array {Fnrm,Fbar,E,G,M,T} is indexed
  # {[1..N]}; element 0 is not used ({None}).
  
  Nf_exp = 16 + 14 + 18  # Expected face count.
  Ne_exp = 44 + 46 + 46  # Expected non-ghost edge count.
  Ng_exp =  2 +  6 +  4  # Expected ghost edge count.
    
  E = [ None ]
  G = [ None ]
  T = [ None ]
  Fbar = [ None ]
  Fnrm = [ None ]

  # Variables that are global to the nested functions:
  debug_face = False
  Fp = []    # Vertices of current face, indexed from 0.
  
  def Sved(p_org, p_dst, ghost):
    # Saves the edge from {p_org} to {p_dst} in {E} or {G}
    # depending on {ghost}.

    nonlocal E, G, T, Fbar, Fnrm, Fp, debug_face
    kv_org = p_org[3]
    kv_dst = p_dst[3]
    # Avoid duplicate edges (even ghost ones):
    if kv_org < kv_dst:
      if ghost:
        G.append((kv_org, kv_dst))
      else:
        E.append((kv_org, kv_dst))
    return None

  def Ptit(tit, debug = False):
    # Title for a face or set of faces.

    nonlocal E, G, T, Fbar, Fnrm, Fp, debug_face
    wro.write("\n");
    wro.write("# %s\n" % tit);
    sys.stderr.write("writing %s \n" % (tit));
    debug_face = debug
    return None
  
  def Bof():
    #  Start of a new face.

    nonlocal E, G, T, Fbar, Fnrm, Fp, debug_face
    assert len(Fp) == 0
    return None
  
  def Prtv(key, iv, ghost, nd):
    # Adds vertex {vtot[key][iv]} to the current face.
    # If {ghost} is {True}, assumes that the edge from previous
    # vertex is a ghost one.
    # The patameter {nd} is the number of triangutalion diagonals
    # that end at that vertex.

    nonlocal E, G, T, Fbar, Fnrm, Fp, debug_face
    p = vtot[key][iv]
    # Saves vertex in list {Fp} of face vertices, with {nd}:
    Fp.append(p + ( nd, ));
    if len(Fp) >= 2: Sved(Fp[-2], Fp[-1], ghost)
    return None
    
  def Eof(ghost, order):
    # End of face. If {ghost}, assumes that 
    # closing edge is a ghost one.
    # Prints in given or reverse order depending on {order} is {+1} or {-1}.

    nonlocal E, G, T, Fbar, Fnrm, Fp, debug_face
    assert len(Fp) >= 3
    Sved(Fp[-1], Fp[0], ghost)

    # Write the face:
    wro.write("f")
    deg = len(Fp)
    for rv in range(deg):
      jv = rv if dir == +1 else deg-1-rv
      p = Fp[jv]
      kv = p[3]
      if debug_face:
        sys.stderr.write(" v%04d = v%s[%2d] = ( %9.*f %9.*f %9.*f )\n" % (kv, key,iv, prec, p[0], prec, p[1], prec, p[2]));
      wro.write(" %d" % kv);
    wro.write("\n");

    # Generate the triangulation edges:
    get_triangulation_edges(Fp, T)
    
    # Compute normal, barycenter:
    Fn, Fb = face_normal_and_barycenter(Fp)
    Fnrm.append(Fn)
    Fbar.append(Fb)
    Fp = []; 
    return None
    
  # Here are the faces:

  for sz in -1, +1:
    tz = 'mop'[sz+1]
    
    Ptit(f"faces with sz = {sz:+d}")
    
    # Centrad {±Z} horz face of main cube:
    Bof()
    Prtv('op'+tz,  0, False, 2)
    Prtv('mp'+tz,  0, False, 0)
    Prtv('mm'+tz,  0, False, 1)
    Prtv('om'+tz,  0, False, 2)
    Prtv('pm'+tz,  0, False, 0)
    Prtv('pp'+tz,  0, False, 1)
    Eof(False, sz)
    
    for sx in -1, +1:
      tx = 'mop'[sx+1]
      
      Ptit(f"faces with sz = {sz:+d}, sx = {sx:+d}")

      # Distad {±Z} horz face of main cube:
      Bof()
      Prtv(tx+'p'+tz,  3, False, 0)
      Prtv(tx+'m'+tz,  3, False, 1)
      Prtv(tx+'m'+tz,  4, False, 0)
      Prtv(tx+'p'+tz,  4, False, 1)
      Eof(False, sx*sz)
      
      # Centrad vert wall of slot on {±Z} face of cube:
      Bof()
      Prtv(tx+'p'+tz,  0, False, 0)
      Prtv(tx+'m'+tz,  0, False, 1)
      Prtv(tx+'m'+tz,  1, False, 0)
      Prtv(tx+'p'+tz,  1, False, 1)
      Eof(False, sx*sz)
      
      # Horz floor face of slot on {±Z} face of cube:
      Bof()
      Prtv(tx+'p'+tz,  1, False, 0)
      Prtv(tx+'m'+tz,  1, False, 1)
      Prtv(tx+'m'+tz,  2, False, 0)
      Prtv(tx+'p'+tz,  2, False, 1)
      Eof(False, sx*sz)
      
      # Distad vert wall of slot on {±Z} face of cube:
      Bof()
      Prtv(tx+'p'+tz,  2, False, 1)
      Prtv(tx+'m'+tz,  2, False, 0)
      Prtv(tx+'m'+tz,  3, False, 1)
      Prtv(tx+'p'+tz,  3, False, 0)
      Eof(False, sx*sz)
      
      # Horz face of hole on main cube:
      Bof()
      Prtv(tx+'p'+tz,  5, False, 0)
      Prtv(tx+'p'+tz,  6, False, 1)
      Prtv(tx+'m'+tz,  6, False, 0)
      Prtv(tx+'m'+tz,  5, False, 1)
      Eof(False, sx*sz)
      
      # Horz {±Z} face on handle:
      Bof()
      Prtv(tx+'p'+tz,  7, False, 1)
      Prtv(tx+'m'+tz,  7, False, 2)
      Prtv(tx+'m'+tz,  8, False, 2)
      Prtv(tx+'p'+tz,  8, False, 2)
      Prtv(tx+'p'+tz,  7, False, 0)
      Prtv(tx+'p'+tz,  9, True,  1)
      Prtv(tx+'p'+tz, 10, False, 2)
      Prtv(tx+'m'+tz, 10, False, 2)
      Prtv(tx+'m'+tz,  9, False, 2)
      Prtv(tx+'p'+tz,  9, False, 0)
      Eof(True, sx*sz)
      
  for sx in -1, +1:
    tx = 'mop'[sx+1]

    Ptit(f"faces with sx = {sx:+d}")
    
    # Vert {±X} face of cube:
    Bof()
    Prtv(tx+'pp',  4, False, 0)
    Prtv(tx+'pp',  7, True,  1)
    Prtv(tx+'pm',  7, False, 2)
    Prtv(tx+'mm',  7, False, 2)
    Prtv(tx+'mp',  7, False, 2)
    Prtv(tx+'pp',  7, False, 0)
    Prtv(tx+'pp',  4, True,  1)
    Prtv(tx+'mp',  4, False, 2)
    Prtv(tx+'mm',  4, False, 2)
    Prtv(tx+'pm',  4, False, 2)
    Eof(False, sx)
    
    # Centrad {±X} vert wall of hole on cube:
    Bof()
    Prtv(tx+'pp',  5, False, 1)
    Prtv(tx+'mp',  5, False, 0)
    Prtv(tx+'mm',  5, False, 1)
    Prtv(tx+'pm',  5, False, 0)
    Eof(False, sx)
    
    # Distad {±X} vert wall of hole on cube:
    Bof()
    Prtv(tx+'pp',  6, False, 0)
    Prtv(tx+'pm',  6, False, 1)
    Prtv(tx+'mm',  6, False, 0)
    Prtv(tx+'mp',  6, False, 1)
    Eof(False, sx)
    
    # Centrad {±X} vert wall of hole on handle:
    Bof()
    Prtv(tx+'pp',  9, False, 0)
    Prtv(tx+'pm',  9, False, 1)
    Prtv(tx+'mm',  9, False, 0)
    Prtv(tx+'mp',  9, False, 1)
    Eof(False, sx)
    
    # Distad {±X} vert wall of hole on handle:
    Bof()
    Prtv(tx+'pp', 10, False, 0)
    Prtv(tx+'pm', 10, False, 1)
    Prtv(tx+'mm', 10, False, 0)
    Prtv(tx+'mp', 10, False, 1)
    Eof(False, sx)
    
    # Extreme {±X} vert wall of handle:
    Bof()
    Prtv(tx+'pp',  8, False, 1)
    Prtv(tx+'pm',  8, False, 0)
    Prtv(tx+'mm',  8, False, 1)
    Prtv(tx+'mp',  8, False, 0)
    Eof(False, sx)

  for sy in -1, +1:
    ty = 'mop'[sy+1]
    
    Ptit(f"faces with sy = {sy:+d}")
    
    # Vert {±Y} face of main cube:
    Bof()
    Prtv('o'+ty+'m',  0, False, 1)
    Prtv('o'+ty+'m',  1, True,  1)
    
    Prtv('p'+ty+'m',  5, True,  2)
    Prtv('p'+ty+'m',  6, False, 2)
    Prtv('p'+ty+'p',  6, False, 3)
    Prtv('p'+ty+'p',  5, False, 5)
    Prtv('p'+ty+'m',  5, False, 0)
    
    Prtv('o'+ty+'m',  1, True,  2)
    
    Prtv('m'+ty+'m',  5, True,  0)
    Prtv('m'+ty+'p',  5, False, 5)
    Prtv('m'+ty+'p',  6, False, 3)
    Prtv('m'+ty+'m',  6, False, 2)
    Prtv('m'+ty+'m',  5, False, 2)
    
    Prtv('o'+ty+'m',  1, True,  1)
    Prtv('o'+ty+'m',  0, True,  1)
    
    Prtv('m'+ty+'m',  0, False, 0)
    Prtv('m'+ty+'m',  1, False, 3)
    Prtv('m'+ty+'m',  2, False, 3)
    Prtv('m'+ty+'m',  3, False, 0)
    Prtv('m'+ty+'m',  4, False, 3)

    Prtv('m'+ty+'p',  4, False, 2)
    Prtv('m'+ty+'p',  3, False, 0)
    Prtv('m'+ty+'p',  2, False, 3)
    Prtv('m'+ty+'p',  1, False, 2)
    Prtv('m'+ty+'p',  0, False, 0)
    
    Prtv('o'+ty+'p',  0, False, 4)
    
    Prtv('p'+ty+'p',  0, False, 0)
    Prtv('p'+ty+'p',  1, False, 2)
    Prtv('p'+ty+'p',  2, False, 3)
    Prtv('p'+ty+'p',  3, False, 0)
    Prtv('p'+ty+'p',  4, False, 2)

    Prtv('p'+ty+'m',  4, False, 3)
    Prtv('p'+ty+'m',  3, False, 0)
    Prtv('p'+ty+'m',  2, False, 3)
    Prtv('p'+ty+'m',  1, False, 3)
    Prtv('p'+ty+'m',  0, False, 0)

    Eof(False, sy)
     
    for sx in -1, +1:
      tx = 'mop'[sx+1]
      
      Ptit(f"faces with sy = {sy:+d}, sx = {sx:+d}")
    
      # Vert {±Y} outer face of handle:
      Bof()
      Prtv(tx+ty+'p',  7, False, 1)
      Prtv(tx+ty+'p',  8, False, 0)
      Prtv(tx+ty+'m',  8, False, 1)
      Prtv(tx+ty+'m',  7, False, 0)
      Eof(False, sy*sx)
    
      # Vert {±Y} inner face of handle:
      Bof()
      Prtv(tx+ty+'p', 10, False, 1)
      Prtv(tx+ty+'p',  9, False, 0)
      Prtv(tx+ty+'m',  9, False, 1)
      Prtv(tx+ty+'m', 10, False, 0)
      Eof(False, sy*sx)
    
  sys.stderr.write("wrote %d faces (expected %d)\n" % (len(Fnrm)-1, Nf_exp))
  assert len(Fnrm) == Nf_exp + 1
  assert len(Fbar) == Nf_exp + 1

  sys.stderr.write("found %d non-ghost edges (expected %d)\n" % (len(E)-1, Ne_exp))
  assert len(E) == Ne_exp + 1

  sys.stderr.write("found %d ghost edges (expected %d)\n" % (len(G)-1, Ng_exp))
  assert len(G) == Ng_exp + 1

  return Fnrm, Fbar, E, G, T
  
def face_normal_and_barycenter(Fp):
  # Compute normal {Fn} and barycenter {Fb}
  # given a list of vertices {Fp[0..deg-1]}
  deg = len(Fp)
  assert deg >= 3
  po = Fp[0][:3]
  pa = Fp[1][:3]
  pb = Fp[2][:3]
  u = rn.sub(pa,po)
  v = rn.sub(pb,po)
  Fn, sz = rn.dir(rn.cross3d(u, v))

  Fb = (0, 0, 0)
  for jv in range(deg):
    pj = Fp[jv][:3]
    Fb = rn.mix(1.0, Fb, 1.0/deg, pj)
    
  # Check planarity:
  for jv in range(deg):
    pj = Fp[jv][:3]
    u = rn.sub(pj, Fb)
    s = rn.dot(u, Fn)
    assert abs(s) < 0.5*eps, f"face is not planar s = {s}"
  return Fn, Fb

def get_triangulation_edges(Fp, T):
  # Appends to {T} the diagonals of a face that triangulate it.
  #
  # Assumes that {Fp[0..deg-1]} are the vertices of the face, in
  # either order around the border.
  # 
  # Assumes that each vertext is a tuple {(X, Y, Z, kv, nd)}
  # where {kv} is the vertex's index in the OBJ file and {nd}
  # is the number of triangulation diagonals that end at that vertex.
  #
  # On return the {nd} items will be all zeroed.
  
  deg = len(Fp)  # Vertices in the original polygon.
  assert deg >= 3

  if deg > 3:
    stack = [ iv for iv in range(deg) if Fp[iv][4] == 0 ]     # The vertices with zero {nd} are {stack[0..]}

    # Creates a doubly-linked list of all vertices:
    inext = [ (iv + 1) % deg for iv in range(deg) ]
    iprev = [ (iv - 1) % deg for iv in range(deg) ]
    np = deg  # Cont of vertices in area still not triangulated.
    hv = 0    # Index into {Fp} of some vertex still in the list.

    while np > 3:
      # At this point the /remaining region/ that is still to be
      # triangulated has the vertices linked by {inext} and {iprev}
      # starting at {hv}. The {nd} field of every vertex in
      # {Fp[0..deg-1]} is the number of diagonals incident to that
      # vertex that remains to be collected. The vertices
      # in that set with zero {nd} are in {stack[0..]}.
      
      # Get a vertex with {nd=0}
      assert len(stack) > 0, "inconsitent {nd} fields"
      iv = stack.pop()
      assert Fp[iv][4] == 0
      
      # Trim that vertex off by a diagonal between the adjacent vertices:
      iv_org = iprev[iv]
      iv_dst = inext[iv]
      assert iv_org != iv_dst
      assert iv_org != iv and iv_dst != iv
      nd_org = Fp[iv_org][4]
      nd_dst = Fp[iv_dst][4]
      assert nd_org > 0 and nd_dst > 0
      
      kv_org = Fp[iv_org][3]
      kv_dst = Fp[iv_dst][3]
      assert kv_org >= 1 and kv_dst >= 1 and kv_org != kv_dst
      T.append(( kv_org, kv_dst ))
      
      # Decrement the {nd} fields of the two vertices:
      Fp[iv_org] = Fp[iv_org][:4] + ( nd_org - 1, )
      Fp[iv_dst] = Fp[iv_dst][:4] + ( nd_dst - 1, )
      
      #  If either became zero, stack them:
      if nd_org == 1: stack.append(iv_org)
      if nd_dst == 1: stack.append(iv_dst)
      
      # Exclude {iv} from the lists:
      inext[iv_org] = inext[iv]
      iprev[iv_dst] = iprev[iv]
      if hv == iv: hv = iv_dst
      
      np = np - 1
  
    assert len(stack) == 3
  
def write_triangles_OBJ(wro, vtot):
  # Writes the triangles of triangulations of the faces of the object to
  # OBJ file {wro}, assuming that the vertex indices are in the dict
  # {vtot}.
  
  # Does not return anything.
  
  Nt_exp = 52 + 64 + 84; # Expected triangle count.
  Nt_cmp = 0;            # Actual triangle count.
    
  # Variables that are global to the nested functions:
  debug_face = False

  def Ptit(tit, debug = False):
    # Title for a face or set of faces.

    wro.write("\n");
    wro.write("# %s\n" % tit);
    sys.stderr.write("writing %s \n" % (tit));
    debug_face = debug
    return None
  
  def Prtt(key0, iv0, key1, iv1, key2, iv2, order):
    # Outputs triangle with vertices {vtot[key][iv]}
    # where {(key,iv)} is {(key0,iv0)}, {(key1,iv1)}, {(key2,iv2)}.
    # The vertices are listed in that order or the reverse
    # depending on whether {order} is {+1} or {-1}.

    nonlocal debug_face, Nt_cmp
    keys = [ key0, key1, key2 ]
    ivs = [ iv0, iv1, iv2 ]
    wro.write("f")
    for rv in range(3):
      jv = rv if dir == +1 else 2-rv
      key = keys[jv]
      iv = ivs[jv]
      p = vtot[key][iv] 
      kv = p[3]
      if debug_face:
        sys.stderr.write(" v%04d = v%s[%2d] = ( %9.*f %9.*f %9.*f )\n" % (kv, key,iv, prec, p[0], prec, p[1], prec, p[2]));
      wro.write(" %d" % kv);
    wro.write("\n");
    Nt_cmp += 1
    return None
     
  # Here are the faces:

  for sz in -1, +1:
    tz = 'mop'[sz+1]
    
    Ptit(f"faces with sz = {sz:+d}")
    
    # Centrad {±Z} horz face of main cube:
    Prtt('op'+tz, 0,  'om'+tz, 0,  'pp'+tz, 0, sz)
    Prtt('om'+tz, 0,  'pm'+tz, 0,  'pp'+tz, 0, sz)
    Prtt('om'+tz, 0,  'op'+tz, 0,  'mm'+tz, 0, sz)
    Prtt('op'+tz, 0,  'mp'+tz, 0,  'mm'+tz, 0, sz)
    
    for sx in -1, +1:
      tx = 'mop'[sx+1]
      
      Ptit(f"faces with sz = {sz:+d}, sx = {sx:+d}")

      # Distad {±Z} horz face of main cube:
      Prtt(tx+'m'+tz, 3,  tx+'m'+tz, 4,  tx+'p'+tz, 4, sz*sx)
      Prtt(tx+'m'+tz, 3,  tx+'p'+tz, 4,  tx+'p'+tz, 3, sz*sx)
      
      # Centrad vert wall of slot on {±Z} face of cube:
      Prtt(tx+'m'+tz, 1,  tx+'p'+tz, 1,  tx+'m'+tz, 0, sz*sx)
      Prtt(tx+'p'+tz, 1,  tx+'p'+tz, 0,  tx+'m'+tz, 0, sz*sx)
      
      # Horz floor face of slot on {±Z} face of cube:
      Prtt(tx+'m'+tz, 1,  tx+'m'+tz, 2,  tx+'p'+tz, 2, sz*sx)
      Prtt(tx+'m'+tz, 1,  tx+'p'+tz, 2,  tx+'p'+tz, 1, sz*sx)
      
      # Distad vert wall of slot on {±Z} face of cube:
      Prtt(tx+'m'+tz, 2,  tx+'p'+tz, 2,  tx+'m'+tz, 3, sz*sx)
      Prtt(tx+'p'+tz, 2,  tx+'p'+tz, 3,  tx+'m'+tz, 3, sz*sx)
      
      # Horz face of hole on main cube:
      Prtt(tx+'p'+tz, 5,  tx+'p'+tz, 6,  tx+'m'+tz, 5, sz*sx)
      Prtt(tx+'m'+tz, 5,  tx+'m'+tz, 6,  tx+'p'+tz, 6, sz*sx)
      
      # Horz {±Z} face on handle:
      Prtt(tx+'p'+tz, 9,  tx+'p'+tz, 8,  tx+'p'+tz, 7, sz*sx)
      Prtt(tx+'p'+tz, 9,  tx+'p'+tz,10,  tx+'p'+tz, 8, sz*sx)
      Prtt(tx+'m'+tz, 8,  tx+'p'+tz, 8,  tx+'p'+tz,10, sz*sx)
      Prtt(tx+'m'+tz, 8,  tx+'p'+tz,10,  tx+'m'+tz,10, sz*sx)
      Prtt(tx+'m'+tz, 7,  tx+'m'+tz, 8,  tx+'m'+tz,10, sz*sx)
      Prtt(tx+'m'+tz, 7,  tx+'m'+tz,10,  tx+'m'+tz, 9, sz*sx)
      Prtt(tx+'m'+tz, 7,  tx+'m'+tz, 9,  tx+'p'+tz, 7, sz*sx)
      Prtt(tx+'m'+tz, 9,  tx+'p'+tz, 9,  tx+'p'+tz, 7, sz*sx)
      
  for sx in -1, +1:
    tx = 'mop'[sx+1]

    Ptit(f"faces with sx = {sx:+d}")
    
    # Vert {±X} face of cube:
    Prtt(tx+'pm', 4,  tx+'pp', 4, tx+'pp', 7, sx)
    Prtt(tx+'mm', 4,  tx+'mm', 7, tx+'mp', 4, sx)
    Prtt(tx+'mm', 4,  tx+'pm', 4, tx+'pm', 7, sx)
    Prtt(tx+'mp', 7,  tx+'pp', 4, tx+'mp', 4, sx)
    
    Prtt(tx+'pp', 7,  tx+'pp', 4, tx+'mp', 7, sx)
    Prtt(tx+'pm', 4,  tx+'pp', 7, tx+'pm', 7, sx)
    Prtt(tx+'mm', 4,  tx+'pm', 7, tx+'mm', 7, sx)
    Prtt(tx+'mm', 7,  tx+'mp', 7, tx+'mp', 4, sx)
    
    # Centrad {±X} vert wall of hole on cube:
    Prtt(tx+'pp', 5,  tx+'mp', 5, tx+'mm', 5, sx)
    Prtt(tx+'pp', 5,  tx+'mm', 5, tx+'pm', 5, sx)
    
    # Distad {±X} vert wall of hole on cube:
    Prtt(tx+'mp', 6,  tx+'pp', 6, tx+'pm', 6, sx)
    Prtt(tx+'mp', 6,  tx+'pm', 6, tx+'mm', 6, sx)
    
    # Centrad {±X} vert wall of hole on handle:
    Prtt(tx+'pp', 9,  tx+'mp', 9, tx+'pm', 9, sx)
    Prtt(tx+'mm', 9,  tx+'pm', 9, tx+'mp', 9, sx)
    
    # Distad {±X} vert wall of hole on handle:
    Prtt(tx+'pp',10,  tx+'pm',10, tx+'mp',10, sx)
    Prtt(tx+'mp',10,  tx+'pm',10, tx+'mm',10, sx)
    
    # Extreme {±X} vert wall of handle:
    Prtt(tx+'pp', 8,  tx+'mp', 8, tx+'mm', 8, sx)
    Prtt(tx+'mm', 8,  tx+'pm', 8, tx+'pp', 8, sx)

  for sy in -1, +1:
    ty = 'mop'[sy+1]
    
    Ptit(f"faces with sy = {sy:+d}")
    
    # Vert {±Y} face of main cube:
    Prtt('m'+ty+'m', 4,  'm'+ty+'m', 3,  'm'+ty+'m', 2, sy)
    Prtt('m'+ty+'m', 4,  'm'+ty+'m', 2,  'm'+ty+'m', 6, sy)
    Prtt('m'+ty+'m', 4,  'm'+ty+'m', 6,  'm'+ty+'p', 6, sy)
    Prtt('m'+ty+'m', 4,  'm'+ty+'p', 6,  'm'+ty+'p', 4, sy)
    Prtt('m'+ty+'p', 6,  'm'+ty+'p', 2,  'm'+ty+'p', 4, sy)
    Prtt('m'+ty+'p', 2,  'm'+ty+'p', 3,  'm'+ty+'p', 4, sy)
    
    Prtt('m'+ty+'m', 2,  'm'+ty+'m', 1,  'm'+ty+'m', 5, sy)
    Prtt('m'+ty+'m', 2,  'm'+ty+'m', 5,  'm'+ty+'m', 6, sy)
    Prtt('m'+ty+'p', 6,  'm'+ty+'p', 5,  'm'+ty+'p', 2, sy)
    Prtt('m'+ty+'p', 5,  'm'+ty+'p', 1,  'm'+ty+'p', 2, sy)
    
    Prtt('m'+ty+'m', 0,  'o'+ty+'m', 0,  'm'+ty+'m', 1, sy)
    Prtt('o'+ty+'m', 0,  'o'+ty+'m', 1,  'm'+ty+'m', 1, sy)
    Prtt('o'+ty+'m', 1,  'm'+ty+'m', 5,  'm'+ty+'m', 1, sy)
    Prtt('o'+ty+'m', 1,  'm'+ty+'p', 5,  'm'+ty+'m', 5, sy)
    Prtt('o'+ty+'p', 0,  'm'+ty+'p', 1,  'm'+ty+'p', 5, sy)
    Prtt('o'+ty+'p', 0,  'm'+ty+'p', 0,  'm'+ty+'p', 1, sy)
    
    Prtt('m'+ty+'p', 5,  'p'+ty+'p', 5,  'o'+ty+'p', 0, sy)
    Prtt('p'+ty+'p', 5,  'm'+ty+'p', 5,  'o'+ty+'m', 1, sy)
    
    Prtt('p'+ty+'m', 0,  'p'+ty+'m', 1,  'o'+ty+'m', 0, sy)
    Prtt('o'+ty+'m', 0,  'p'+ty+'m', 1,  'o'+ty+'m', 1, sy)
    Prtt('o'+ty+'m', 1,  'p'+ty+'m', 1,  'p'+ty+'m', 5, sy)
    Prtt('o'+ty+'m', 1,  'p'+ty+'m', 5,  'p'+ty+'p', 5, sy)
    Prtt('o'+ty+'p', 0,  'p'+ty+'p', 5,  'p'+ty+'p', 1, sy)
    Prtt('o'+ty+'p', 0,  'p'+ty+'p', 1,  'p'+ty+'p', 0, sy)
    
    Prtt('p'+ty+'m', 1,  'p'+ty+'m', 2,  'p'+ty+'m', 5, sy)
    Prtt('p'+ty+'m', 2,  'p'+ty+'m', 6,  'p'+ty+'m', 5, sy)
    Prtt('p'+ty+'p', 5,  'p'+ty+'p', 6,  'p'+ty+'p', 2, sy)
    Prtt('p'+ty+'p', 2,  'p'+ty+'p', 1,  'p'+ty+'p', 5, sy)
    
    Prtt('p'+ty+'m', 4,  'p'+ty+'m', 2,  'p'+ty+'m', 3, sy)
    Prtt('p'+ty+'m', 4,  'p'+ty+'m', 6,  'p'+ty+'m', 2, sy)
    Prtt('p'+ty+'m', 4,  'p'+ty+'p', 6,  'p'+ty+'m', 6, sy)
    Prtt('p'+ty+'m', 4,  'p'+ty+'p', 4,  'p'+ty+'p', 6, sy)
    Prtt('p'+ty+'p', 4,  'p'+ty+'p', 2,  'p'+ty+'p', 6, sy)
    Prtt('p'+ty+'p', 4,  'p'+ty+'p', 3,  'p'+ty+'p', 2, sy)
    
    for sx in -1, +1:
      tx = 'mop'[sx+1]
      
      Ptit(f"faces with sy = {sy:+d}, sx = {sx:+d}")
     
      # Vert {±Y} outer face of handle:
      Prtt(tx+ty+'p', 7,  tx+ty+'p', 8,  tx+ty+'m', 8, sx*sy)
      Prtt(tx+ty+'p', 7,  tx+ty+'m', 8,  tx+ty+'m', 7, sx*sy)
    
      # Vert {±Y} inner face of handle:
      Prtt(tx+ty+'p',10,  tx+ty+'p', 9,  tx+ty+'m', 9, sx*sy)
      Prtt(tx+ty+'p',10,  tx+ty+'m', 9,  tx+ty+'m',10, sx*sy)
    
  sys.stderr.write("wrote %d faces (expected %d)\n" % (Nt_cmp, Nt_exp))
  # assert Nt_cmp == Nt_exp

  return None
  
def face_normal_and_barycenter(Fp):
  # Compute normal {Fn} and barycenter {Fb}
  # given a list of vertices {Fp[0..deg-1]}
  deg = len(Fp)
  assert deg >= 3
  po = Fp[0][:3]
  pa = Fp[1][:3]
  pb = Fp[2][:3]
  u = rn.sub(pa,po)
  v = rn.sub(pb,po)
  Fn, sz = rn.dir(rn.cross3d(u, v))

  Fb = (0, 0, 0)
  for jv in range(deg):
    pj = Fp[jv][:3]
    Fb = rn.mix(1.0, Fb, 1.0/deg, pj)
    
  # Check planarity:
  for jv in range(deg):
    pj = Fp[jv][:3]
    u = rn.sub(pj, Fb)
    s = rn.dot(u, Fn)
    assert abs(s) < 0.5*eps, f"face is not planar s = {s}"
  return Fn, Fb


def rotation_matrix(xrot,yrot):
  # Returns a 3x3 orthonormal matrix that
  # rotates by {xrot} degrees towatds the {X}-axis
  # and {yrot} degrees toward the {Y}-axis.
  
  if xrot == 0 and yrot == 0:
    rmat = rmxn.ident_matrix(3,3)
  else:
    t, et = rn.dir((tan(xrot*pi/180), tan(yrot*pi/180), 1))
    assert et != 0
    s, es = rn.dir(rn.cross3d((0,0,1), t))
    assert es != 0
    r, er = rn.dir(rn.cross3d(t, s))
    assert abs(er - 1.0) <= 0.001
    rmat = (r, s, t)
    
  return rmat

def reven(C):
  # Rounds {C} to even multiple of {eps}.
  return eps*2*floor(C/eps/2 + 0.5)

def radians(deg):
  # Converts degrees to radians.
  return pi*deg/180

def checkpow(n, msg):
  # Checks that {n} is a power of 2:
  pp = n;
  while pp % 2 == 0: pp = pp//2
  assert pp == 1, msg

def main():
  pert = int(sys.argv[1])          # Random perturbation amount, in {eps} units.
  xrot = int(sys.argv[2])          # Rotation in the direction of the {X}-axis (degrees).
  yrot = int(sys.argv[3])          # Rotation in the direction of the {Y}-axis (degrees).
  triang = sys.argv[4] == "T"      # Triangulate the mesh?
  
  sys.stderr.write("slicing_hel_example.py: ")
  sys.stderr.write(f" pert = {pert}")
  sys.stderr.write(f" xrot = {xrot:.4f}")
  sys.stderr.write(f" yrot = {yrot:.4f}")
  sys.stderr.write(f" triang = {triang}")
  sys.stderr.write("\n")
  
  assert 0 <= pert and pert < 50, "invalid pert"
  assert abs(xrot) < 90, "invalid xrot"
  assert abs(yrot) < 90, "invalid yrot"

  S = 100       # Side of cube
  
  rmat = rotation_matrix(xrot,yrot)

  vref = ref_vertices(S)
  vtot, Nv = all_vertices(vref)
  perturb_and_round_vertices(vtot, rmat,pert)

  file_prefix = f"out/hel_pa{pert:03d}_xr{xrot:03d}_yr{yrot:03d}_tri{str(triang)[0]}"

  wro = open(file_prefix + ".obj", 'w')
  wrp = open(file_prefix + ".inc", 'w')
  write_vertices_OBJ(wro, vtot)
  if triang:
    write_triangles_OBJ(wro, vtot)
    wrp.write("kaboom()")
  else:
    Fnrm, Fbar, E, G, T = write_faces_OBJ(wro, vtot)

    Ne = len(E) - 1
    Ng = len(G) - 1
    Nt = len(T) - 1
    Nf = len(Fnrm) - 1

    write_povray = False

    if write_povray:
      povray.write_parms(wrp, S, Nv,Ne,Ng,Nt,Nf)
      povray.write_vertices(wrp, vobj, Vlab)
      povray.write_edges(wrp, E, G, T, vobj, Vlab)
      povray.write_faces(wrp, Fnrm, Fbar)

  wrp.close()
  wro.close()
  
  return 0

main()

 
