#! /usr/bin/python3
# Implementation of module {paper_figures_B}
# Last edited on 2021-10-24 15:19:08 by stolfi

import paper_figures_B
import paper_example_B
import move
import move_example
import move_parms
import contact
import path
import path_example
import rootray_shape
import rootray
import raster
import block
import hacks

import rn

import pyx
import random
import sys
from math import sqrt, hypot, sin, cos, atan2, floor, ceil, inf, nan, pi

def plot_figure(fname, fig, subfig):
  
  title = "### figure %s sub-figure %s ###" % (fig,subfig)
  sys.stderr.write("#"*80 + "\n")
  sys.stderr.write(title + "#"*(80-len(title)) + "\n")
  sys.stderr.write("#"*80 + "\n")
  sys.stderr.write("\n")
  
  style = make_style_dict()
  color = make_color_dict()

  if fig == "tableau":
    c = plot_figure_tableau(subfig, style,color)
  elif fig == "paths":
    c = plot_figure_paths(subfig, style,color)
  elif fig == "moves":
    c = plot_figure_moves(subfig, style,color)
  elif fig == "cover":
    c = plot_figure_cover(subfig, style,color)
  elif fig == "input":
    c = plot_figure_input(subfig, style,color)
  elif fig == "zigzag":
     c = plot_figure_scanline(subfig, style,color)
  elif fig == "cold":
     c = plot_figure_cold(subfig, style,color)
  elif fig == "rivers":
     c = plot_figure_rivers(subfig, style,color)
  elif fig == "canon":
     c = plot_figure_canon(subfig, style,color)
  elif fig == "blocks":
     c = plot_figure_blocks(subfig, style,color)
  else:
     assert False, ("figure '%s ' not implemented.\n" % fig)

  if c != None:
    hacks.write_plot(c, fname)
  else:
    sys.stderr.write("!! figure '%s subfigure %s' not plotted.\n" % (fig,subfig))

  sys.stderr.write("#"*80 + "\n")
  sys.stderr.write("\n")

  return
  # ----------------------------------------------------------------------


def plot_figure_tableau(subfig, style,color):
  # Plots the figure that shows a path covering one contact and closing another.
  
  assert subfig == "basic", ("invalid subfig %s" % subfig)

  m = 35   # Number of scanlines.
  iX = 17  # The {i} argument to {MinFullPath}
  jX = 23  # The {j} argument to {MinFullPath}.
  L = [ 0, 7, 27, 31, m, m+1 ] # Indices of essential cut-lines, plus a sentinel {m+1}.
  
  elsz = 1.0         # X and Y size of a tableau cell.
  tbsz = (m+1)*elsz  # Total X and Y size of tableau, minus row and column labels. 

  B = ((0, 0), (tbsz, tbsz))  # Bounding box of tabeau.
  
  # Compute the figure's bounding box {Bfig}, allowing for row and column labels:
  Bfig = rn.box_expand(B, (2.6,2.2), (2.6,2.2))

  autoscale = False
  c = make_figure_canvas(Bfig, autoscale, style,color)

  dp = (0, 0)
  random.seed(4615 + 418)
  
  elmrg = 0.11 # Indent of element boxes.
  wdess = 0.15 # Width of essential cut-line strokes.
  wddot = 0.50 # Diameter of dots.
  labwd = 5.00 # Estra width for index labels.
  esswd = 0.60 # Overshoot of essential row/cols backgroud stripe.
  loXY = 0 - esswd          # Low coordinate of essential row/cols backgroud stripe.
  hiXY = (m+1)*elsz + esswd # High coordinate of essential row/cols backgroud stripe.
  
  wd = 0.05 # Width of lines.
  clr_unused = pyx.color.rgb( 0.830, 0.820, 0.800 ) # Entries unused because {i >= j}.
  clr_noband = pyx.color.rgb( 0.570, 0.580, 0.600 ) # Entries unused/{None} due to essential cut-line.
  clr_nothot = pyx.color.rgb( 1.000, 0.000, 0.000 ) # Entries that are {None} because of Tcool violation.
  clr_exists = pyx.color.rgb( 0.000, 0.700, 0.000 ) # Entries defined.
  clr_tocomp = pyx.color.rgb( 0.000, 0.000, 0.000 ) # Entries not computed yet.
  clr_phnext = pyx.color.rgb( 0.000, 1.000, 1.000 ) # Dot on entry that is to be computed next.
  clr_phneed = pyx.color.rgb( 1.000, 1.000, 1.000 ) # Dot on entry that is needed for that entry.
  clr_cutess = pyx.color.rgb( 1.000, 0.000, 0.500 ) # Highlight of rows/columns that are essential cut-lines.
  clr_cutcmp = pyx.color.rgb( 0.900, 0.900, 0.900 ) # Highlight of rows/columns that {iX} or {jX}.

  rnext = 0;
  while iX >= L[rnext]: rnext += 1
  IessX = L[rnext-1]  # Index of scanline before {iX}
  
  def plot_tableau_row_col_deco(i, Iess):
    # If the row and/or column {i} is relevant, plots its index and
    # possibly draws line across it. The parameter {Iess} is the index
    # of the highest essential cut-line equal to or less than {i}.

    if i == Iess or i == iX or i == jX:
      # Row and/or column {i} may be relevant:
      
      pri = (loXY, (m - i + 0.5)*elsz)  # Left endpoint of line though row {i}.
      qri = (hiXY, pri[1])              # Right endpoint of line though row {i}.
      pci = ((i + 0.5)*elsz, loXY)      # Bottom endpount of line through column {i}.
      qci = (pci[0], hiXY)              # Top endpoint of line through column {i}.

      # Draw lines all along row {i} and/or column {i}:
      if i == Iess:
        # Row/column {i} is essential cut-line
        hacks.plot_line(c, pri, qri, dp, clr_cutess, wdess, None)
        hacks.plot_line(c, pci, qci, dp, clr_cutess, wdess, None)
      elif i == iX or i == jX:
        # Column {i} is interesting:
        hacks.plot_line(c, pci, qci, dp, clr_cutcmp, wdess, None)
        if i == iX:
          # Row {i} is intersting too:
          hacks.plot_line(c, pri, qri, dp, clr_cutcmp, wdess, None)
        
      # Draw label on column {i}:
      for txY in loXY - 1.3, hiXY + 0.5:
        dxci = 0.3 if i <= 9 else 0.8
        tpci = (pci[0] - dxci, txY)
        hacks.plot_text(c, (r"\texttt{%d}" % i), tpci, dp, style['tbixfsize'], None)

      if i == Iess or i == iX:
        # Draw label on row {i}:
        dxri = 0.3 if i <= 9 else 0.8     # Extra displacement of label on levt
        for txX in loXY - 1.0 - dxri, hiXY + 0.3:
          tpri = (txX, pri[1] - 0.4)
          hacks.plot_text(c, (r"\texttt{%d}" % i), tpri, dp, style['tbixfsize'], None)

    return
    # ....................................................................

  def plot_tableau_cell(i, Iess, j, Jess):
    # Plots the cell {i,j} of the tableau. The parameter {Iess} is the
    # index of the highest essential cut-line equal to or less than {i}.
    # The parameter {Jess} is the index of the lowest essential cut-line
    # greater than {i}.

    # Plot the box:
    pij = (j*elsz, (m-i)*elsz)
    qij = rn.add(pij, (elsz, elsz))
    Bij = (pij, qij)
    Bij = rn.box_expand(Bij, (-elmrg, -elmrg), (-elmrg, -elmrg))
    # The probability of {Q[i,j] = None} should grow as {i,j} moves away from the diagonal:
    probij = (min(0.95,(j - i)/21))**0.75
    if i == m or j == 0 or i >= j:
      clr = clr_unused
    elif j > Jess:
      clr = clr_noband
    elif j > jX or (j == jX and i >= iX):
      clr = clr_tocomp
    elif random.random() > probij:
      clr = clr_exists
    else:
      clr = clr_nothot
    hacks.plot_box(c, Bij, dp, clr)

    # Plot the dot, if any:
    cij = rn.add(pij, (0.5*elsz, 0.5*elsz)) # Center of cell.
    if i == iX and j == jX:
      # Paint a yellow dot to indicate the entry to be computed bext:
      clr = clr_phnext
    elif i < j and i >= IessX and j == iX:
      clr = clr_phneed
    else:
      clr = None

    if clr != None:
      hacks.plot_line(c, cij, cij, dp, clr, wddot, None)
    return
    # ....................................................................

  for ipass in range(2):
    # When {ipass} is 0, draws lines for essential rows/colums,
    # When {ipass} is 1, paints the tableau entries.
  
    rnext = 0 # The next essential cutline greater than or equal to {i} is {L[r]}
    for i in range(m+1):

      # Compute/choose the break values {Jmax,Jess} of {j}:
      while i >= L[rnext]: rnext += 1
      assert i < L[rnext]
      Iess = L[rnext-1] # Index of last cut-line less than or equal to {i}.
      Jess = L[rnext]   # Index of first cut-line greater than {i}.
      
      if ipass == 0:
        # Paint the background of rows that are essential cut-lines:   
        assert rnext > 0
        plot_tableau_row_col_deco(i, Iess)
      else:
        # Paint row {j} of the tableau:
        for j in range(m+1):
          plot_tableau_cell(i, Iess, j, Jess)
  
  return c
  # ----------------------------------------------------------------------

def plot_figure_cover(subfig, style,color):
  # Plots the figure that shows a path covering one contact and closing another.
  
  assert subfig == "basic", ("invalid subfig %s" % subfig)

  # Get the two paths {P,Q}, and the contacts {ctA,ctB}, where {ctA} is covered by {P} and {Q},
  # {ctB} is closed by {P}:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OPHS, CTS = paper_example_B.make_simple_cover(mp_fill, mp_jump)
  
  nph = len(OPHS); assert nph == 2
  nct = len(CTS); assert nct == 2
  
  P = OPHS[0] # Main path.
  Q = OPHS[1] # Secondary path.

  B = path.bbox(OPHS) # Bounding box of all move endpoints.
  B = rn.box_join(B, contact.bbox(CTS)) # Just paranoia.
  
  # Compute the figure's bounding box {Bfig}:
  Bfig = rn.box_expand(B, (2.2,2.2), (1.2,1.2))

  # # Widen {Bfig} symmetrically to standard "math figure" widtdh:
  # Bfig = widen_box_for_math_figure(Bfig, style)
 
  autoscale = False
  c = make_figure_canvas(Bfig, autoscale, style,color)

  dp = (0, 0)

  # Plot the paths:
  rwdf = style['rwd_fill']
  clrP = color['fill'] 
  clrQ = ghostify_colors([color['hifi'],], color['ghost'])[0] 
  deco = True
  plot_paths(c, OPHS, dp, [clrP,clrQ], rwdf,deco, style,color)
  
  # Label the paths and the endpoints:
  mfsize = style['mathfsize'] # Font size for math labels.
  pfsize = style['detafsize'] # Font size for points.

  ptPe0 = path.pini(P)
  plot_dot_on_path(c, ptPe0, dp, style,color)
  Pe0lab_tx = "A"
  Pe0lab_dp = (-1.5, -0.5)
  plot_point_label(c, Pe0lab_tx, ptPe0, Pe0lab_dp, dp, pfsize, style,color)

  ptPe1 = path.pfin(P)
  plot_dot_on_path(c, ptPe1, dp, style,color)
  Pe1lab_tx = "E"
  Pe1lab_dp = (-0.5, +0.7)
  plot_point_label(c, Pe1lab_tx, ptPe1, Pe1lab_dp, dp, pfsize, style,color)
  
  ptphP = rn.mix(0.5,ptPe0, 0.5,ptPe1)
  phPlab_tx = r"$P$"
  phPlab_dp = (+2.0,-3.8)
  plot_math_label(c, phPlab_tx, ptphP, phPlab_dp, dp, mfsize, style)
 
  ptphQ = path.pini(Q)
  phQlab_tx = r"$Q$"
  phQlab_dp = (-1.2,-2.0)
  plot_math_label(c, phQlab_tx, ptphQ, phQlab_dp, dp, mfsize, style)

  # Plot the contacts:
  interbc = False
  shadow = True
  trim = False
  plot_contacts(c, CTS, dp, interbc, shadow,trim, style,color)
  
  # Label points on the path {P} where it covers the midpoint of each contact:
  for ict in range(nct):
    ct = CTS[ict]
    ctends = contact.endpoints(ct)
    pmd = rn.mix(0.5,ctends[0], 0.5,ctends[1]) # Midpoint of contact.
    for jsd in range(2):
      mv = contact.side_move(ct, jsd)
      # Find the move {mv} in either {P} or {Q}:
      for kph in range(2):
        oph = OPHS[kph]
        imv = path.find_move(oph, mv)
        if imv != None: break
      assert imv != None
      omv = path.elem(oph, imv) # Oriented version of {mv}.

      # Find point {pt} on {mv} closest to midpoint of {ct}:
      r = rn.pos_on_line(move.pini(omv), move.pfin(omv), pmd)
      assert 0 < r and r < 1, "move does not cover midpoint of contact"
      pt = rn.mix(1-r,move.pini(omv), r,move.pfin(omv))

      # Label points on the path, moves, and contacts:
      du = (+0.5,+0.2) # Displacement along contacts.
      dv = (-0.2,+0.5) # Displacement perpendicular to contacts.
      if ict == 0 and jsd == 0:
        ptlab_tx = "B";        ptlab_dp = rn.mix(1.0,(+0.2, -1.5), +0.3,dv)
        mvlab_tx = r"$r_1'$";  mvlab_dp = rn.mix(1.0,ptlab_dp, -3.5,du)
        ctlab_tx = r"$s_1$";   ctlab_dp = rn.mix3(1.0,(0.0,0.0), -5.6,du, +2.0,dv)
      elif ict == 0 and jsd == 1:
        ptlab_tx = None;       ptlab_dp = rn.mix(1.0,(-0.7, +0.7), +0.2,dv)
        mvlab_tx = r"$r_1''$"; mvlab_dp = rn.mix(1.0,ptlab_dp, +0.3,du)
        ctlab_tx = None
      elif ict == 1 and jsd == 0:
        ptlab_tx = "C";        ptlab_dp = rn.mix(1.0,(+0.2, -1.5), +0.3,dv)
        mvlab_tx = r"$r_2'$";  mvlab_dp = rn.mix(1.0,ptlab_dp, -4.5,du)
        ctlab_tx = r"$s_2$";   ctlab_dp = rn.mix3(1.0,(0.0,0.0), -5.3,du, +2.0,dv)
      elif ict == 1 and jsd == 1:
        ptlab_tx = "D";        ptlab_dp = rn.mix(1.0,(-0.7, +0.7), +0.2,dv)
        mvlab_tx = r"$r_2''$"; mvlab_dp = rn.mix3(1.0,ptlab_dp, +4.5,du, +0.2,dv)
        ctlab_tx = None

      if ptlab_tx != None:
        plot_point_label(c, ptlab_tx, pt, ptlab_dp, dp, pfsize, style,color)
        plot_dot_on_path(c, pt, dp, style,color)
      if mvlab_tx != None:
        plot_math_label(c, mvlab_tx, pt, mvlab_dp, dp, mfsize, style)
      if ctlab_tx != None:
        ctlab_txx = r"\textcolor{red}{%s}" % ctlab_tx
        plot_math_label(c, ctlab_txx, pt, ctlab_dp, dp, mfsize, style)
  
  return c
  # ----------------------------------------------------------------------

def plot_figure_paths(subfig, style,color):
  # Plots the figure that shows a smple path {OPHS[0]} with labels "pini", etc.
  # and its reverse.
  
  assert subfig == "basic", ("invalid subfig %s" % subfig)
  
  # Get the paths:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OPHS = paper_example_B.make_simple_path(mp_fill, mp_jump)
  
  assert len(OPHS) == 1
  oph = OPHS[0]
  nmv = path.nelems(oph) 

  B = path.bbox([oph,]) # Bounding box of all move endpoints.

  Xstep = B[1][0] - B[0][0] + 8.0  # Displacement betwwen the two versions of the path.
  
  # Compute the figure's bounding box {Bfig}:
  Bfig = (B[0], rn.add(B[1], (Xstep,0))) # Bounding box for both versions of the path.

  # Add space at bottom and top for labels:
  mrg0 = (3.5, 1.8) # Extra margin for labels at left and bottom
  mrg1 = (3.5, 2.4) # Extra margin for labels at right and top.
  Bfig = rn.box_expand(Bfig, mrg0, mrg1)

  # # Widen {Bfig} symmetrically to standard "math figure" widtdh:
  # Bfig = widen_box_for_math_figure(Bfig, style)
 
  autoscale = False
  c = make_figure_canvas(Bfig, autoscale, style,color)

  for rev in False, True:
    dp = (int(rev)*Xstep, 0)
    
    ophr = path.rev(oph) if rev else oph

    # Plot the matter shadow:
    plot_trace_matter(c, [ophr,], dp, style,color)

    # Plot the path:
    rwdf = style['rwd_fill']
    clr = color['fill'] 
    deco = True
    plot_paths(c, [ophr,], dp, [clr,], rwdf,deco, style,color)
    
    # Plot the endpoints of the paths in black and larger:
    plot_path_endpoints(c, [ophr,], dp, style,color)
    
    # Labeled points ({xxxP}) and label displacements ({xxxD}) on the unreversed path:
    piniP = path.pini(oph); piniD = (-1.2, +1.2)
    pfinP = path.pfin(oph); pfinD = (-1.0, +1.2)
    pmovP = [ rn.mix(0.5,move.pini(mvk),0.5,move.pfin(mvk)) for k in range(nmv) for mvk in (path.elem(oph,k),) ]
    pmovD = [ 
      (-3.0, -0.9), # P[0]
      (-1.0, -1.8), # P[1]
      (-1.6, -1.8), # P[2]
      (+0.8, -0.9), # P[3]
    ]
    pmidP = rn.mix(0.5,piniP, 0.5,pfinP); pmidD = (-0.3, +1.0)

    txphr = r"P" if not rev else r"{\mkern-0.7\thinmuskip\overleftarrow{P}\mkern-0.7\thinmuskip}"
    
    txpini = r"$\mathop{p_{\su{INI}}}(%s)$" % txphr
    txpfin = r"$\mathop{p_{\su{FIN}}}(%s)$" % txphr
    txpmid = r"$%s$" % txphr
    txmovS = [ (r"$%s[%d]$" % (txphr,k)) for k in range(nmv) ]

    mfsize = style['mathfsize']
    plot_math_label(c, txpmid, pmidP, pmidD, dp, mfsize, style)
    if not rev:
      plot_math_label(c, txpini, piniP, piniD, dp, mfsize, style)
      plot_math_label(c, txpfin, pfinP, pfinD, dp, mfsize, style)
      for k in range(nmv):
        plot_math_label(c, txmovS[k], pmovP[k], pmovD[k], dp, mfsize, style)
    else:
      plot_math_label(c, txpfin, piniP, piniD, dp, mfsize, style)
      plot_math_label(c, txpini, pfinP, pfinD, dp, mfsize, style)
      for k in range(nmv):
        plot_math_label(c, txmovS[nmv-1-k], pmovP[k], pmovD[k], dp, mfsize, style)
  
  return c
  # ----------------------------------------------------------------------

def plot_figure_moves(subfig, style,color):
  # If {subfig} == "dir" shows a simple move {OPHS[0]} and a simple jump {OPHS[1]}, with labels "pini", etc.
  # If {subfig} == "rev" plots the reversed moves instead.
  
  assert subfig == "dir" or subfig == "rev", ("invalid subfig %s" % subfig)
  
  # Get the two single-move paths:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OPHS = paper_example_B.make_simple_moves(mp_fill, mp_jump)

  assert len(OPHS) == 2

  B = path.bbox(OPHS) # Bounding box of endpoints of both moves (undisplaced).
  Xstep = B[1][0] - B[0][0] + 11.0  # Displacement betwwen the two versions of the path.
  
  # Compute the figure's bounding box {Bfig}:
  Bfig = (B[0], rn.add(B[1], (Xstep,0))) # Bounding box for both versions of the path.
  
  # Add space for labels at top and bottom:
  mrg0 = (5.0,1.0) # Extra margin for labels at left and bottom
  mrg1 = (5.0,1.5) # Extra margin for labels at right and top.
  Bfig = rn.box_expand(Bfig, mrg0, mrg1)
  
  # # Widen to standars "math figure" width:
  # Bfig = widen_box_for_math_figure(Bfig, style)
  
  autoscale = False
  c = make_figure_canvas(Bfig, autoscale, style,color)
    
  rev = (subfig == "rev")

  for iph in range(2):
    dp = (iph*Xstep, 0)
    
    oph = OPHS[iph]
    assert path.nelems(oph) == 1
    omv = path.elem(oph,0)
    
    ophr = path.rev(oph) if rev else oph
    omvr = move.rev(omv) if rev else omv

    # Plot the matter shadow:
    plot_trace_matter(c, [oph,], dp, style,color)
    
    # Plot the move:
    clr = color['fill'] 
    rwdf = style['rwd_fill']
    deco = True
    plot_paths(c, [ophr,], dp, [clr,], rwdf,deco, style,color)
    
    # Labeled points ({xxxP}) and label displacements ({xxxD}) on the unreversed moves:
    piniP = move.pini(omv); piniD = (-3.7 + 0.4*iph - 0.6*int(rev), -0.3)
    pfinP = move.pfin(omv); pfinD = (+1.0 - 0.4*iph, +0.1)
    pmidP = rn.mix(0.5,piniP, 0.5,pfinP); pmidD = (-0.5, +1.0 - 0.3*iph)

    txmvr = r"{\mkern-0.7\thinmuskip\overleftarrow{r}\mkern-0.7\thinmuskip}" if rev else r"r"
    
    txpini = r"$\mathop{p_{\su{INI}}}(%s)$" % txmvr
    txpfin = r"$\mathop{p_{\su{FIN}}}(%s)$" % txmvr
    txpmid = r"$%s$" % txmvr

    mfsize = style['mathfsize']
    plot_math_label(c, txpmid, pmidP, pmidD, dp, mfsize, style)
    if not rev:
      plot_math_label(c, txpini, piniP, piniD, dp, mfsize, style)
      plot_math_label(c, txpfin, pfinP, pfinD, dp, mfsize, style)
    else:                                          
      plot_math_label(c, txpfin, piniP, piniD, dp, mfsize, style)
      plot_math_label(c, txpini, pfinP, pfinD, dp, mfsize, style)
  
  return c
  # ----------------------------------------------------------------------

def plot_figure_input(subfig, style,color):
  # Plots the figures with individual rasters, as slected by {subfig}: 
  #
  #   subfig = "rasct" rasters and contacts.
  #   subfig = "links" links.
  #   subfig = "graph" contact graph.
  #
  # Returns the canvas {c} with the figure.
  
  assert subfig == "rasct" or subfig == "links" or subfig == "graph", ("invalid subfig %s" % subfig)

  # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OCRS,OPHS,OLKS,CTS,VGS,EGS = paper_example_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 
  
  assert len(OPHS) > 0
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  autoscale = True
  c = make_figure_canvas(Bfig, autoscale, style,color)

  dp = (0,0)
  
  if subfig == "rasct":

    # RASTERS AND CONTACTS
    
    show_path_statistics(OPHS, CTS)

    plot_trace_matter(c, OCRS + OPHS + OLKS, dp, style,color)

    plot_contours(c, OCRS, dp, style, color['cont'])

    # Plot the filling path(s) with strong color:
    clr = color['fill']
    rwdf = style['rwd_fill']
    deco = False
    plot_paths(c, OPHS, dp, [clr,], rwdf,deco, style,color)

    # Plot contacts.
    interbc = False
    shadow = False
    trim = False
    plot_contacts(c, CTS, dp, interbc, shadow,trim, style,color)

  elif subfig == "links" and len(OLKS) > 0:

    # LINKS

    plot_contours(c, OCRS, dp, style, color['ghost'])

    # Plot the filling path(s) with weak color:
    clr = color['ghost']
    rwdf = style['rwd_fill']
    deco = False
    plot_paths(c, OPHS, dp, [clr,], rwdf,deco, style,color)

    # Plot individual links with various colors: 

    xdir = (1,0)
    ydir = (0,1)
    ystep,yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
    bicolor = True
    plot_links(c, OLKS, dp, ystep,yphase, bicolor, style,color) 

    # Plot dots at start and end of links: 
    plot_path_endpoints(c, OLKS, dp, style,color)

  elif subfig == "graph":

    # CONTACT GRAPH

    plot_contours(c, OCRS, dp, style, color['ghost'])

    # Plot the filling path(s) with weak color:
    clr = color['ghost']
    rwdf = style['rwd_fill']
    deco = False
    plot_paths(c, OPHS, dp, [clr,], rwdf,deco, style,color)

    plot_contact_graph(c, VGS, EGS, dp, style,color)
    
  else:
    assert False, ("invalid subfig %s" % subfig)

  return c
  # ----------------------------------------------------------------------

def plot_figure_scanline(subfig, style,color):
  # Plots the figures with scanline tool-path using the rasters {OPHS}: 
  #
  #   subfig = "nonalt" non-alternating
  #   subfig = "alt" alternating
  #
  # Returns the canvas {c} with the figure.

  assert subfig == "nonalt" or subfig == "alt", ("invalid subfig %s" % subfig)

  # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OCRS,OPHS,OLKS,CTS,VGS,EGS = paper_example_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 
  
  alt = (subfig == "alt") # True for alternating path.
  
  dp = (0,0)
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  autoscale = True
  c = make_figure_canvas(Bfig, autoscale, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Reverse alpernate scanlines and get list {CTSsel} of selected contacts to plot:
  if alt:
    # Reverse order and orientation of raster elements of {OPHS} on alternate scanlines:
    PFSnew = [] # Reaster paths, reversed if needed.
    rev = False # True if next scan-line is to be reversed.
    for Si in SCS:
      # Now {Si} is a list of indices of rasters in one scanline.
      if rev:
        PFSi = [ path.rev(OPHS[jrs]) for jrs in Si ]
        PFSi.reverse()
      else:
        PFSi = [ OPHS[jrs] for jrs in Si ]
      rev = not rev
      PFSnew += PFSi
    OPHS = PFSnew

  # Assemble the path:
  use_links = alt
  ph = path.concat(OPHS, use_links, mp_jump)
  
  # Find the coldest contact and its cooling time:
  ncold = 1 # Number of coldest contacts to show.
  CTScold = find_coldest_contacts(ph, CTS, ncold)

  show_path_statistics([ph,], CTScold)

  plot_contours(c, OCRS, dp, style, color['ghost'])
  
  # Plot path:
  clr = color['fill']
  rwdf = style['rwd_fill']
  deco = False
  plot_paths(c, [ph,], dp, [clr,], rwdf,deco, style,color) # ??? Should plot only arrows without axes ???

  # Plot the path endpoints:
  plot_path_endpoints(c, [ph,], dp, style,color)

  # Plot selected contact(s).
  interbc = False
  shadow = True
  trim = False
  plot_contacts(c, CTScold, dp, interbc, shadow,trim, style,color)
    
  # Label path endpoints:
  tdA = (-1.8, -1.6)
  tdB = (-1.6, +0.5) if alt else (+0.8, +0.8)
  pfsize = style['fullfsize']
  plot_path_endpoint_labels(c, ph, "A", tdA, "B", tdB, dp, pfsize, style,color)

  return c
  # ----------------------------------------------------------------------

def get_cold_paths(OPHS):
  # Retruns a list {CRSS} that describes a typical tool-path produced by
  # {RP3} or {slic3r}. Each element of the list {CRSS} specifies the
  # rasters that comprise a snake sub-path ("continuous raster sequence)
  # of that tool-path. It is a list of pairs {(isc,jrs)}, meaning raster {jrs}
  # of scanline {isc}.
  
  CRSS = ( 
    ( (0,0), (1,0), (2,0), (3,0), (4,0), (5,0), (6,0), (7,0), (8,0), 
      (9,0), (10,0), (11,0), (12,0), (13,0), (14,0),
      (15,0), (16,0), (17,0),
    ),
    ( (14,1), (13,1), (12,1), (11,1), (10,1), (9,1), (8,1), (7,1), (6,1), (5,1), (4,1), (3,1), ),
    ( (0,1), (1,1), (2,1), (3,3), (4,3), (5,3), (6,3), (7,2), (8,2), 
      (9,3), (10,3), (11,3), (12,3), (13,3), (14,3), (15,1),
    ),
    ( (14,2), (13,2), ),
    ( (12,2), (11,2), (10,2), (9,2), ),
    ( (6,2), (5,2), (4,2), (3,2), ),
  )
  return CRSS
  # ----------------------------------------------------------------------
  
def plot_figure_cold(subfig, style,color):
  # Plots the figures with scanline path, alternating or not according to {alt}: 
  #
  #   subfig = "path" the path, colored by CRS, with coldest contact(s)
  #
  # Returns the canvas {c} with the figure.

  assert subfig == "path", ("invalid subfig %s" % subfig)
  
  # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OCRS,OPHS,OLKS,CTS,VGS,EGS = paper_example_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 
  
  dp = (0,0)
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  autoscale = True
  c = make_figure_canvas(Bfig, autoscale, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Collect the rasters in the guessed {RP3} or {slic3r} order.
  CRSS = get_cold_paths(OPHS)
  
  def choose_direction(p, ij):
    # Returns true if the raster identified by the index pair {ij}
    # should be reversed, based on which end is closer to {p}.
    isc,jrs = ij
    oph = OPHS[SCS[isc][jrs]]
    d0 = rn.dist(p, path.pini(oph))
    d1 = rn.dist(p, path.pfin(oph))
    return d0 > d1
    # ....................................................................

  # Join each Alternate directions between adjacent scanlines.
  p_prev = (15,0) # Last position of the nozzle.
  EFS = []
  for CRS in CRSS:
    # Decide orientation {rev} of first raster in {CRS}
    rev = choose_direction(p_prev, CRS[0])

    # Now collect the raster fill paths specified by {CRS}, alternating directions:
    for isc,jrs in CRS:
      oph = OPHS[SCS[isc][jrs]]
      if rev: oph = path.rev(oph)
      EFS.append(oph)
      p_prev = path.pfin(oph)
      rev = not rev
  
  # Assemble the path:
  use_links = True
  ph = path.concat(EFS, use_links, mp_jump)
      
  # Find the coldest contact and its cooling time:
  ncold = 2 # Number of coldest contacts to show.
  CTScold = find_coldest_contacts(ph, CTS, ncold)

  show_path_statistics([ph,], CTScold)
  
  # SPlit at jumps for coloring:
  OSKS, JMPS = path.split_at_jumps(ph)
  Ylum = color['Ytrace']
  CLRS = hacks.trace_colors(len(OSKS), Ylum)

  plot_contours(c, OCRS, dp, style, color['ghost'])

  # Plot trace components:
  rwdf = style['rwd_fill']
  deco = False
  plot_paths(c, OSKS, dp, CLRS, rwdf,deco, style,color)

  # Plot the jumps:
  plot_jumps(c, JMPS, dp, style,color)
  
  # Plot the path endpoints:
  plot_path_endpoints(c, [ph,], dp, style,color)

  # Plot selected contact(s).
  interbc = False
  shadow = True
  trim = False
  plot_contacts(c, CTScold, dp, interbc, shadow,trim, style,color)
  
  # Label path endpoints:
  tdA = (+0.8, -1.4)
  tdB = (-2.0, -0.4)
  pfsize = style['fullfsize']
  plot_path_endpoint_labels(c, ph, "A", tdA, "B", tdB, dp, pfsize, style,color)
 
  return c
  # ----------------------------------------------------------------------

def get_rivers(OPHS, CTS):
  # Retruns a list {GROUPS} that describes the rivers in the filling raster set {OPHS} 
  # defined by the contacts {CTS}. Each element of the list {GROUPS} specifies the
  # rasters that comprise a river. It is a list of pairs {(isc,jrs)}, meaning raster {jrs}
  # of scanline {isc}.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  GROUPS = ( 
    ( (0,0), (1,0), (2,0), ),
    ( (3,0), (4,0), (5,0), (6,0), (7,0), (8,0), (9,0), (10,0), (11,0), (12,0), (13,0), (14,0), ),
    ( (3,1), (4,1), (5,1), (6,1), (7,1), (8,1), (9,1), (10,1), (11,1), (12,1), (13,1), (14,1), ),
    ( (15,0), (16,0), (17,0), ),
    ( (14,2), (13,2), ),
    ( (13,3), (14,3), (15,1), ),
    ( (9,2), (10,2), (11,2), (12,2), ),
    ( (9,3), (10,3), (11,3), (12,3), ),
    ( (7,2), (8,2), ),
    ( (3,3), (4,3), (5,3), (6,3), ),
    ( (3,2), (4,2), (5,2), (6,2), ),
    ( (0,1), (1,1), (2,1), ),
  )
  return GROUPS
  # ----------------------------------------------------------------------

def get_sub_rivers(OPHS, CTS):
  # Retruns a list {GROUPS} that describes the sub-rivers in the filling raster set {OPHS} 
  # defined by the contacts {CTS}.
  #
  # Each element of the list {GROUPS} specifies the rasters that comprise
  # a sub-river. It is a list of pairs {(isc,jrs)}, meaning raster {jrs} of
  # scanline {isc}.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  GROUPS = ( 
    ( (0,0), (1,0), (2,0), ),
    ( (0,1), (1,1), (2,1), ),

    ( (3,0), (4,0), (5,0), (6,0), ), 
    ( (3,1), (4,1), (5,1), (6,1), ),
    ( (3,2), (4,2), (5,2), (6,2), ),
    ( (3,3), (4,3), (5,3), (6,3), ),

    ( (7,0), (8,0), ),
    ( (7,1), (8,1), ),
    ( (7,2), (8,2), ),

    ( (9,0), (10,0), (11,0), (12,0), ),
    ( (9,1), (10,1), (11,1), (12,1), ),
    ( (9,2), (10,2), (11,2), (12,2), ),
    ( (9,3), (10,3), (11,3), (12,3), ),

    ( (13,0), (14,0), ),
    ( (13,1), (14,1), ),
    ( (13,2), (14,2), ),
    ( (13,3), (14,3), ),

    ( (15,0), ),
    ( (15,1), ),

    ( (16,0), (17,0), ),
  )
 
  return GROUPS
  # ----------------------------------------------------------------------
 
def get_essential_cut_lines(OPHS, CTS):
  # Returns a list {LNS} of essential cut-lines.
  # Each element of {LNS} is a pair {(icu, t)} where {icu} is the index of the scan-line just above 
  # the cut-line and {t} is a code (2 for essential, 1 for non-essential, 0 for not relevant). 

  LNS = (
    (0,2),
    (3,2),
    (7,2),
    (9,2),
    (13,2),
    (15,2),
    (16,2),
    (18,2),
  )

  return LNS
  # ----------------------------------------------------------------------

def plot_figure_rivers(subfig, style,color):
  # Plots the figure with rasters grouped into rivers.
  #
  #   subfig == "whole" Shows the rivers, color-coded, and the separating contacts.
  #   subfig == "subs"  Shows the sub-rivers defined by the essential scan-lines.
  #
  # Returns the canvas {c} with the figure.
 
  assert subfig == "whole" or subfig == "subs", ("invalid subfig %s" % subfig)
  
  # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OCRS,OPHS,OLKS,CTS,VGS,EGS = paper_example_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 
 
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  dp = (0,0)
  autoscale = True
  c = make_figure_canvas(Bfig, autoscale, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Assign group indices to rivers.
  # Each element of the list {GROUPS} below is a river. 
  # Each river is a list of pairs {(isc,jrs)}, meaning raster {jrs} of scanline {isc}.
  
  if subfig == "whole":
    GROUPS = get_rivers(OPHS, CTS)
  elif subfig == "subs":
    GROUPS = get_sub_rivers(OPHS, CTS)
  else:
    assert False, "invalid subfig"
  
  ngr = len(GROUPS)
  igr = 0
  for RIV in GROUPS:
    # Assign group index {igr} to rasters in {RIV}:
    for isc,jrs in RIV:
      oph = OPHS[SCS[isc][jrs]]
      assert path.nelems(oph) == 1
      path.set_group(oph, igr)
    igr += 1
  assert igr == ngr
  
  plot_contours(c, OCRS, dp, style, color['ghost'])

  # Make a color list, one for each river:
  Ylum = color['Ytrace']
  CLRS = hacks.trace_colors(ngr, Ylum)

  # Plot the filling path(s) with color based on group:
  rwdf = style['rwd_fill']
  deco = False
  for oph in OPHS:
    igr = path.get_group(oph)
    assert igr != None, "group was not assigned"
    clr = CLRS[igr]
    plot_paths(c, [oph,], dp, [clr,], rwdf,deco, style,color)

  if subfig == "whole":
    interbc = True # Only inter-block contacts please
    shadow = True
    trim = False
    plot_contacts(c, CTS, dp, interbc, shadow,trim, style,color)
  elif subfig == "subs":
    LNS = get_essential_cut_lines(OPHS, CTS);
    plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, dp, style,color)
  else:
    assert False, "invalid subfig"

  return c
  # ----------------------------------------------------------------------
 
def get_canon_cut_lines(OPHS, CTS, icumax):
  # Returns a list {LNS} of essential and non-essential cut-lines to plot on the canonical path figure.
  # Each element of {LNS} is a pair {(icu, t)} where {icu} is the index of the scan-line just above 
  # the cut-line and {t} is a code (2 for essential, 1 for non-essential, 0 for not relevant). 

  LNS = get_essential_cut_lines(OPHS, CTS)
  LNS += (
      (5,1),
      (12,1),
    )

  # Remove excess cut lines:
  LNS = [ (icu, t) for icu, t in LNS if icu <= icumax ]

  return LNS
  # ----------------------------------------------------------------------

def get_canon_blocks(OPHS,SCS,icumax,full,mp_jump):
  # Returns a list of lists {BCSS} of the canonical snake blocks
  # for the canonical path figure, and a list {RSrest} with the rasters
  # of {OPHS} not used in the snakes.
  #
  # The parameter {OPHS} should be a list of all the filling raster elements.
  # each element oriented from left to right.
  #
  # The parameter {SCS} should be a list of lists of indices. The raster
  # element {jrs} (from 0, left to right) on scanline {isc}, as returned by
  # {raster.separate_by_scanline}.
  # 
  # The returned snake blocks are separated according to the canonical bands
  # defined by the cut-lines of {get_canon_cut_lines}, up to the cut-line {icumax}.
  # Each element {BCSS[ibd]} of the result is a list of the blocks that
  # appear in one of those bands. The returned blocks will use all the rasters
  # up to the cut-line {icumax}.  The rasters of {OPHS} above that cut-line 
  # are returned in {RSrest}.
  # 
  # Each block will be constructed from a canonical sub-river -- a
  # subset of of raster paths in {OPHS} that lie on adjacent scan-lines,
  # with exactly one contact between conscutive rasters, and span the
  # band in quation. The rasters will be connected by the link paths
  # attached to those rasters, if any.
  # 
  # If {full} is true, the choices of each block will be all the snake
  # paths that can be built from those rasters (either 2 or 4, snake
  # paths, depending on the number of scan-lines that it spans)
  #
  # If {full} is false, each block will have only one choice -- the one 
  # that starts with the bottom-most raster, oriented from left to right.
  # 
  # The links from the input rasters are copied to the block choices
  # when applicable.
  #
  # No new {Contact} objects are created, but the attachments between
  # the existing contacts and paths of {OPHS} that are used in blocks,
  # as given by {path.get_contacst} and {contact.get_side_paths}, are
  # broken and the contacts are attached to the choices of the blocks
  # instead.
  #
  # The procedure also sets group index (as returned by
  # {path.get_group}) of all original raster paths in {OPHS} to the
  # sequential index of the block in which they were used; or to {None}
  # if they were not used in any block.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  
  # {SNIXSSS} is a list of list of list of pairs. If
  # {SNIXSSS[ibd][isn][irs]} is {(isc,jrs)}, it means that raster {irs}
  # (from bottom) of snake {isn} (from left) on band {ibd} (from bottom)
  # is raster {jrs} of scanline {isc}.
  SNIXSSS = ( 
    # band (0,3):
    ( ( (0,0), (1,0), (2,0), ),
      ( (0,1), (1,1), (2,1), ),
    ),

    # Band (3,5):
    ( ( (3,0), (4,0), ), 
      ( (3,1), (4,1), ),
      ( (3,2), (4,2), ),
      ( (3,3), (4,3), ),
    ),

    # Band (5,7):
    ( ( (5,0), (6,0), ),
      ( (5,1), (6,1), ),
      ( (5,2), (6,2), ),
      ( (5,3), (6,3), ),
    ),

    # Band (7,9):
    ( ( (7,0), (8,0), ),
      ( (7,1), (8,1), ),
      ( (7,2), (8,2), ),
    ),

    # Band (9,12):
    ( ( (9,0), (10,0), (11,0), ),
      ( (9,1), (10,1), (11,1), ),
      ( (9,2), (10,2), (11,2), ),
      ( (9,3), (10,3), (11,3), ),
    ),
    
    # Band (12,13):
    ( ( (12,0), ),
      ( (12,1), ),
      ( (12,2), ),
      ( (12,3), ),
    ),
    
    # Band (13,15):
    ( ( (13,0), (14,0), ),
      ( (13,1), (14,1), ),
      ( (13,2), (14,2), ),
      ( (13,3), (14,3), ),
    ),

    # Band (15,16):
    ( ( (15,0), ),
      ( (15,1), ),
    ),
    
    # Band (16,18):
    ( ( (16,0), (17,0), ), 
    ),
  )
  
  # Clear the goup index of all raster paths in {OPHS}:
  for oph in OPHS: path.set_group(oph, None)

  nbc = 0 # Number of snake blocks created.
  
  BCSS = [] # List of lists of canonical snake blocks.
  RSrest = []
  for SNIXSSk in SNIXSSS:
    # SNIXSSk is a list of lists of rasters (represented as index pairs) in the band with index {k},
    # separated by snake. Create snake blocks of that band:
    BCSk = [] # List of canonical snake blocks in band.
    for SNIXSkj in SNIXSSk:
      # SNIXSkj is a list of index pairs of the rasters in one canonical snake block of that band.
      # Create the list of blocks {BCSk}
      OPBSj = collect_and_tag_snake_rasters(SNIXSkj, SCS, OPHS, icumax, nbc, RSrest)
      bckj = make_snake_block_from_rasters(OPBSj, full, nbc, mp_jump)
      if bckj != None: 
        BCSk.append(bckj) 
        nbc += 1
    if len(BCSk) != 0: BCSS.append(BCSk)
  
  sys.stderr.write("got %d bands of blocks and %d leftover rasters\n" % (len(BCSS), len(RSrest)))
  for ibd in range(len(BCSS)):
    SNB = BCSS[ibd]
    sys.stderr.write("  band %d has %d blocks\n" % (ibd,len(SNB)));

  return BCSS, RSrest
  # ----------------------------------------------------------------------

def get_generic_blocks(OPHS,SCS,full,mp_jump):
  # Returns a collection of snake blocks built from the raster paths in
  # {OPHS}, as could be the input to
  # {BestHotPath}. 
  #
  # For compatibility with {get_canon_blocks}, the procedure returns a
  # list {BCSS} where each element is not a block but a list that
  # contains a single block. It also also returns an (empty) list
  # {RSrest} of unused rasters, for the same reason.
  #
  # The parameter {OPHS} should be a list of all the filling raster elements.
  # each element oriented from left to right.
  #
  # The parameter {SCS} should be a list of lists of indices. The raster
  # element {jrs} (from 0, left to right) on scanline {isc}, as returned by
  # {raster.separate_by_scanline}.
  # 
  # Each block will be constructed from a subset of rasters from {OPHS}
  # that lie on adjacent scan-lines, with exactly one contact between
  # conscutive rasters. The rasters will be connected by the link paths
  # attached to those rasters, if any. Apart from that constraint, the
  # partition of rasters into blocks is arbitrarily chosen by the
  # procedure
  # 
  # If {full} is true, the choices of each block will be all the snake
  # paths that can be built from those rasters (either 2 or 4, snake
  # paths, depending on the number of scan-lines that it spans)
  #
  # If {full} is false, each block will have only one choice,
  # chosen arbitrarily.
  # 
  # The links from the input rasters are copied to the block choices
  # when applicable.
  #
  # No new {Contact} objects are created, but the attachments between
  # the existing contacts and paths of {OPHS} that are used in blocks,
  # as given by {path.get_contacst} and {contact.get_side_paths}, are
  # broken and the contacts are attached to the choices of the blocks
  # instead.
  #
  # The procedure also sets group index (as returned by
  # {path.get_group}) of all original raster paths in {OPHS} to the
  # sequential index of the block in which they were used; or to {None}
  # if they were not used in any block.
  
  # !!! Should use {raster_regroup.split_by_group} !!!
  
  # {SNIXSS} is a list of lists of pairs.  If {SNIXSS[ibc][irs]} is {(isc,jrs)}, 
  # it means that raster {irs} (from bottom) of block {ibc} 
  # is raster {jrs} of scanline {isc}.
  SNIXSS = ( 
    ( (0,0), (1,0), (2,0), ),
    ( (0,1), (1,1), (2,1), ),

    ( (3,0), (4,0), (5,0), ), 
    ( (6,0), (7,0), (8,0), ),
    ( (9,0), (10,0), (11,0), ),
    ( (12,0), (13,0), (14,0), ),
    
    ( (3,1), (4,1), (5,1), (6,1), (7,1), (8,1), ),
    ( (9,1), (10,1), (11,1), (12,1), (13,1), (14,1), ),

    ( (3,2), (4,2), (5,2), (6,2), ),

    ( (7,2), (8,2), ),

    ( (9,2), (10,2), ),
    ( (11,2), (12,2), ),
    ( (13,2), (14,2), ),

    ( (3,3), (4,3), (5,3), (6,3), ),
    ( (9,3), (10,3), (11,3), (12,3), ),

    ( (13,3), (14,3), (15,1), ),

    ( (15,0), (16,0), (17,0), ), 
  )
  
  # Clear the goup index of all raster paths in {OPHS}:
  for oph in OPHS: path.set_group(oph, None)
  
  nsc = len(SCS) # Number of scan-lines.
  nbc = 0 # Number of blocks created.
  
  BCSS = [] # List of lists of canonical snake blocks.
  RSrest = []
  for SNIXSj in SNIXSS:
    # SNIXSj is a list of index pairs, specifying the rasters of one block.
    # Create the block {bcj}
    icumax = nsc # Cut-line above last scan-line.
    OPBSj = collect_and_tag_snake_rasters(SNIXSj, SCS, OPHS, icumax, nbc, RSrest)
    bcj = make_snake_block_from_rasters(OPBSj, full, nbc, mp_jump)
    assert bcj != None
    BCSS.append([bcj,]) 
    nbc += 1

  # All rasters must have been used:
  assert len(RSrest) == 0
  
  sys.stderr.write("got %d blocks\n" % len(BCSS))
  for jbc in range(len(BCSS)):
    bcj = BCSS[jbc][0]
    if full:
      assert isinstance(bcj, block.Block)
      nchj = block.nchoices(bcj)
      nrsj = path.nelems(block.choice(bcj,0))
      sys.stderr.write("  block %d has %d choices of %d rasters\n" % (jbc,nchj,nrsj))
    else:
      assert isinstance(bcj, path.Path)
      nrsj = path.nelems(bcj)
      sys.stderr.write("  path %d has %d rasters\n" % (jbc,nrsj));

  RSrest = []
  return BCSS, RSrest
  # ----------------------------------------------------------------------

def collect_and_tag_snake_rasters(SNIXS, SCS, OPHS, icumax, ibc, RSrest):
  # Parameters {SCS}, {OPHS}, {icumax} are as in {get_canon_blocks}.
  # Parameter {SNIXS} is a list of pairs {(isc,jrs)} that identify the rasters of 
  # one canonical snake block.  Returns the list {OPKS} of those raster paths,
  # in the same orientation as they are in {OPHS}.
  #
  # However, only rasters below the cut-line {icumax} are used in the snake block.
  # Rasters above that cut-line are instead appended to the list {RSrest} 
  # with no change.  If all rasters specified by {SNIXS} are above that cut-line,
  # returns the empty list.
  #
  # Also sets the group index of all the selected raster paths to {ibc}.

  OPKS = [] # Raster elements in this snake.
  for isc, jrs in SNIXS:
    ors = OPHS[SCS[isc][jrs]]
    if isc >= icumax:
      RSrest.append(ors)
    else:
      path.set_group(ors, ibc)
      OPKS.append(ors)
  return OPKS
  # ----------------------------------------------------------------------

def make_snake_block_from_rasters(OPBS, full, ibc, mp_jump):
  # Builds and returns a snake block from the raster paths in the list {OPBS}. They should be in consecutive scan-lines
  # and all oriented from left to right.  
  #
  # The raster paths in {OPBS} should all have group index {ibc}, which
  # should be distinct from the groups of al other original raster paths
  # and of all choices of previously created blocks.
  #
  # If {full} is true, returns a block whose choices are all snake paths
  # that can be built from the rasters on {OPBS}; usually 2 if {OPBS}
  # has a single raster, and 4 if it has two or more. If {full} is
  # false, returns a block with only one choice, specifically the snake
  # path that begins with the bottom-most raster oriented from left to
  # right.
  #
  # However, if the list {OPBS} is empty, returns {None}.
  #
  # Also copies the applicable link paths from the raster paths in {OPBS}
  # to the choices of the new block.
  #
  # Also modifies the contact-path attachments of to refer to the block
  # choices instead of those raster paths. Specifically, any contact
  # {ct} that is attached to a raster path {ors} of {OPBS} and to some
  # path not in {OPBS} (original raster or choice of a previosuly
  # created block) will be detached from {ors} and attached to all the
  # choices of the new block.

  # Collect raster paths {OPHS0,OPHS1} of the snake and (if {full}) of the "mirror" snake:
  OPHS0 = [] # Raster elements in this snake.
  OPHS1 = [] # Reversed raster elemens in this snake (if {full}).
  rev = False # Should reverse the next raster?
  for opb in OPBS:
    assert path.get_group(opb) == ibc
    if rev: opb = path.rev(opb)
    OPHS0.append(opb)
    if full: OPHS1.append(path.rev(opb))
    rev = not rev

  # Did we get any rasters at all?
  if len(OPHS0) == 0: return None
  
  # Now join the rasters in each of {OPHS0} and {OPHS1} into a snake block
  use_links = True
  ph0 = path.concat(OPHS0, use_links, mp_jump) # The snake path.
  if not full: 
    bc = block.from_paths([ph0,])
  else:
    ph1 = path.concat(OPHS1, use_links, mp_jump) # The other snake path.
    if len(OPHS0) == 1:
      bc = block.from_paths([ph0, ph1])
    else:
      bc = block.from_paths([ph0, path.rev(ph0), ph1, path.rev(ph1)])

  reattach_snake_block_contacts(OPHS0, bc, ibc)
 
  for ich in range(block.nchoices(bc)): path.set_group(block.choice(bc, ich), ibc)

  return bc
  # ----------------------------------------------------------------------
 
def reattach_snake_block_contacts(OPHS, bc, ibc):
  # Given a list of raster-fill paths {OPHS} that were used to make one
  # snake block. detaches them from their attached contacts and attaches
  # the latter to the choices of the block {bc}.
  #
  # On input, all the raster paths in {OPHS} should have group index
  # {ibc}, and all other paths (original raster paths or choices of
  # previoulsy created blocks) must have group index {None} or strictly
  # less than {ibc}.  The group index of the choices of {bc} is not used.

  debug = False

  # Get all the contacts {CTS} whose sides were used in this block:
  CTS = set()
  for ophi in OPHS:
    for isd in range(2):
      CTS = set.union(CTS, path.get_contacts(ophi, isd))
  
  # Fix the attachments of all those contacts:
  for ct in CTS:
    if debug: contact.show(sys.stderr, "  ct = ", ct, "\n", 0)
    for ksd in range(2):
      # Get the paths that are attached to side {ksd} of {ct}.
      # These are either a single old raster path, possibly used in this block, 
      # or one or more choices of a previously conctructed block.
      SDPSk = contact.get_side_paths(ct, ksd)
      assert len(SDPSk) > 0 # Since the contact must originally have had have both sides on raster paths of {OPHSD}.
      
      # Set {ibck} to the index of the block that contains side {k} of {ct} (which may be {ibc})
      # or to {None} if that side is still not in any block:
      # sys.stderr.write("ibc = %d contact %s side %d SPDSk = %s:\n" % (ibc, contact.get_name(ct),isd, str(SDPSk)))
      igrk = None
      for phkj, drkj, imvkj in SDPSk:
        igrkj = path.get_group(phkj)
        if debug: path.show(sys.stderr, ("    side %d @ " % isd), phkj, (" igr = %s\n" % str(igrkj)), True, 0,0)
        # sys.stderr.write("  igrkj = %s igrk = %s\n" % (str(igrkj),str(igrk)))
        if igrkj == ibc or igrkj == None:
          # This side of the contact is in one of original rasters, either used by this
          # block or not yet used by any block:
          assert igrk == None # Since there should be only one such raster.
          assert len(SDPSk) == 1 # Since there should be only one such raster.
          assert imvkj == 0 # Since those original rasters paths have length 1.
        else:
          # This side of the contact is in a choice of a previously constructed block.
          assert igrkj < ibc # Assuming blocks are numbered consecutively by creation.
          assert igrk == None or igrk == igrkj # Since all side paths must be in the same block.
        igrk = igrkj 
        
      if igrk == ibc:
        # Path attachments on side {ksd} of {ct} must be changed from old raster to choices of new block:
        contact.clear_side_paths(ct, ksd)
        for phkj, drkj, imvkj in SDPSk: path.clear_contacts(phkj, ksd)
          
        # Attach side {ksd} of {ct} to the choices of {bc}
        mvk = contact.side_move(ct, ksd)
        for ich in range(block.nchoices(bc)):
          ochi = block.choice(bc, ich)
          jmvki = path.find_move(ochi, mvk)
          if jmvki != None:
            path.add_contact(ophi, ksd, ct)
            contact.add_side_path(ct, ksd, ochi, jmvki)
  return
  # ----------------------------------------------------------------------

def plot_figure_canon(subfig, style,color):
  # Plots the figure with canonical paths and canonical bandpath.
  #
  # Returns the canvas {c} with the figure.
  
  assert subfig == "paths", ("invalid subfig %s" % subfig)

  # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OCRS,OPHS,OLKS,CTS,VGS,EGS = paper_example_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 
  
  Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)
  autoscale = True
  c = make_figure_canvas(Bfig, autoscale, style,color)

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  
  # Build a list of list of blocks {BCSS} with one choice each, the canonical snake,
  # up to some cut-line {icumax} below the top one, separated into bands:
  
  dp = None
  icumax = 12
  full = False  
  BCSS, RSrest = get_canon_blocks(OPHS,SCS,icumax,full,mp_jump)

  plot_contours(c, OCRS, dp, style, color['ghost'])

  # Plot the relevant cut-lines that separate the chosen bands:
  LNS = get_canon_cut_lines(OPHS, CTS, icumax);
  plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, dp, style,color)
  
  # Concatenate the canonical snake paths in {BCSS} into a {(k,i)} canonical path
  # {cph}, up to the next-to-lasr band, and the following {(i,j)} canonical
  # band-path {bph}:
  nbd = len(BCSS)
  CPS = [] # List of canonical snake paths that make up {cph}
  BPS = [] # List of canonical snake paths that make up {bph}
  for ibd in range(nbd):
    for bc in BCSS[ibd]: # Blocks in band {ibd}
      assert block.nchoices(bc) == 1 # Since {full} was false.
      och = block.choice(bc, 0)
      (BPS if ibd == nbd-1 else CPS).append(och)
  use_links = True
  cph = path.concat(CPS, use_links, mp_jump)
  bph = path.concat(BPS, use_links, mp_jump)
  
  # Plot the canonical path and the canonical sub-path:
  rwdf = style['rwd_fill']
  ccph = color['fill'] # Color for the canonical {(k,i)} path.
  deco = False
  plot_paths(c, [cph,], dp, [ccph,], rwdf,deco, style,color)

  cbph = color['hifi'] # Color for the canonical {(i,j)} band-path.
  deco = False
  plot_paths(c, [bph,], dp, [cbph,], rwdf,deco, style,color)
  
  # Plot dots at the path endpoints:
  plot_path_endpoints(c, [cph,], dp, style,color)
  plot_path_endpoints(c, [bph,], dp, style,color)

  # Plot the unused rasters:
  crest = color['ghost']
  deco = False
  for oph in RSrest:
    plot_paths(c, [oph,], dp, [crest,], rwdf,deco, style,color)
    
  # Plot the path endpoint labels:
  pfsize = style['fullfsize']
  tdA = (-1.8, -1.6)
  tdB = (-1.8, -0.4)
  plot_path_endpoint_labels(c, cph, "A", tdA, "B", tdB, dp, pfsize, style,color)
  tdC = (-2.0, -0.1)
  tdD = (+0.7, -1.0)
  plot_path_endpoint_labels(c, bph, "C", tdC, "D", tdD, dp, pfsize, style,color)
  
  # ??? Should plot the contacts between {cph} and {bph} ???

  return c
  # ----------------------------------------------------------------------

def plot_figure_blocks(subfig, style,color):
  # A figure showing rasters partitioned into blocks or pre-blocks ("If").
  # 
  #   {subfig} == "canstacks", plots the rasters classified into stacks with essential and some non-essential cut-lines.
  #   {subfig} == "canblocks", plots those stacks turned into blocks, as would be created from the output of {HotPath}
  #   {subfig} == "genblocks", plots the figure with the generic blocks, including links and contacts. 
  #   {subfig} == "genchoices", plots the four alternatives of a selected block from the generic blocks figure. 
  #   {subfig} == "gengraph", plots the contact graphs of the generic blocks. 
  #
  # Returns the canvas {c} with the figure.

  # Get the contours, fillers, links, contacts, and graph of the "turkey" part:
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();
  OCRS,OPHS,OLKS,CTS,VGS,EGS = paper_example_B.make_turkey(mp_cont, mp_fill,mp_link,mp_jump) 

  # Separate the rasters by scan-line:
  xdir = (1,0)
  ydir = (0,1)
  ystep, yphase = raster.get_spacing_and_phase(OPHS, xdir, ydir)
  SCS = raster.separate_by_scanline(OPHS, xdir, ydir, ystep, yphase)
  nsc = len(SCS) # Number of scan-lines.
  
  # Create the blocks to be plotted, as a list {BCSS} of lists of
  # blocks. For the sub-figures with canonical blocks, each element of
  # {BCSS} is a list of the blocks of one of the bands defined by the
  # cut-lines returned by {get_canon_cut_lines}. For the sub-figures
  # with generic blocks, each element of {BCSS} is a list with a single
  # block.

  # Get the blocks, canonical or generic, possibly with links and contacts or cut-lines,
  # comprising all rasters; and pick a block {bcX} to be highlighted or expanded;

  dp = (0,0)

  if subfig == "canstacks" or subfig == "canblocks":
    # Get the canonical blocks, in all the bands up to the top.  
    # Also set the groups of the rasters in {OPHS} to the block index:
    full = True  # Generate blocks with all possible choices.
    icumax = nsc # Go all the way to the top.
    BCSS, RSrest = get_canon_blocks(OPHS,SCS,icumax,full,mp_jump)
    assert len(RSrest) == 0
    # No selected block:
    ibcX = None
  elif subfig == "genblocks" or subfig == "genchoices" or subfig == "gengraph":
    # Created generic blocks, sub-rivers but not defined by cut-lines:
    full = True
    BCSS, RSrest = get_generic_blocks(OPHS,SCS,full,mp_jump)
    # Select a block to highlight in the generic blocks figure:
    ibcX = 13
  else:
    assert False, ("invalid subfig = %s" % subfig)

  # Make a flat list {BCS} of all blocks:
  BCS = [ bc for BCSi in BCSS for bc in BCSi ]
  nbc = len(BCS) # Number of blocks.
  bcX = None if ibcX == None else BCS[ibcX]
  
  # Paranoia:
  for ibc in range(len(BCS)):
    bc = BCS[ibc]
    for ich in range(block.nchoices(bc)):
      och = block.choice(bc,ich)
      assert path.get_group(och) == ibc
  
  # Determine the figure's bounding box {Bfig} and the cut-line X range {cuxlo,cuxhi}:
  if subfig == "genchoices":
    # Bounding box is for the block {bcX} only:
    assert bcX != None
    B = block.bbox([bcX,], True, False)
    Xstep = B[1][0] - B[0][0] + 3.0 # # Displacement between choices.
    Bfig = (B[0], rn.add(B[1], (3*Xstep, 0)))
    wdf = style['wd_fill']
    mrg = (1.0*wdf, 1.0*wdf) # To account for sausage overflow and links.
    Bfig = rn.box_expand(Bfig, mrg, mrg)
    cuxlo = None; cuxhi = None
  else:
    # Bounding box is for the whole slice including contours.
    # Assume that links are contained in that box.
    B = path.bbox(OCRS+OPHS)
    Bfig, cuxlo, cuxhi = get_turkey_figure_bbox(OCRS, OPHS, OLKS, style)

  autoscale = True
  c = make_figure_canvas(Bfig, autoscale, style,color)

  if subfig == "canstacks" or subfig == "canblocks" or subfig == "genblocks" or subfig == "gengraph":
  
    plot_figure_blocks_whole(c, subfig, OCRS, OPHS, CTS, BCS, ibcX, dp, cuxlo,cuxhi, nsc,ystep,yphase, style,color)
    
  elif subfig == "genchoices":
  
    assert bcX != None
    plot_figure_blocks_choices(c, subfig, bcX, dp, Xstep, nsc,ystep,yphase, style,color)
    
  else:
    assert False, ("invalid subfig %s" % subfig)

  return c
  # ----------------------------------------------------------------------

def plot_figure_blocks_whole(c, subfig, OCRS, OPHS, CTS, BCS, ibcX, dp, cuxlo,cuxhi, nsc,ystep,yphase, style,color):
  # Plots version {subfig} of the "blocks" figure, where {subfig is not 3.
  # The parameters {OCRS,OPHS,CTS} as as returned by {paper_example_B.make_turkey}.
  # The {BCS} parameter is a flat list of full-choice blocks buit from those rasters;
  # {ibcX} is {None} or the index of a block to be highlighted; {nsc} is the number of 
  # scan-lines, and {ystep,yphase} definde their positions;
  # {cuxlo,cuxhi} is the X-range of the cut-lines (excluding their labels).

  plot_contours(c, OCRS, dp, style, color['ghost'])
  
  nbc = len(BCS)

  # Plot either the cut-lines for the canonical blocks figure,
  # or the link paths for the generic blocks figure:
  if subfig == "canstacks" or subfig == "canblocks":
    # Plot the cut-lines used to define the canonical stacks or blocks:
    icumax = nsc # Go all the way to the top.
    LNS = get_canon_cut_lines(OPHS, CTS, icumax)
    plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, dp, style,color)
  elif subfig == "genblocks":
    # Plot the links beneath the blocks:
    bicolor = False
    for bc in BCS:
      plot_block_links(c, [bc,], dp, ystep,yphase, bicolor, style,color)
  elif subfig == "gengraph":
    # Neither cut-lines nor links:
    pass
  else:
    assert False

  # Plot the blocks, highlighting the selected one:

  if subfig == "canstacks":
    # Plot the rasters, colored by stacks:
    Ylum = color['Ytrace']
    CLRS = hacks.trace_colors(nbc, Ylum)
    rwdf = style['rwd_fill']
    deco = False
    for oph in OPHS:
      igr = path.get_group(oph)
      assert igr != None and igr >= 0 and igr < nbc, ("invalid raster group index %s" % str(igr))
      clr =  CLRS[igr]
      plot_paths(c, [oph,], dp, [clr,], rwdf,deco, style,color)
  
  elif subfig == "canblocks" or subfig == "genblocks" or subfig == "gengraph":
    # Plot the blocks, with uniform color:
    rwdf = style['rwd_fill']
    deco = False
    for ibc in range(nbc):
      bc = BCS[ibc]
      assert isinstance(bc, block.Block)
      clr = color['hifi'] if ibc == ibcX else color['fill']
      if subfig == "gengraph": clr = ghostify_colors([clr,], color['ghost'])[0]
      nch = block.nchoices(bc)
      for ich in range(nch):
        oph = block.choice(bc, ich)
        assert path.get_group(oph) == ibc
        plot_paths(c, [oph,], dp, [clr,], rwdf,deco, style,color)
       
  sys.stderr.write("ibcX = %s\n" % str(ibcX))
  if ibcX != None:
    bcX = BCS[ibcX]
    ptbc = block.barycenter(bcX)
    sys.stderr.write("ptbc = %s\n" % str(ptbc))
    bclab_dp = (+3.5,-1.5)
    bclab_tx = r"$B$"
    mfsize = style['fumafsize']
    plot_math_label(c, bclab_tx, ptbc, bclab_dp, dp, mfsize, style)

  if subfig == "genblocks":
    # Plot the inter-block contacts:
    interbc = True # Only inter-block contacts.
    shadow = False
    trim = True
    plot_contacts(c, CTS, dp, interbc, shadow,trim, style,color)
  elif subfig == "gengraph":
    # Compute the vertices of the contact graph as the barycenters of blocks.
    VGS = []
    for bc in BCS:
      pbar = block.barycenter(bc)
      VGS.append(pbar)
    # Compute the edges of the graph:
    EGS = []
    for ct in CTS:
      igr = contact_side_groups(ct)
      assert igr[0] != None and igr[1] != None, "contact side not in any block"
      if igr[0] != igr[1]:
        EGS.append(igr)
    # Plot the graph:
    plot_contact_graph(c, VGS, EGS, dp, style,color) 
  return
  # ----------------------------------------------------------------------
        
def plot_figure_blocks_choices(c, subfig, bcX, dp, Xstep, nsc,ystep,yphase, style,color):
  # Plots version {subfig} of the "blocks" figure, where {subfig is 3,
  # showing the choices of a specific block with the attached links. The
  # parameters {OCRS,OPHS,CTS} as as returned by
  # {paper_example_B.make_turkey}. The {bcX} paramerer is the block
  # whose choices are to be shown, and {ystep,yphase} defines the
  # position of scan-lines.
    
  assert subfig == "genchoices"

  assert block.nchoices(bcX) == 4
  nch = block.nchoices(bcX)

  # Plot the block choices and links:
  clr = color['hifi']
  rwdf = style['rwd_fill']
  deco = True
  for ich in range(nch):
    dpi = rn.add(dp, ( ich*Xstep, 0.0 ))
    ophi = block.choice(bcX, ich)

    # Plot the matter shadow of the choice and links:
    plot_trace_matter(c, [ophi,], dpi, style,color)
    for olk in path.get_links(ophi) + path.get_links(path.rev(ophi)):
      plot_trace_matter(c, [olk,], dpi, style,color)

    # Plot the choice and links:
    bicolor = False
    plot_path_links(c, [ophi,], dpi, ystep,yphase, bicolor, style,color)
    plot_paths(c, [ophi,], dpi, [clr,], rwdf,deco, style,color)
    # ??? Should show the attached contacts too ???
  return
  # ----------------------------------------------------------------------
  
# ######################################################################
# PLOTTING STUFF

def plot_paths(c, OPHS, dp, CLRS, rwd,deco, style,color):
  # Plot the traces and jumps of the paths {OPHS}, /without/ the matter traces.
  # The trace widths are reduced by {rwd} from what their {MoveParms} records say.
  # If {deco} is true, plots axes, arrows, and endpoints of all moves.
  # If {deco} is false, plots arrows and endpoints only of jumps.
  wd_axes = style['wd_axes']
  axes = deco
  dots = deco
  arrows = deco
  matter = False
  path.plot_standard(c, OPHS, dp, None, CLRS, rwd,wd_axes, axes, dots, arrows, matter)
  return
  # ----------------------------------------------------------------------

def plot_path_links(c, OPHS, dp, ystep,yphase, bicolor, style,color):
  # Plots onto {c} the link paths of the paths in {OPHS}. 
  for oph in OPHS:
    for ophr in oph, path.rev(oph):
      OLKS = path.get_links(ophr)
      plot_links(c, OLKS, dp, ystep,yphase, bicolor, style,color)
  return
  # ----------------------------------------------------------------------
      
def plot_links(c, OLKS, dp, ystep,yphase, bicolor, style,color):
  # Plots the paths in {OLKS} with the proper link style.
  # The color is chosen
  # depending on the parity of the scan-line index of the lowest endpoint of the link.
  # 
  rwd_link = style['rwd_link']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  for ilk in range(len(OLKS)):
    olki = OLKS[ilk]
    # Make sure the link is directed up:
    if path.pini(olki)[1] > path.pfin(olki)[1]: olki = path.rev(olki)
    # Determine the scan-line index:
    isc = raster.point_scanline_index(path.pini(olki), (1,0), (0,1), ystep, yphase)
    clr = color['links'] if not bicolor else color['link0'] if isc % 2 == 0 else color['link1']
    path.plot_standard(c, [olki,], dp, None, [clr,], rwd_link, wd_axes, axes, dots, arrows, matter)
  return
  # ----------------------------------------------------------------------

def plot_block_links(c, BCS, dp, ystep,yphase, bicolor, style,color):
  # Plots onto {c} the link paths of the choices of the blocks in {BCS}.  The color is chosen
  # depending on the parity of the scan-line index of the lowest endpoint of the link.
  
  for bc in BCS:
    nch = block.nchoices(bc)
    for ich in range(nch):
      oph = block.choice(bc, ich)
      plot_path_links(c, [oph,], dp, ystep,yphase, bicolor, style,color)
  return
  # ----------------------------------------------------------------------
    
def plot_dot_on_path(c, p, dp, style,color):
  clr = color['dots']
  wd_edots = style['wd_edots']
  hacks.plot_line(c, p, p, dp, clr, wd_edots, None)
  return
  # ----------------------------------------------------------------------

def plot_path_endpoints(c, OPHS, dp, style,color):
  # Plots the endpoints of the paths in {OPHS}.
  for oph in OPHS:
    for p in path.endpoints(oph):
      plot_dot_on_path(c, p, dp, style,color)
  return
  # ----------------------------------------------------------------------
  
def plot_jumps(c, JMPS, dp, style,color):
  # Plots the jump moves in the list {JMPS} with the standard style and color.
  rwdf = style['rwd_fill'] # Not used, really
  clr = color['jump']
  wd_axes = style['wd_axes']
  axes = True
  dots = True
  arrows = True
  matter = False
  move.plot_standard(c, JMPS, dp, 3, [clr,], rwdf, wd_axes, axes, dots, arrows, matter) 
  return
  # ----------------------------------------------------------------------

def plot_contacts(c, CTS, dp, interbc, shadow,trim, style,color):
  # Plots the contacts in {CTS} with the style used in the paper, with color {color['ctac']}.
  # If {interbc} is true, plots only contacts between paths that have different group indices.
  # If {shadow} is true, plots a white shadow unde the contacts.
  # If {trim}, shrinks the contacts to avoid interference with link paths.
   
  wd_shad = style['wd_shad']
  wd_ctac = style['wd_ctac']
  if trim:
    wd_link = style['rwd_link']*style['wd_fill']
    ext = - (0.5*wd_link + wd_ctac) # To avoid links in some plots.
  else:
    ext = 0
  dashpat = (1.0*wd_ctac, 2.0*wd_ctac)
  sz_tics = 0
  arrows_ct = False
  ncp = 0 # Interblock contacts plotted.
  for ct in CTS:
    igr = contact_side_groups(ct)
    plotit = True if not interbc else igr[0] != igr[1]
    if plotit:
      if shadow: 
        contact.plot_single(c, ct, dp, color['white'], None, ext, wd=wd_ctac+wd_shad, sz_tic=sz_tics, arrow=arrows_ct)
      contact.plot_single(c, ct, dp, color['ctac'], dashpat, ext, wd=wd_ctac, sz_tic=sz_tics, arrow=arrows_ct)
      ncp += 1
      
  sys.stderr.write("plotted %d out of %d contacts\n" % (ncp,len(CTS)))
  return
  # ----------------------------------------------------------------------

def plot_cut_lines(c, LNS, cuxlo,cuxhi,ystep,yphase, dp, style,color):
  # The parameter {LNS} must be a list of pairs {(icu, t)}, as in the 
  # output of {get_essential_cut_lines}.
  # Plots the essential ({t=2}) and non-essential ({t=1}) cut-lines listed in {LNS}.
  # Ignores cut-lines with {t=0}
  #
  wd_cuts = style['wd_cuts'] 
  for icu, t in LNS:
    if t != 0:
      clr = color['cuess'] if t == 2 else color['cunon']
      dashpat = [ 0.100, 0.250 ] if t == 1 else None
      y = ystep*(icu-0.5) + yphase
      p = (cuxlo, y)
      q = (cuxhi, y)
      hacks.plot_line(c, p, q, dp, clr, wd_cuts, dashpat)

      # Label the cut-line:
      fsize = style['cutlfsize']
      td = (1,-0.3)
      pt = rn.add(q, td)
      tx = r"\texttt{%d}" % icu
      hacks.plot_text(c, tx, pt, dp, fsize, None)

  return
  # ----------------------------------------------------------------------
    
def plot_trace_matter(c, OPHS, dp, style,color):
  # Plot the matter traces of all the paths in {OPHS}:
  clr = color['matter']
  jmp = False     # Plot traces only.
  wd_matter = 0   # Extra width of material footprint.
  rwd_matter = style['rwd_matter']
  dashed = False
  wd_dots = 0
  sz_arrows = 0
  for ph in OPHS: 
    path.plot_layer(c, ph, dp, jmp, clr, rwd_matter, wd_matter, dashed, wd_dots, sz_arrows)
  return
  # ----------------------------------------------------------------------
  
def plot_contours(c, OCRS, dp, style, clr):
  # Plot the contour paths {OCRS} with color {clr}:

  rwd_cont = style['rwd_cont']
  wd_axes = style['wd_axes']
  axes = False
  dots = False
  arrows = False
  matter = False
  path.plot_standard(c, OCRS, dp, None, [clr,], rwd_cont, wd_axes, axes, dots, arrows, matter)
  return
  # ----------------------------------------------------------------------

def plot_math_label(c, tx, qP, qD, dp, mfsize, style):
  # Plots the text {tx} at point {dp+qP+qD} with font size {mfsize}.
  # If {dp} is {None}, assumes {(0,0)}
  
  q = rn.add(qP, qD)
  hacks.plot_text(c, tx, q, dp, mfsize, None)
  return 
  # ----------------------------------------------------------------------
    
def plot_contact_graph(c, VGS, EGS, dp, style,color):
  # Plots the contact graph given the vertices {VGS} (a list of points) and the edges {EGS}
  # (a list of pairs of indices into {VGS}).
  
  # Plot edges:
  wd_edges = style['wd_edges']
  clr_edges = color['edges']
  for e in EGS:
    p = VGS[e[0]]
    q = VGS[e[1]]
    hacks.plot_line(c, p, q, dp, clr_edges, wd_edges, None)

  # Plot vertices:
  wd_verts = style['wd_verts'] # Diameter of vertices.
  clr_verts = color['verts']
  for p in VGS:
    hacks.plot_line(c, p, p, dp, clr_verts, wd_verts, None)
  return
  # ----------------------------------------------------------------------

def plot_point_label(c, lab, p, td, dp, pfsize, style,color):
  # Places label {lab} next with the lower left corner at the point {p}
  # displaced by {td} and {dp} (if not {None}), with font size {pfsize}.  Also plots a white shadow all around, with width
  # proportional to the font size.
  
  pt = rn.add(p, td)
  tad = 0.002*pfsize
  for kx in range(5):
    for ky in range(5):
      dx = kx - 2; dy = ky - 2;
      if dx != 0 or dy != 0:
        ptk = rn.add(pt, (dx*tad, dy*tad))
        txk = r"\textbf{\textsf{%s}}" % lab
        hacks.plot_text(c, txk, ptk, dp, pfsize, 'white')
  tx = r"\textbf{\textsf{%s}}" % lab
  hacks.plot_text(c, tx, pt, dp, pfsize, None)
  return
  # ----------------------------------------------------------------------

def plot_path_endpoint_labels(c, oph, labA, tdA, labB, tdB, dp, pfsize, style,color):
  # Places labels {labA} and {labB} next to the start and end of 
  # path {oph}, displaced by {tdA} and {tdB}, respectively, with fonts size {pfsize}.
  plot_point_label(c, labA, path.pini(oph), tdA, dp, pfsize, style,color)
  plot_point_label(c, labB, path.pfin(oph), tdB, dp, pfsize, style,color)
  return
  # ----------------------------------------------------------------------

# ######################################################################
# UTILITY FUNCTIONS

def show_path_statistics(OPHS, CTS):
  # Prints the fabtimes (total, trace, jump) of all the paths in {OPHS}.
  # If {ophs} has a single path, prints the cooling times of the contacts in {CTS}
  # determined by that path.
  
  # Compute and show the fabrication times:
  tfab_tot = 0 # Total fabtime of paths in {OPHS}
  tfab_trc = 0 # Total fabtime of traces in {OPHS}
  for oph in OPHS:
    tfab_tot += path.fabtime(oph)
    OSQS, OJMS = path.split_at_jumps(oph)
    for osq in OSQS:
      tfab_trc += path.fabtime(osq)
  sys.stderr.write("total fabtime =  %7.3f\n" % tfab_tot)
  sys.stderr.write("extrusion time = %7.3f\n" % tfab_trc)
  sys.stderr.write("air time =       %7.3f\n" % (tfab_tot-tfab_trc))
    
  if len(OPHS) == 1:
    # Compute and show the cooling times:
    oph = OPHS[0]
    for ct in CTS:
      sys.stderr.write("contact %12s" % contact.get_name(ct))
      tcool = contact.tcool(oph, ct)
      if tcool == None:
        sys.stderr.write(" not closed\n")
      else:
        sys.stderr.write(" tcool = %7.3f\n" % tcool)
  return
  # ----------------------------------------------------------------------

def contact_side_groups(ct):
  # Returns the group indices of the paths that contain the two sides of {ct}.
  # Uses {contact.get_side_paths} and {path.get_group}. 
  # Each side must have at least one attached path, and all groups attached 
  # to the same side must have the same group index.
  
  debug = False
  if debug: contact.show(sys.stderr, "  ct = ", ct, "\n", 0)
  igr = [-1, -1] # Group indices on the two sides of contact:
  for isd in range(2):
    for phij, drij, imvij in contact.get_side_paths(ct, isd):
      igrij = path.get_group(phij)
      assert igrij == None or igrij >= 0
      if debug: path.show(sys.stderr, ("    side %d @ " % isd), phij, (" igr = %s\n" % str(igrij)), True, 0,0)
      if igr[isd] == -1:
        igr[isd] = igrij
      else:
        assert igr[isd] == igrij, ("side %d of contact on different groups" % isd)
    assert igr[isd] != -1, "contact has no side paths"
  return tuple(igr)
  # ----------------------------------------------------------------------

def find_coldest_contacts(oph, CTS, n):
  # Finds the {n} contacts in {CTS} that are closed by the path {oph}
  # and have the maximum cooling time.
  WORST = [] # List of pairs {(ict, tc)} where {ict} is index into {CTS} and {tc} is its cooling time.
  for ict in range(len(CTS)):
    ct = CTS[ict] 
    tc = contact.tcool(oph, ct)
    if tc != None:
      WORST.append((ict, tc))
  WORST.sort(key = lambda x: -x[1])
  if len(WORST) > n: WORST = WORST[0:n]
  CTScold = []
  if len(WORST) > 0:
    sys.stderr.write("coldest contacts (%d):\n" % len(WORST))
    for ict, tc in WORST:
      ct = CTS[ict]
      sys.stderr.write("  %10.6f s %s\n" % (tc, contact.get_name(ct)))
      CTScold.append(ct)
  else:
    assert False, "no contacts are closed by the path."
  return CTScold
  # ----------------------------------------------------------------------

def make_move_parms():
  # Returns the {MoveParms} records to use. They should be the same as 
  # used in the tests section of the paper, exccept that the trace width 
  # will be 1.0 (instead of 0.4) to match the raster spacing in the figures.
  
  ac = 3000      # Acceleration/deceleration for moves and jumps (mm/s^2).
  sp_trace = 40  # Cruise speed for traces (mm/s).
  sp_jump = 130  # Cruise speed for jumps (mm/s).
  ud = 0.05      # Trace/jump transition penalty (s).
  
  wd_cont = 0.75  # Contour trace width (mm).
  wd_fill = 1.00  # Fill trace width (mm).
  
  mp_cont = move_parms.make(wd_cont, ac, sp_trace, 0.0)
  mp_fill = move_parms.make(wd_fill, ac, sp_trace, 0.0)
  mp_link = mp_fill
  mp_jump = move_parms.make(0.0,     ac, sp_jump,  0.0)

  sys.stderr.write("printer parameters:\n")
  move_parms.show(sys.stderr, "contours = { ", mp_cont, " }\n")
  move_parms.show(sys.stderr, "filling =  { ", mp_fill, " }\n")
  move_parms.show(sys.stderr, "links   =  { ", mp_link, " }\n")
  move_parms.show(sys.stderr, "jumps =    { ", mp_jump, " }\n")

  return mp_cont, mp_fill, mp_link, mp_jump
  # ----------------------------------------------------------------------

def make_style_dict():
  # Returns a Python dict with the plot dimension parameters ({wd_fill}, {wd_axes}, etc) to be used 
  # in figures of the paper.
  
  mp_cont, mp_fill, mp_link, mp_jump = make_move_parms();

  wd_cont = move_parms.width(mp_cont)
  wd_fill = move_parms.width(mp_fill)

  wd_min = min(wd_fill,wd_cont)

  rwd_path = 0.80 # Plot trace sausages with relative width {rwd_path} on isolated path plots.
  rwd_fill = 0.50 # Plot fill trace sausages with relative width {rwd_fill} on turkey plots.

  wd_fisa = rwd_fill*wd_fill # Width of plotted sausages in fillin.

  wd_axes = 0.10*wd_fill
  wd_ctac = 0.20*wd_fill
  wd_cuts = 0.10*wd_fill

  style = {
    'detafsize':  30,             # Font size for point labels in detail and notation figures.
    'mathfsize':  36,             # Font size for math labels in notation figures.
    'fullfsize':  48,             # Font size for point labels in full-size slice figures.
    'fumafsize':  58,             # Font size for math labels in notation figures.
    'cutlfsize':  36,             # Font size for labels of cut-lines on full-size slice figure.
    'tbixfsize':  40,             # Font size for row and column indices on tableau figure.
    
    'cuext':      2.0,            # Extra left and right overhang of cut-lines, excl. label (mm).
    'culab':      2.5,            # Estimated width of cut-line labels, including space (mm).

    'wd_cont':    wd_cont,        # Nominal width of contour traces.
    'wd_fill':    wd_fill,        # Nonimal width of fill traces.

    'wd_axes':    wd_axes,        # Width of jumps and axis lines.
    'wd_ctac':    wd_ctac,        # Width of contact lines.
    'wd_shad':    wd_ctac,        # Width of white shadow under contacts.

    'wd_cuts':    wd_cuts,        # Width of cut-lines.
    'wd_edots':   0.75*wd_fisa,   # Diameter of black dots at end of paths.
    
    'wd_edges':   0.20*wd_fill,   # Width of graph edges.
    'wd_verts':   1.20*wd_fisa,   # Diameter of graph vertices.
    
    'rwd_path':   rwd_path,       # Relative width of fill trace sausages in isolated path plots.
    'rwd_fill':   rwd_fill,       # Relative width of fill trace sausages in turkey plots.
    'rwd_cont':   rwd_fill,       # Relative width of contour trace sausages in turkey plots.
    'rwd_link':   0.67*rwd_fill,  # Relative width of link trace sausages in turkey plots.
    'rwd_matter': 1.13,           # Relative width of material footprint in plots.

    'wd_mfig':    34.0,           # Standard width of figures with math formulas.
  }
  return style
  # ----------------------------------------------------------------------
  
def make_color_dict():
  # Returns a Python dict with the standard color scheme for paper figures.
  color = {
    'white':  pyx.color.rgb( 1.000, 1.000, 1.000 ),  # Color for invisible frame and label shadow.
    'black':  pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Default color.
    'matter': pyx.color.rgb( 0.900, 0.870, 0.850 ),  # Color for estimated material footprints.
    'fill':   pyx.color.rgb( 0.000, 0.900, 0.050 ),  # Color for relevant fill traces.
    'hifi':   pyx.color.rgb( 0.000, 0.700, 1.000 ),  # Color for highlighted fill traces.
    'cont':   pyx.color.rgb( 0.600, 0.700, 0.800 ),  # Color for contours, when somewhat relevant.
    'ghost':  pyx.color.rgb( 0.850, 0.850, 0.850 ),  # Color for non-relevant traces.
    'ctac':   pyx.color.rgb( 1.000, 0.200, 0.000 ),  # Color for contacts.
    'dots':   pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color of dots (maybe not used).
    'links':  pyx.color.rgb( 0.850, 0.050, 1.000 ),  # Color of links without distinction.
    'link0':  pyx.color.rgb( 0.950, 0.750, 0.000 ),  # Color of links above even scan-lines.
    'link1':  pyx.color.rgb( 0.450, 0.000, 1.000 ),  # Color of links above odd scan-line.
    'edges':  pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color or edges of contact graph
    'verts':  pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color of vertices of contact graph.
    'jump':   pyx.color.rgb( 0.000, 0.000, 0.000 ),  # Color of jumps (maybe not used).
    'cuess':  pyx.color.rgb( 1.000, 0.200, 1.000 ),  # Color of essential cut-lines.
    'cunon':  pyx.color.rgb( 0.400, 0.400, 1.000 ),  # Color of non-essential cut-lines.
    
    'Ytrace': 0.600  # Luminosity of colors in trace/path/block palette.
  }
  return color
  # ----------------------------------------------------------------------
  
def make_figure_canvas(Bfig, autoscale, style,color):
  # Creates the canvas for a figure whose contents has bounding box {Bfig}.
  
  # Extra extra margin:
  mrg = (0.2,0.2)
  Bplot = rn.box_expand(Bfig, mrg, mrg)

  dp = None
  scale = None if autoscale else 0.5
  c, szx, szy = hacks.make_canvas(Bplot, dp, scale, False, False, 1, 1)

  # # Plot an invisible frame to keep the figure sizes uniform:
  # wd_frame = 0.5*style['wd_axes']
  # clr_frame = color['white'] # Invisible.
  # # clr_frame = color['black'] # Just to check.
  # hacks.plot_frame(c, clr_frame, wd_frame, dp, Bfig, wd_frame/2)
  
  return c
  # ----------------------------------------------------------------------
  
def get_turkey_figure_bbox(OCRS, OPHS, OLKS, style):
  # Given the lists of contours {OCRS}, raster paths {OPHS}, and link
  # paths {OLKS} of the "turkey" slice, returns a bounding box suitable
  # for {make_canvas}.
  #
  # The box includes all the move endpoints in those paths, plus some extra space on the sides for 
  # the cut-lines that may be inserted in it, and some extra space to account for matter and sausage width.
  #
  # Also returns the abscissas {cuxlo} and {cuxhi} of the endpoints of cut-lines (excluding the labels).

  # Get the bounding box {B} of all contents:
  B = path.bbox(OPHS)
  Bsize = rn.sub(B[1], B[0])
  sys.stderr.write("bounding box (excl. contours) = %6.2f x %6.2f mm\n" % (Bsize[0], Bsize[1]))
  if len(OCRS) != 0: B = rn.box_join(B, path.bbox(OCRS))
  if len(OLKS) != 0: B = rn.box_join(B, path.bbox(OLKS))
  # Assume that the contacts are automatically included in the box.

  # Compute X-range of cut-lines:
  cuxlo = B[0][0] - style['cuext']
  cuxhi = B[1][0] + style['cuext']

  # Expand box for cut-lines and their labels:
  dxcut = style['cuext'] + style['culab']
  Bfig = B
  Bfig = rn.box_expand(Bfig, (dxcut,0.3), (dxcut,0.3))

  # Extra margin for sausage widths:
  wdf = style['wd_fill']
  mrg = (wdf, wdf)
  Bfig = rn.box_expand(Bfig, mrg, mrg)

  return Bfig, cuxlo, cuxhi
  # ----------------------------------------------------------------------

def ghostify_colors(CLRS, cgh):
  # Mixes a lot of {cgh} into each color of {CLRS}.
  CLRS_new = []
  f = 0.3
  for clr in CLRS:
    R = f*clr.r + (1-f)*cgh.r
    G = f*clr.g + (1-f)*cgh.g
    B = f*clr.b + (1-f)*cgh.b
    clr_new = pyx.color.rgb(R,G,B)
    CLRS_new.append(clr_new)
  return CLRS_new
  # ----------------------------------------------------------------------
  
def widen_box_for_math_figure(B, style):
  wd_mfig = style['wd_mfig']
  xmrg = (wd_mfig - (B[1][0] - B[0][0]))/2
  assert xmrg >= 0, "math-containing box is wider than standard"
  B = rn.box_expand(B, (xmrg,0), (xmrg,0))
  return B
  # ----------------------------------------------------------------------
 
