﻿#include <QString>
#include <er_ECLES_exact_ls.h>
#include <affirm.h>

#include <iostream>
using namespace std;

#define DEBUG_EXACT_LS TRUE


/** ****************************************************************************************************************/
int factoring(int m, int n, double A[], fmpz_mat_t U, fmpz_mat_t L, fmpz_mat_t D, int Pr[], int Pc[])
{
    fmpz_mat_t Ai;
    fmpz_mat_init(Ai, m, n);

    for (int i = 0; i < m; i ++){
        for (int j = 0; j < n; j++) {
            fmpz_set_d(fmpz_mat_entry(Ai, i, j), A[j+i*n]);
        }
    }

    return  factoring(m, n, Ai, U, L, D, Pr, Pc);
}


/** ****************************************************************************************************************/
int factoring(int m, int n, fmpz_mat_t A, fmpz_mat_t U, fmpz_mat_t L, fmpz_mat_t D, int Pr[], int Pc[])
{
    int r;
    int max = n;
    if(m > n){
        max = m;
    }

    fmpz_mat_t V, U_aux, L_aux, D_aux;
    mp_limb_signed_t perm1[max], perm2[max], perm3[max];
    fmpz_t old_pivot, m1, m2;
    int i, j, s, t, k;

    // inicializa fmpz_t
    fmpz_init(old_pivot);
    fmpz_init(m1);
    fmpz_init(m2);

    if(DEBUG_EXACT_LS){
        cout << "\n\n:::::: TRIANGULARIZE ====================================== \n" << endl;
        cout << "A (" << m << "x" << n << ") = \n" << endl;
        fmpz_mat_print_pretty(A);
    }

    // inicializa variáveis
    initializeMatrices(max, A, U_aux, L_aux, D_aux, V);
    for(i = 0; i < max; i++){ perm1[i] = perm2[i] = perm3[i] = i; }
    i = j = r = 0;

    // percorre toda matriz enquanto houver linhas e colunas pendentes
    while ( i < m && j < n){

        // escolhe pivô Ukj na coluna j (primeiro elemento diferente de zero, se existir. Senão pula para a próxima coluna)
        k = fmpz_mat_find_pivot_any(U_aux, i, m, j);

        if(k == -1){
            j++;

        }else{

            r++;

            // troca linhas i e k, se necessário
            if(i != k){
                if(DEBUG_EXACT_LS){
                    cout << "\nTroca linha " << "i: "  << i << " com k: "<< k << endl;
                }
                fmpz_mat_swap_rows(U_aux, perm1, i, k);
                fmpz_mat_swap_rows(L_aux, perm3, i, k);
            }

            // armazena pivô da linha i de U na matriz diagonal da fatoração LU
            fmpz_set(fmpz_mat_entry(L_aux, i, j), fmpz_mat_entry(U_aux, i, j));
            fmpz_set(fmpz_mat_entry(D_aux, i, j), fmpz_mat_entry(U_aux, i, j));
            fmpz_set(fmpz_mat_entry(V, i, 0), fmpz_mat_entry(D_aux, i, j));

            for(t = i+1; t < m; t++){

                // armazena elemento tj de U na matriz inferior L_aux, a qual possui diagonal nula
                fmpz_set(fmpz_mat_entry(L_aux, t, j), fmpz_mat_entry(U_aux, t, j));

                // executa Eliminação de Gauss Livre de Fração nas linhas abaixo da linha i na matriz U
                for(s = j+1; s < n; s++){

                    fmpz_mul(m1, fmpz_mat_entry(U_aux, i, j), fmpz_mat_entry(U_aux, t, s));
                    fmpz_mul(m2, fmpz_mat_entry(U_aux, t, j), fmpz_mat_entry(U_aux, i, s));

                    fmpz_sub(fmpz_mat_entry (U_aux, t, s), m1, m2);

                }

                // zera elementos da coluna j que estão abaixo da linha i
                fmpz_set_d(fmpz_mat_entry(U_aux, t, j), 0);

            }

            // elimina fatores comuns (pivô do passo anterior i-1) da matriz U
            if(i >= 1){
                for(t = i+1; t < m; t++){
                    for(s = j+1; s < n; s++){

                        // executa divisão exata para eliminar o fator comum (pivô anterior) da operação ffge
                        if(fmpz_mat_entry(V, i-1, 0) != 0){
                            fmpz_divexact(fmpz_mat_entry(U_aux, t, s), fmpz_mat_entry(U_aux, t, s), fmpz_mat_entry(V, i-1, 0));

                        }else{
                            cerr << "Erro no algoritmo, divisão por zero!" << endl;
                            return -1;
                        }
                    }
                }
            }

//            if(DEBUG_EXACT_LS){
//                flint_printf("==============");
//                flint_printf("\n\nL_aux = \n"); fmpz_mat_print_pretty(L_aux);
//                flint_printf("\n\nD_aux = \n"); fmpz_mat_print_pretty(D_aux);
//                flint_printf("\n\nU_aux = \n"); fmpz_mat_print_pretty(U_aux);
//                flint_printf("==============");
//            }

            i++;
            j++;
        }        
    }

    // configura vetor de permutação de linhas
    for(int w=0; w<m; w++){
        Pr[w] = perm1[w];
    }    
    //calculateInverseVector(m, Pr_aux, Pr);

    // configura a matriz L_aux = L_aux + D_aux (matriz inferior auxiliar)
    //fmpz_mat_add(L_aux, L_aux, D_aux);

    // faz permutação de colunas, se necessário
    for(i=0; i<max; i++){

        if(fmpz_is_zero(fmpz_mat_entry(D_aux, i, i))){

            k = j = i;

            while(k < n && fmpz_is_zero(fmpz_mat_entry(D_aux, i, k))){
                k++;
            }

            if(k < n && !fmpz_is_zero(fmpz_mat_entry(D_aux, i, k))){

                if(DEBUG_EXACT_LS){
                    cout << "\n\nTroca coluna " << "j: "  << j << " com k: "<< k << endl;
                }

                // (**** ver porque permuta colunas de L e D)

                fmpz_mat_swap_columns(L_aux, perm1, j, k);
                fmpz_mat_swap_columns(D_aux, perm1, j, k);
                fmpz_mat_swap_columns(U_aux, perm2, j, k);                
            }
        }        
    }

    // configura vetor de permutação de colunas
    for(int w=0; w<n; w++){
        Pc[w] = perm2[w];
    }

    // alterando matriz diagonal para fatoração LU (***** Ver porque uso 'm-1' ao inves de 'max')
    for(i=m-1; i>0; i--){
        fmpz_mul(fmpz_mat_entry(D_aux, i, i), fmpz_mat_entry(D_aux, i, i), fmpz_mat_entry(D_aux, i-1, i-1));
    }

    if(DEBUG_EXACT_LS){
        cout << "\n\nMatrizes Auxiliares .................";

        flint_printf("\n\nL_aux = \n"); fmpz_mat_print_pretty(L_aux);
        flint_printf("\n\nD_aux = \n"); fmpz_mat_print_pretty(D_aux);
        flint_printf("\n\nU_aux = \n"); fmpz_mat_print_pretty(U_aux) ;
    }

    fmpz_mat_clear(V);
    fmpz_clear(old_pivot);
    fmpz_clear(m1);
    fmpz_clear(m2);

//    // calcula rank da matriz A triangularizada (U_aux)
//    int zero;
//    int r = m;
//    for(i=0; i<m; i++){
//        zero = 0;
//        for(j=0; j<n; j++){
//            if(fmpz_is_zero(fmpz_mat_entry(U_aux, i, j))){
//                zero++;
//            }
//        }

//        if(zero == n){ //linha nula
//            r = i;
//            break;
//        }
//    }

    if(DEBUG_EXACT_LS){
        cout << "\n\nRank: " << r << endl;
    }

    // inicializa matrizes Pr, L, D e U (sem redundância)
    fmpz_mat_init(L, m, r);
    fmpz_mat_init(U, r, n);
    fmpz_mat_init(D, r, r);

    // preenche matrizes L, D e U
    for(int i=0; i<m; i++){
        if(i < r){ fmpz_set(fmpz_mat_entry(D, i, i), fmpz_mat_entry(D_aux, i, i));}

        for(int j=0; j<n; j++){
            if(i < r){ fmpz_set(fmpz_mat_entry(U, i, j), fmpz_mat_entry(U_aux, i, j));}
            if(j < r){ fmpz_set(fmpz_mat_entry(L, i, j), fmpz_mat_entry(L_aux, i, j));}
        }
    }

    // limpa matrizes auxiliares
    fmpz_mat_clear(L_aux);
    fmpz_mat_clear(D_aux);
    fmpz_mat_clear(U_aux);

    // imprime matrizes finais
    if(DEBUG_EXACT_LS){
        cout << "\n\nMatrizes Finais .................";
        flint_printf("\n\nPr = \n"); printPermutationVector(m, Pr);
        flint_printf("\n\nL = \n"); fmpz_mat_print_pretty(L);
        flint_printf("\n\nD = \n"); fmpz_mat_print_pretty(D);
        //------------------

        fmpq_mat_t Dinv, Daux;
        fmpq_mat_init(Daux, r, r);
        fmpq_mat_set_fmpz_mat(Daux, D);

        fmpq_mat_init(Dinv, r, r);
        int inv = fmpq_mat_inv(Dinv, Daux);
        if(inv){ flint_printf("\n\nD_inv = \n"); fmpq_mat_print(Dinv);}

        //-----------
        flint_printf("\n\nU = \n"); fmpz_mat_print_pretty(U);
        flint_printf("\n\nPc = \n"); printPermutationVector(n, Pc);
    }

//    fmpz_mat_clear(A);

    return r;
}

//define vetor B = Pr_{-1} * (Q - REA * PhiA -REF *PhiF)
/** ****************************************************************************************************************/
void defineB(int r, int m, int p, int Pr[], fmpz_mat_t QE, fmpz_mat_t REA, fmpz_mat_t PhiA, fmpz_mat_t REF, fmpz_mat_t PhiF, fmpz_mat_t b, fmpz_mat_t B, fmpz_mat_t B_pr)
{
    fmpz_mat_t aux1, aux2, aux3, Bpr_aux;
    int Pr_inv[m];

    fmpz_mat_init(aux1, m, p);
    fmpz_mat_mul(aux1, REA, PhiA);

    fmpz_mat_init(aux2, m, p);
    fmpz_mat_mul(aux2, REF, PhiF);

    fmpz_mat_init(aux3, m, p);
    fmpz_mat_sub(aux3, QE, aux1);

    fmpz_mat_init(b, m, p);
    fmpz_mat_sub(b, aux3, aux2);

    if(DEBUG_EXACT_LS){
        cout << "\n\n:::::: DEFINE_B ======================================" << endl;
        flint_printf("\n\nREA: \n"); fmpz_mat_print_pretty(REA);
        flint_printf("\n\nPhiA: \n"); fmpz_mat_print_pretty(PhiA);
        flint_printf("\n\naux1 = REA * PhiA: \n"); fmpz_mat_print_pretty(aux1);

        flint_printf("\n\nREF: \n"); fmpz_mat_print_pretty(REF);
        flint_printf("\n\nPhiF: \n"); fmpz_mat_print_pretty(PhiF);
        flint_printf("\n\naux2 = REF * PhiF: \n"); fmpz_mat_print_pretty(aux2);

        flint_printf("\n\naux3 = QE - aux1: \n"); fmpz_mat_print_pretty(aux3);

        flint_printf("\n\nb = aux3 - aux2: \n"); fmpz_mat_print_pretty(b);
    }

    computeInverseVector(m, Pr, Pr_inv);
    permuteRows(m, p, Pr, b, Bpr_aux);

    fmpz_mat_init(B, m, p);
    fmpz_mat_set(B, Bpr_aux);

    // B_pr são as primeiras r linhas da matrix Bpr_aux
    fmpz_mat_init(B_pr, r, p);
    for(int i=0; i<r; i++){
        for(int j=0; j<p; j++){
            fmpz_set(fmpz_mat_entry(B_pr, i, j), fmpz_mat_entry(Bpr_aux, i, j));
        }
    }

    if(DEBUG_EXACT_LS){
        cout << "\n\nPr: " << endl;
        printPermutationVector(m, Pr);

        cout << "\n\nPr_inv: " << endl;
        printPermutationVector(m, Pr_inv);

        flint_printf("\n\nb: \n"); fmpz_mat_print_pretty(b);

        flint_printf("\n\nB: \n"); fmpz_mat_print_pretty(B);

        flint_printf("\n\nB_pr: \n"); fmpz_mat_print_pretty(B_pr);
    }
}

/** ****************************************************************************************************************/
bool verifySolvability(int r, int m, fmpz_mat_t L, int p, fmpz_mat_t B)
{
    int mr = m-r;
    fmpq_mat_t L1_inv, L1_aux, L2_aux, B1_aux, B2_aux, aux, X;
    fmpz_mat_t B1, L2, B2, L1;

    if(m == r) return true;

    cout << "=============================================" << endl;
    cout << "Verificando solvabilidade.... " << endl;
    cout << "=============================================" << endl;

    // divide matrizes L e B
    splitRows(m, r, r, L, L1, L2);
    splitRows(m, r, p, B, B1, B2);

    if(DEBUG_EXACT_LS){
        flint_printf("\n\nL1 = \n"); fmpz_mat_print_pretty(L1);
        flint_printf("\n\nL2 = \n"); fmpz_mat_print_pretty(L2);
        flint_printf("\n\nB1 = \n"); fmpz_mat_print_pretty(B1);
        flint_printf("\n\nB2 = \n"); fmpz_mat_print_pretty(B2);
    }

    // converte matrizes inteira para racionais
    fmpq_mat_init(L1_aux, r, r);
    fmpq_mat_set_fmpz_mat(L1_aux, L1);

    // se a matriz L1 tiver inversa
    fmpq_mat_init(L1_inv, r, r);
    if(fmpq_mat_inv(L1_inv, L1_aux)){

        fmpq_mat_init(L2_aux, mr, r);
        fmpq_mat_set_fmpz_mat(L2_aux, L2);

        fmpq_mat_init(aux, mr, r);
        fmpq_mat_mul(aux, L2_aux, L1_inv);

        if(DEBUG_EXACT_LS){
            flint_printf("\n\nL1_inv = \n"); fmpq_mat_print(L1_inv);
            flint_printf("\n\nL2 * L1_inv = \n"); fmpq_mat_print(aux);
        }

        fmpq_mat_init(B1_aux, r, p);
        fmpq_mat_set_fmpz_mat(B1_aux, B1);

        fmpq_mat_init(X, mr, p);
        fmpq_mat_mul(X, aux, B1_aux);

        fmpq_mat_init(B2_aux, mr, p);
        fmpq_mat_set_fmpz_mat(B2_aux, B2);

        if(DEBUG_EXACT_LS){
            flint_printf("\n\nX = \n"); fmpq_mat_print(X);
            flint_printf("\n\nB2 = \n"); fmpq_mat_print(B2_aux);
        }

        // o sistema não tem solução
        if(fmpq_mat_equal(X, B2_aux) == 0) return false;

        return true;

    }else{

        cout << "\n\n >>> Verificando solvabilidade :: L1 não tem inversa!" << endl;
        return false;

    }    
}


int solveSystem(int n, fmpz_mat_t A, int p, fmpz_mat_t B, double Xd[])
{
    fmpz_mat_t X;
    fmpz_t den;

    // inicializa fmpz_t
    fmpz_init(den);

    // resolve o sistema AX = B.den
    int singular = solveSystem(n, A, p, B, X, den);

    if(singular != 0){

        // converte o vetor X para double e resolve Xd = X / den;
        convertFmpzToDoubleMatrix(n, p, X, Xd, den);
    }

    return singular;
}


/** ****************************************************************************************************************/
int solveSystem(int n, fmpz_mat_t A, int p, fmpz_mat_t B, fmpz_mat_t X, fmpz_t den)
{
    // encontra solução inteira a menos de um fator comum,
    fmpz_mat_init(X, n, p);
    int singular = fmpz_mat_solve(X, den, A, B);

    if(DEBUG_EXACT_LS){
        cout << "\n\n:::::: SOLVE_SYSTEM ======================================" << endl;
        cout << "Singular: " << singular << endl;
        flint_printf("\n\nden = \n"); fmpz_print(den);
        flint_printf("\n\nA = \n"); fmpz_mat_print_pretty(A);
        flint_printf("\n\nB = \n"); fmpz_mat_print_pretty(B);
        flint_printf("\n\nX = \n"); fmpz_mat_print_pretty(X);
    }

    return singular;
}


/** ****************************************************************************************************************/
int solveY(int m, int r, fmpz_mat_t D, fmpz_mat_t L, int p, fmpz_mat_t B, fmpz_mat_t Y)
{
    fmpq_mat_t L1inv, L1aux, aux1, Daux;
    fmpz_mat_t L1, B1, aux;

    cout << "_____________________________________________" << endl;
    flint_printf("\nB = \n"); fmpz_mat_print_pretty(B);
    flint_printf("\n\nL = \n"); fmpz_mat_print_pretty(L);
    cout << "_____________________________________________" << endl;

    // divide matrizes L e B
    splitRows(m, r, r, L, L1);
    splitRows(m, r, p, B, B1);

    // calcula matriz inversa da matriz L1    
    fmpq_mat_init(L1aux, r, r);
    fmpq_mat_set_fmpz_mat(L1aux, L1);

    fmpq_mat_init(L1inv, r, r);
    if(!fmpq_mat_inv(L1inv, L1aux)){
        cout << "\n\n >>> Resolvendo Y :: L1 não tem inversa!" << endl;
        return 0;
    }

    fmpq_mat_init(Daux, r, r);
    fmpq_mat_set_fmpz_mat(Daux, D);

    fmpq_mat_init(aux1, r, r);
    fmpq_mat_mul(aux1, Daux, L1inv);

    fmpz_mat_init(aux, r, r);
    fmpq_mat_get_fmpz_mat(aux, aux1);

    fmpz_mat_init(Y, r, p);
    fmpz_mat_mul(Y, aux, B1);

    if(DEBUG_EXACT_LS){
        cout << "\n\n:::::: SOLVE_Y ======================================" << endl;
        flint_printf("\nB1 = \n"); fmpz_mat_print_pretty(B1);
        flint_printf("\n\nLinv = \n"); fmpq_mat_print(L1inv);
        flint_printf("\n\nY = \n"); fmpz_mat_print_pretty(Y);
    }

    return 1;
}

/** ****************************************************************************************************************/
int solveX(int r, int n, fmpz_mat_t U, int p, fmpz_mat_t Y, double Xd[])
{
    fmpq_mat_t X;

    // resolve o sistema AX=B (devolve solução inteira e o fator comum da solução)
    solveX(r, n, U, p, Y, X);

    // converte o vetor X para double retirando o fator comum "d" (devolve solução real)
    Xd = rmxn_alloc(n, p);
    convertFmpqToDoubleMatrix(n, p, X, Xd);

    return 1;
}


/** ****************************************************************************************************************/
int solveX(int r, int n, fmpz_mat_t U, int p, fmpz_mat_t Y, fmpq_mat_t X)
{
    fmpz_mat_t U1, U2;
    fmpq_mat_t U1inv, U1aux, Yaux;

    splitColumns(r, n, U, U1, U2);

    // calcula inversa de U
    fmpq_mat_init(U1aux, r, r);
    fmpq_mat_set_fmpz_mat(U1aux, U1);

    fmpq_mat_init(U1inv, r, r);
    if(!fmpq_mat_inv(U1inv, U1aux)){
        cout << "\n\n >>> Resolvendo X :: U1 não tem inversa!" << endl;
        return 0;
    }

    // calcula a solução final
    fmpq_mat_init(Yaux, r, p);
    fmpq_mat_set_fmpz_mat(Yaux, Y);

    fmpq_mat_init(X, r, p);
    fmpq_mat_mul(X, U1inv, Yaux);

    if(DEBUG_EXACT_LS){
        cout << "\n\n:::::: SOLVE_X ======================================" << endl;
        flint_printf("\nU_inv = \n"); fmpq_mat_print(U1inv);
        flint_printf("\n\nY = \n"); fmpz_mat_print_pretty(Y);
        flint_printf("\n\nX = \n"); fmpq_mat_print(X);
    }

    return 1;
}


/** *********************************;*******************************************************************************/
int splitRows(int m, int r, int n, fmpz_mat_t A, fmpz_mat_t A1)
{
    return splitRows(m, r, n, A, A1, NULL);
}



/** ****************************************************************************************************************/
int splitRows(int m, int r, int n, fmpz_mat_t A, fmpz_mat_t A1, fmpz_mat_t A2)
{    
    // inicializa matrizes A1 e A2
    fmpz_mat_init(A1, r, n);

    // retorna 0 se não for necessário dividir a matriz A
    if(m == r){

        // configura A1 = A;
        fmpz_mat_set(A1, A);

        if(DEBUG_EXACT_LS){
            cout << "\n\n:::::: SPLIT_ROWS ======================================" << endl;
            cout << "M: " << m << " R: " << r << endl;
            flint_printf("\nA = \n"); fmpz_mat_print_pretty(A);
            flint_printf("\n\nA1 = \n"); fmpz_mat_print_pretty(A1);
        }

        return 0;
    }

    if(A2){
        fmpz_mat_init(A2, m-r, n);
    }

    flint_printf("\nA = \n"); fmpz_mat_print_pretty(A);

    // divide a matriz A em A1 (primeiras r linhas de A) e A2 (últimas m-r linhas de A)
    for(int j=0; j<n; j++){
        for(int i=0; i<r; i++){
            fmpz_set(fmpz_mat_entry(A1, i, j), fmpz_mat_entry(A, i, j));
        }

        if(A2){
            for(int i=r; i<m; i++){
                fmpz_set(fmpz_mat_entry(A2, i-r, j), fmpz_mat_entry(A, i, j));
            }
        }
    }

    if(DEBUG_EXACT_LS){
//        cout << "\n\n:::::: SPLIT_ROWS ======================================" << endl;
//        flint_printf("\nA = \n"); fmpz_mat_print_pretty(A);
//        flint_printf("\n\nA1 = \n"); fmpz_mat_print_pretty(A1);
//        if(A2){ flint_printf("\n\nA2 = \n"); fmpz_mat_print_pretty(A2);}
    }

    return 1;
}


/** ****************************************************************************************************************/
int splitColumns(int r, int n, fmpz_mat_t A, fmpz_mat_t A1)
{
    return splitColumns(r, n, A, A1, NULL);
}

/** ****************************************************************************************************************/
int splitColumns(int r, int n, fmpz_mat_t A, fmpz_mat_t A1, fmpz_mat_t A2)
{
    // retorna 0 se não for necessário dividir a matriz A
    if(n == r){

        // inicializa matrizes A1 e configura A1 = A;
        fmpz_mat_init(A1, r, r);
        fmpz_mat_set(A1, A);

        if(DEBUG_EXACT_LS){
            cout << "\n\n:::::: SPLIT_COLUMNS ======================================" << endl;
            flint_printf("\nA = \n"); fmpz_mat_print_pretty(A);
            flint_printf("\n\nA1 = \n"); fmpz_mat_print_pretty(A1);
        }

        return 0;
    }

    // inicializa matrizes A1 e A2
    fmpz_mat_init(A1, r, r);
    fmpz_mat_init(A2, r, n-r);

    // divide a matriz A em A1 (primeiras r colunas de A) e A2 (últimas n-r colunas de A)
    for(int i=0; i<r; i++){
        for(int j=0; j<r; j++){ fmpz_set(fmpz_mat_entry(A1, i, j), fmpz_mat_entry(A, i, j)); }
        for(int j=r; j<n; j++){ fmpz_set(fmpz_mat_entry(A2, i, j-r), fmpz_mat_entry(A, i, j)); }
    }

    if(DEBUG_EXACT_LS){
        cout << "\n\n:::::: SPLIT_COLUMNS ======================================" << endl;
        flint_printf("\nA = \n"); fmpz_mat_print_pretty(A);
        flint_printf("\n\nA1 = \n"); fmpz_mat_print_pretty(A1);
        flint_printf("\n\nA2 = \n"); fmpz_mat_print_pretty(A2);
    }

    return 1;

}

/** ****************************************************************************************************************/
void convertFmpzToDoubleMatrix(int m, int n, fmpz_mat_t M, double *Md, fmpz_t d)
{
    double dd;

    dd = fmpz_get_d(d);

    // converte matriz inteira M para double
    if(dd != 0){
        for(int i=0; i<m; i++){
            for(int j=0; j<n; j++){

                Md[j+i*n] = fmpz_get_d(fmpq_mat_entry(M, i, j));
                Md[j+i*n] = Md[j+i*n] / dd;

            }
        }
    }
}

/** ****************************************************************************************************************/
void convertFmpqToDoubleMatrix(int m, int n, fmpq_mat_t M, double *Md)
{
    double num1, den1;
    fmpz_mat_t num, den;

    fmpz_mat_init(num, m, n);
    fmpz_mat_init(den, m, n);

    fmpq_mat_get_fmpz_mat_entrywise(num, den, M);

    // converte matriz inteira M para double
    for(int i=0; i<m; i++){
        for(int j=0; j<n; j++){

            num1 = fmpz_get_d(fmpz_mat_entry(num, i, j));
            den1 = fmpz_get_d(fmpz_mat_entry(den, i, j));

            Md[j+i*n] = num1 / den1;

        }
    }

    if(DEBUG_EXACT_LS){
        cerr << "\n\n=============="<< endl;
        gsel_print_array(stderr, QString("%5.4f").toLatin1().data(), QString("Md:").toLatin1().data(),  m, n, Md, QString("").toLatin1().data());
        cerr << "==============\n"<< endl;
    }

}



///** ****************************************************************************************************************/
//void convertMatrixToDouble(int m, int n, fmpz_mat_t M, double *Md, double d)
//{
//    // converte matriz inteira M para double
//    for(int i=0; i<m; i++){
//        for(int j=0; j<n; j++){
//            Md[j+i*n] = fmpz_get_d(fmpz_mat_entry(M, i, j));
//            if(d != 0){ Md[j+i*n] = Md[j+i*n] / d; }
//        }
//    }

//    if(DEBUG_EXACT_LS){
//        cerr << "\n\n=============="<< endl;
//        gsel_print_array(stderr, QString("%5.4f").toLatin1().data(), QString("Md:").toLatin1().data(),  m, n, Md, QString("").toLatin1().data());
//        cerr << "==============\n"<< endl;
//    }

//}

/** **********************************************************************************************************************/
void convertMatrixToExact (int m, int n, double *Md, fmpz_mat_t M)
{
    // converte matriz inteira M para double
    for(int i=0; i<m; i++){
        for(int j=0; j<n; j++){
            fmpz_set_d(fmpz_mat_entry(M, i, j), Md[j+i*n]);
        }
    }

    if(DEBUG_EXACT_LS){ flint_printf("\n\nM = \n"); fmpz_mat_print_pretty(M); }

}


/** **********************************************************************************************************************/
void initializeMatrices(int max, fmpz_mat_t A, fmpz_mat_t U_aux, fmpz_mat_t L_aux, fmpz_mat_t D_aux, fmpz_mat_t V)
{
    // inicializa matrix triangular superior auxiliar (U_aux) e inicializa com elementos de AB
    fmpz_mat_init(U_aux, max, max);
    fmpz_mat_set(U_aux, A);

    // inicializa matrizes triangular inferior auxiliar (L_aux)
    fmpz_mat_init(L_aux, max, max);

    // inicializa matrizes diagonal auxiliar (D_aux)
    fmpz_mat_init(D_aux, max, max);

    // inicializa vetor de pivô
    fmpz_mat_init(V, max, 1);
}


/** ****************************************************************************************************************/
void computeInverseVector(int n, int P[], int Q[])
{
    for(int i=0; i<n; i++){
        Q[P[i]] = i;
    }
}



/** ****************************************************************************************************************/
void printPermutationVector(int m, int P[])
{
    cout << "[ "; for(int i=0; i<m; i++){ cout << P[i] << " "; } cout << "] ";
}


/** ****************************************************************************************************************/
void initPermutationVector(int m, int *P)
{
    for(int i=0; i<m; i++){ P[i] = i; }
}


/** ****************************************************************************************************************/
void permuteRows(int m, int n, int Pr[], fmpz_mat_t A, fmpz_mat_t A_pr)
{
    fmpz_mat_init(A_pr, m, n);

    for(int i=0; i<m; i++){
        for(int j=0; j<n; j++){
            //fmpz_set(fmpz_mat_entry(A_pr, Pr[i], j), fmpz_mat_entry(A, i, j));
            fmpz_set(fmpz_mat_entry(A_pr, i, j), fmpz_mat_entry(A, Pr[i], j));
        }
    }
}


/** ****************************************************************************************************************/
void permuteColumns(int m, int n, int Pc[], fmpz_mat_t A, fmpz_mat_t A_pc)
{
    fmpz_mat_init(A_pc, m, n);

    for(int i=0; i<m; i++){
        for(int j=0; j<n; j++){
            //fmpz_set(fmpz_mat_entry(A_pc, i, j),fmpz_mat_entry(A, i, Pc[j]));
            fmpz_set(fmpz_mat_entry(A_pc, i, Pc[j]),fmpz_mat_entry(A, i, j));
        }
    }

}

/** ****************************************************************************************************************/
void gsel_print_array(FILE *wr, char *fmt, char *head, int m, int n, double M[], char *foot)
{
    if (head != NULL) { fprintf(wr, "%s\n", head); }
    int i, j;
    for (i = 0; i < m; i++){
        fprintf(wr, "  ");
        for (j = 0; j < n; j++){
            fprintf(wr, " "); fprintf(wr, fmt, M[i*n + j]);
        }
        fprintf(wr, "\n");
    }
    if (foot != NULL) { fprintf(wr, "%s\n", foot); }
}

/** ****************************************************************************************************************/
double *rmxn_alloc(int m, int n)
{
    void *p = malloc(m*n*sizeof(double));
    affirm(p != NULL, "no memory for rmxn_t");
    return (double *)p;
}



