#! /usr/bin/python3
# Last edited on 2018-07-05 12:27:52 by stolfilocal

HELP = \
  "  muff_mainloop.py {nL} {nV} {nH} {Zstep}\n"

INFO = \
  "  This is the core process in the MUFF 2.0 microscope positioner software suite.  Its task is to loop through the various light settings, view directions, and camera positions. It interacting with the user (through {stderr} and {stdin}), interacting with the Arduino firmware (through a serial port), and sending commands to the camera monitoring and grabbing process (through stdout).\n" \
  "\n" \
  "  The command line arguments are the number {nL} of distinct light settings, the number of {nV} viewing directions, the number {nH} of microscope Z positions (frames per stack), and the distance {Zstep} between consecutive positions (float, in millimeters).  Currently the number of views must be 1.\n" \
  "\n" \
  "  In normal operation, this process should be started with its {stdout} connected to the {stdin} of the process {muff_camview.py}.  It can be started alone for debugging, but then no frames will be grabbed or displayed.\n" \
  "\n" \
  "  The Arduino development environment should be running too, in order to download the firmware to the Arduino and to position the microscope manually for the lowest frame.  (Eventually there will be manual motion buttons on the support itself, so this need will go away.)\n" \
  "\n" \
  "  The images are written with names '{muff_scans}/{datetime}/L{nn}/V{vv}/raw/frame_{fffff}.jpg', where {nn} is the index of the lighting setup (2 digits, from 00), {vv} is the view index (ditto), and {fffff} is the frame index (5 digits, from 0).  The {datetime} is the date, hour, and minute when the program was started."

import os, sys, time, serial, muff_arduino
from datetime import datetime
from sys import stdin, stdout, stderr

# Global parameters:
nLED =  muff_arduino.nLED # Number of LEDs on the lighting dome.
Zstep_min = -0.999   # Minimum step in Z coordinates (mm).
Zstep_max = +0.999   # Maximum step in Z coordinates (mm).
Zrange_max = 100.00  # Max Z position range (mm).
nV_max = 1           # Max viewing directions.
nL_max = 24          # Max number of lighting schemas.
nH_max = 99          # Max number of frames in each stack.
use_uvc = False      # If true uses {uvccapture}, if false uses {muff_camview.py}.

mov_time = 3.00      # Estimated time to raise the microscope.
rot_time = 3.00      # Estimated time to rotate object.
img_time = 3.00      # Estimated time to capture 1 image.

debug = True         # Debugging mode (without the Arduino).
verbose = True       # True to print debugging info.
# ----------------------------------------------------------------------

def main():
  """Main program."""
  
  if len(sys.argv) != 5 or sys.argv[1] == "-help":
    # Display the help text and exit:
    stderr.write("SYNOPSIS\n")
    stderr.write(HELP + "\n\n")
    stderr.write("DESCRIPTION\n")
    stderr.write(INFO + "\n")
    exit(0)
  
  # Get data from command lne:
  nL = int(sys.argv[1])
  nV = int(sys.argv[2]) 
  nH = int(sys.argv[3])
  Zstep = float(sys.argv[4])
  
  # Connect to the Arduino through a serial port:
  sport = muff_arduino.connect(debug,verbose)
  
  # Test the leds:
  muff_arduino.test_lights(sport)
  
  # Define the vertical displacement between frames in each stack:
  muff_arduino.set_Z_step(sport,Zstep)

  # Capture the images:
  ok = capture_image_set(sport,nL,nV,nH, Zstep)

  # Finalization:
  if not use_uvc:
    # Terminate the {muff_camview.py} process:
    stdout.write("Q\n")
    stdout.flush()
    
  stderr.write("[muff_mainloop:] done.\n")
  return 
# ----------------------------------------------------------------------

def capture_image_set(sport, nL,nV,nH, Zstep):
  """Captures a complete MUFF image set, consisting of a 
  multi-focus image stack for each of {nL} lighting schemas and {nV}
  view directions.  Each stack will have {nH} images, at equally
  spaced microscope heights, starting at the current position and
  rising by {Zstep} millimeters at every turn.  Interacts
  with the Arduino through the serial port {sport}.
  
  Returns {True} if finished successfully, {False} if aborted."""
  
  # Parameter checks:
  assert type(nL) is int and nL > 0 and nL <= nL_max     
  assert type(nV) is int and nV > 0 and nV <= nV_max   
  assert type(nH) is int and nH > 0 and nH <= nH_max   
  assert type(Zstep) is float and Zstep >= Zstep_min and Zstep <= Zstep_max
  assert nH*Zstep <= Zrange_max + 0.0001 # Fudged for rounding.
  
  # Compute number of images and estimated time:
  nI = nL*nV*nH;
  stderr.write("[muff_mainloop:] capturing %d images (%d lights, %d views, %d heights)\n" % (nI,nL,nV,nH))
  eTime = img_time * nL # Time for 1 view direction, 1 microscope pos. 
  if nV > 1:
    # Account for object repositioning time:
    eTime = (eTime + rot_time)*nV
  # Account for microscope motion time:
  eTime = eTime * nH + mov_time * (nH-1)
  stderr.write("[muff_mainloop:] estimated time = %.1f minutes\n" % (eTime/60))

  # Position the microscope at first height:
  stderr.write("[muff_mainloop:] manually position the microscope at the lower Z value, type 'ok' when done.\n")
  ok = wait_for_user_ok()
  if not ok:
    # User typed 'CTRL+D'or 'Q':
    stderr.write("** [muff_mainloop:] aborted - no images were captured.\n")
    return False
  
  # Create the directory tree:
  topdir = create_directories(nL,nV);
  stderr.write("[muff_mainloop:] saving images in directory %s\n" % topdir)
  
  # Capture all images: 
  tstart = time.time()
  for H in range(nH):
  
    if H > 0:
      # Move the microscope to the next Z position:
      muff_arduino.move_microscope(sport)
    
    # Capture all frames for this Z position:
    for V in range(nV):
    
      if nV > 1:
        # Rotate the object to viewing direction {V}:
        set_view_direction(sport,V,nV) 
        
      # Capture all frames for this {Z} posiiton and view direction:
      for L in range(nL):
        
        # Choose the set of lights to use, and turn them on:
        Lset = choose_light_set(L, nL) 
        switch_lights_on(sport,Lset)
        
        # Grab the frame and save it to disk:
        capture_frame(topdir,L,V,H)
        
        # Turn the lights off:
        switch_lights_off(sport,Lset)
    
  tstop = time.time();
  stderr.write("[muff_mainloop:] captured %d images in %.1f minutes\n" % (nI, (tstop - tstart)/60))
  return True
# ----------------------------------------------------------------------

def capture_frame(topdir,L,V,H):
  """Issues a call to the external image capture software to grab one
  frame from the microscope, assumed to be for lighting schema {L}, view
  direction {V}, and microscope height {H}. Assumes that the lights,
  view, and microscope have been physically set as appropriate. Writes
  the image file in the proper subdirectory of {topdir}."""
  
  # Compose the file name:
  fname = make_frame_filename(topdir, L, V, H)
  if verbose: stderr.write("[muff_mainloop:] capturing frame %d and writing to '%s'\n" % (H,fname))
  
  # Call the external image capture program:
  if use_uvc:
    if debug: 
      res = 0 
    else: # Capture with {uvccapture}:
      template = "uvccapture -S40 -C30 -G80 -B20 -x2560 -y2048 -o'%s' -v"
      if verbose: stderr.write("[muff_mainloop:] running graber - command = [%s]\n" % template)
      command = (template % fname)
      res = os.system(command)
    if res != 0:
      stderr.write("** [muff_mainloop:] frame capture command returned with nonzero status\n")
      sys.exit(res)
  else: 
    # Use {muff_camview.py}:
    command = "G " + fname + "\n"
    stdout.write(command)
    stdout.flush()

  return
# ----------------------------------------------------------------------
  
def create_directories(nL,nV):
  """Creates the directory structure for a scanset with {nL} lighting schemas, 
  {nV} viewing directions.  Returns the top level directory name,
  'muff_scans/{date}-{minute}'.  Fails if that directory exists. 
  If that happens, wait a minute and retry."""
  
  # Parameter checks (the "<= 99" is because of dir names):
  assert type(nL) is int and nL > 0 and nL <= 99 and nL <= nLED     
  assert type(nV) is int and nV > 0 and nV <= 99 and nV <= nV_max   
  
  # Create the top level directory.  Fails if already exists.
  td = datetime.utcnow()             # UTC date and tofday. 
  tdx = td.strftime("%Y-%m-%d-%H%M-U") # Formatted UTC date, hour, minute.
  topdir = ("muff_scans/%s" % tdx)      # Top level directory name.
  if verbose: stderr.write("[muff_mainloop:] creating top directory '%s' and subdirectories\n" % topdir)
  if not debug: os.makedirs(topdir,exist_ok=FALSE)    # Create top level dir (must not exist).
  
  # Create subdirs for lights and views.
  for L in range(nL):
    for V in range(nV):
      subdir = make_subdir_name(topdir, L, V) # Subdirectory name.
      if verbose: stderr.write("[creating subdirectory '%s']\n" % subdir)
      if not debug: os.makedirs(subdir,exist_ok=FALSE)   # Create subdir (must not exist).
  
  return topdir
# ----------------------------------------------------------------------
  
def make_subdir_name(topdir, L, V):
  """Creates the subdirectory for all raw images with lighting schema {L} and view 
  direction {V}, in the top level directory {topdir}."""
  
  subdir = ("%s/L_%02d/V_%02d/raw" % (topdir, L, V))
  return subdir
# ----------------------------------------------------------------------

def make_frame_filename(topdir, L, V, H):
  """Creates the filename for the raw image with lighting schema {L}, view 
  direction {V}, and microscope height {H}, in the top level directory
  {topdir}."""
  
  subdir = make_subdir_name(topdir, L, V)
  fname = ("%s/frame_%05d.jpg" % (subdir,H))
  return fname
# ----------------------------------------------------------------------

def choose_light_set(L,nL):
  """Chooses the LED subset number {L} from {nL} possible
  light sets.  Returns a list of LED indices in {0..nLED-1}.  The relative
  intensity is a float between 0.00 and 1.00.""" 
  
  assert type(nL) is int and nL > 0 and nL <= nLED
  assert type(L) is int and L >= 0 and L < nL
  if nL == 24:
    # Light sources are individual LEDs:
    Lset = [L]
  else:
    stderr.write("** [muff_mainloop:] Invalid lighting scheme ({nL} = %d)\n" % nL)
    sys.exit(1)
  return Lset
# ----------------------------------------------------------------------

def switch_lights_on(sport,Lset):
  """Sends to the arduino commands to turn the lights specified in {Lset} on. 
  The parameter {Lset} must be a list of LED indices in icreasing order. Each led {led} is
  turned on, and any leds not in the list are turned off."""
  
  nS = len(Lset); # Number of entries in {Lset}
  kS = 0;         # Entry in {Lset}.
  led_prev = -1   # Previous LED index from {Lset}.
  for led in range(nLED):
    # Make sure we have the next index from {Lset} in {led_prev}:
    if led_prev < led:
      # Get next pair from {Lset}:
      if kS < nS:
        led_prev = Lset[kS]
        assert type(led_prev) is int and led_prev >= 0 and led_prev < nLED
      else:
        led_prev = nLED
    assert led_prev >= led

    # Decide the intensity of LED {led}:
    if led == led_prev:
      pwr = 1.0
    else:
      pwr = 0.0
      
    # Turn the LED on or off:
    muff_arduino.switch_LED(sport,led,pwr)
  
  # We must be done:
  assert kS == nS
  return
# ----------------------------------------------------------------------
  
def switch_lights_off(sport,Lset):
  """Sends commands to the Arduino to turn the lights specified in {Lset} off. 
  The parameter {Lset} must be a list of LED indices in increasing order. Each 
  led in the list is turned off; the others are assumed to be off
  already."""
  
  for led in Lset:
    assert type(led) is int and led >= 0 and led < nLED
    muff_arduino.switch_LED(sport,led,0.0)
  return
# ----------------------------------------------------------------------

def set_view_direction(sport,V,nV):
  """Sends commands to the Arduino to rotate the object to the 
  view direction number {V} out of {nV} possible views. 
  Will make sense only when there is a mechanized stage.""" 
  
  assert type(nV) is int and nV == 1
  assert type(V) is int and V >= 0 and V < nV
  if nV > 1:
    if verbose: stderr.write("[muff_mainloop:] rotating object to view direction %d\n" % V)
  return
# ----------------------------------------------------------------------

def wait_for_user_ok():
  """Waits for user to type 'ok[ENTER]' or 'abort[ENTER]' on the python 
  shell window.  When she types 'ok', returns {True}. If she types
  'abort' or CTRL-D (end-of-file), returns {False}.  If the user types
  anything else, keeps asking again."""
  
  while True:
    stderr.write("[muff_mainloop:] type 'ok[ENTER]' or 'abort[ENTER]': \n");
    stderr.flush()
    s = stdin.readline()
    if s == "":
      # End of file:
      return False
    s = s.strip() # Remove leading and trailing whitespace, including EOL.
    if s == "ok":
      return True
    elif s == "abort":
      return False
# ----------------------------------------------------------------------

def show_chars(s, blanks):
  """Given a {bytes} object {s}, returns a {string} object
  with each non-printing char in {s} replaced by '[chr({NNN})]',
  where {NNN} is the character's decimal {ord}.  Also replaces 
  quotes, brackets, parentheses. If {blanks} is true, 
  replaces blanks too."""
  
  n = len(s)
  res = ""
  bad = b"\'\"[]()" # Printable characters that should be converted too.
  for i in range(n):
    c = s[i]
    if (c == b' ' and blanks) or (c < b' ') or (c > b'~') or (bad.find(c) >= 0):
      # Show chr code:
      res = res + ("[chr(%03d)]" % ord(c))
    else:
      res = res + chr(ord(c))
  return res
# ----------------------------------------------------------------------

main()



        
