#! /usr/bin/python3
# Test program for module {hotpath}.
# Last edited on 2021-02-19 15:51:18 by jstolfi

import hotpath
import path
import block
import example_path
import input_data
import hacks
import job_parms
import rn
import pyx
import sys
from math import sqrt, sin, cos, floor, ceil, inf, nan, pi

parms = job_parms.very_slow()
oprefix = "tests/out/hotpath_TST"

job_parms.write(sys.stderr, None, parms, None)

def plot_solution(Q, name):
  # Plots the oriened path {Q} to files called "hotpath_TST_{name}.{ext}",
  # where {ext} is "eps", "jpg", "png", etc.
  c = pyx.canvas.canvas()
  
  # Find the bounding box:
  Pbox = path.bbox(Q)

  plo = (floor(Pbox[0][0])-1, floor(Pbox[0][1])-1)
  phi = (ceil(Pbox[1][0])+1, ceil(Pbox[1][1])+1)
  xsz = phi[0] - plo[0]
  ysz = phi[1] - plo[1]

  hacks.plot_frame(c, pyx.color.rgb.white, 0.02, plo, xsz, ysz, 0.01)
  hacks.plot_grid(c, pyx.color.rgb(0.800, 0.750, 0.600), 0.03, plo, xsz, ysz, 0.25, 1,1)
  pyx.unit.set(uscale=1.00, wscale=1.00, vscale=1.00)
  
  ctrace =  pyx.color.rgb(0.050, 0.850, 0.000) # Color of nominal trace area.
  waxes = 0.05*parms['solid_raster_width']; 
  axes = False
  dots = False
  arrows = False
  matter = True
  path.plot_standard(c, Q, (0,0), None, ctrace, waxes, axes, dots, arrows, matter)
  hacks.write_plot(c, "tests/out/hotpath_TST_sol_" + name)
  return 
  # ----------------------------------------------------------------------

def test_tiny(tag):
  # Three blocks, each with one path and its reversal.
  # The paths are two horizontal raster lines and 
  # a bent 2-trace path, roughly horizontal too.  
  #
  # There are no cooling or max-calls constraints. The starting point is
  # fixed at {(1,1)}. The file name will have "_test1" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = tag
  sys.stderr.write("--- %s ---\n" % name)
  p11 = (2, 2)
  p12 = (5, 2)
  
  p21 = (2, 3)
  p22 = (6, 3)
  
  p31 = (3, 4)
  p32 = (4, 5)
  p33 = (5, 4)
  
  xsz = 7
  ysz = 7
  
  wdf = parms['solid_raster_width']
  
  ph1 = path.from_points((p11, p12,), wdf, parms)
  ph2 = path.from_points((p21, p22,), wdf, parms)
  ph3 = path.from_points((p31, p32, p33,), wdf, parms)
  
  bc1 = block.from_paths((ph1, path.rev(ph1),))
  bc2 = block.from_paths((ph2, path.rev(ph2),))
  bc3 = block.from_paths((ph3, path.rev(ph3),))
  
  o = (1,1)
  BS = (bc2,bc3,bc1,)
  CS = ()
  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)
  
  Delta = inf
  maxcalls = None
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  
  plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

def test_road_of_rasters(n, o, tag):
  # The input is a single road of {n}, raster lines, each line being a single 
  # block with two choices (the path and its reversal).
  #
  # There are no cooling or max-calls constraints. The starting point is
  # the given point {o}. The file name will have "_test2_{tag}" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = "%s_%02d" % (tag, n)
  sys.stderr.write("--- %s ---\n" % name)
  
  wdf = parms['solid_raster_width']
  sys.stderr.write("wdf = %.3f mm\n" % wdf)
  
  BS = []
  for k in range(n):
    p = (2, 1+k*wdf)
    q = (4, 1+k*wdf)
    ph = path.from_points((p, q,), wdf, parms)
    bc = block.from_paths((ph, path.rev(ph),))
    BS.append(bc)
    
  CS = ()

  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)

  Delta = inf
  maxcalls = None
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  
  plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

def test_matrix_of_roads(nx, ny, nr, o, tag):
  # The input has an array of {nx} by {ny} blocks. Each block is a serpentine
  # path of {nr} horizontal rasters and {nr-1} links, in the four orders and directions. 
  #
  # There are no cooling or max-calls constraints. The starting point is
  # the given point {o}. The file name will have "_test3_{tag}" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = "%s_%02dx_%02d_%02d" % (tag,nx,ny,nr)
  sys.stderr.write("--- %s ---\n" % name)
  
  wdf = parms['solid_raster_width']
  xszr = 3   # Length of raster lines.
  sys.stderr.write("wdf = %.3f mm\n" % wdf)
  
  szb = (xszr, (nr-1)*wdf) # Size of each block, not counting trace widths
  spcb = (2.0*wdf, 1)  # Extra spacing between blocks.
  org =(2,2) # Lower left corner of lower block.
  
  BS = []
  for ixb in range(nx):
    for iyb in range(ny):
      # Compute the lower left corner {orgb} of block.
      orgb = rn.add(org, rn.scale((ixb,iyb), rn.add(szb, spcb))) 
      # Create a list {PS} of the four block choices:
      PS = [] # Choices for the block.
      for od in range(2):
        # Create a path of alternating rasters with order {od}:
        use_jumps = False
        ystep = (1 - 2*od)*wdf
        ph = example_path.raster_rectangle(nr, orgb, xszr, ystep, wdf, use_jumps, parms)
        PS.append(ph)
        PS.append(path.rev(ph))
      bc = block.from_paths(PS)
      BS.append(bc)

  CS = ()

  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)

  Delta = inf
  maxcalls = None
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  
  plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

def test_two_roads_and_islands(nmv, nmg, nis, o, Delta, maxcalls, tag):
  # The input is two "roads", each with {nmv} rasters grouped into blocks of {nmg} rasters each, and {nis}
  # "islands" at left, between, and at right of the roads
  #
  # There are constraints between successive rasters. The starting point is
  # the given point {o}. The file name will have "_test4_{nmv}_{nis}_{tag}" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = "%s_%02d_%02d_%02d" % (tag,nmv,nmg,nis)
  sys.stderr.write("--- %s ---\n" % name)
  
  wdf = 0.50
  wdc = wdf/2

  BS, CS = input_data.two_roads_and_islands(nmv, nmg, nis, wdc, wdf, parms)

  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)
    
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  if Q != None:
    plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

wdfill = parms['solid_raster_width']

def test1():
  test_tiny("test1")

def test2():
  n = 5
  o = (1,1)
  test_road_of_rasters(n, o, "test2")

def test3():
  n = 6
  o = (1,1+(n-1)/2*wdfill)
  test_road_of_rasters(n, o, "test3")

def test4():
  nx = 3
  ny = 2
  nr = 7
  o = (1, 1)
  test_matrix_of_roads(nx, ny, nr, o, "test3")

def test5():
  nmv = 7
  nmg = 1
  nis = 1
  o = (1,1)
  Delta = 6
  maxcalls = 3000000
  test_two_roads_and_islands(nmv, nmg, nis, o, Delta, maxcalls, "test5")

def test6():
  nmv = 17
  nmg = 8
  nis = 2
  o = (1,1)
  Delta = 3.5
  maxcalls = 3000000
  test_two_roads_and_islands(nmv, nmg, nis, o, Delta, maxcalls, "test6")

# test1()
# test2()
# test3()
# test4()
test5()
# test6()
