#! /usr/bin/python3
# Last edited on 2024-09-20 11:28:07 by stolfi

from math import sin, cos, tan, sqrt, pi, inf, floor, sqrt
import rn
import sys
import re

from slicing_keg_vertices import make_vertices
from slicing_keg_faces_orig import make_edges_faces_orig
from slicing_keg_faces_triang import triangulate
from slicing_lib import min_edge_length, min_triangle_width

import OBJ_format
import POV_format
 
# The object is a polyhedron inscribed in a barrel-like solid of
# revolution with {Z} as the axis. Its shape is determined by the
# following parameters
#
#   {Ns}       Number of facets in each horizontal section.
#   {Nr}       Number of facets along each vertical half-section.
#   {Nb}       Number of extra vertices inserted in each vertical-ish edge.
#   {R_keg}    Radius of enclosing barrel at equator.
#   {H_keg}    Half of {Z} extent of keg.
# 
# The main part of the object's mesh is a grid of {Ns} by {Nr}
# trapezoids; specifically, {Ns} around the circumference and {Nr} in
# the {Z} direction. Each vertical-ish side of these faces is divided
# into {Nb+1} collinear edges by {Nb} extra vertices inserted into it.
# The top and bottom of the keg are closed by {Ns} horizontal triangles
# meeting at the {Z}-axis.
#
# The faces of the mesh are identified an index {ks} in {0..Ns-1} in CCW
# order seen from {+Z}, and an index {kr} in {0..Nr-1} from bottom to top.
#
# The object has {Ns}-fold rotational symmetry about the {Z}-axis,
# and mirror symmetry about the {Z=0} plane.
#
# COMMAND LINE
#
# The program takes three parameters from the command line: {Tag}, {Ns},
# {Nr}, and {Nb}. The other parameters are defined internally.

def main():
  Tag = sys.argv[1]
  Ns = int(sys.argv[2])
  Nr = int(sys.argv[3])
  Nb = int(sys.argv[4])

  assert Ns >= 3, "{Ns} must be 3 or more"
  assert Nr >= 2, "{Nr} must be 2 or more"
  assert Nb >= 0, "{Nb} must be non-negative"

  R_keg = 40
  H_keg = 50
  Nchk = max(Ns,Nr*Nb)
  prec = 7
  
  Vind, Vlst = make_vertices(Ns, Nr, Nb, R_keg, H_keg, prec, verbose = False)
  Elst, Flst = make_edges_faces_orig(Ns, Nr, Nb, Vind, Vlst, prec, verbose = False)
  Etri, Ftri = triangulate(Ns, Nr, Nb, Vind, Vlst, Elst, Flst, prec, verbose = False)
  
  dmin = min_edge_length(Vlst, Etri)
  sys.stderr.write("min edge length = %.16f\n" % dmin)
   
  hmin = min_triangle_width(Vlst, Ftri)
  sys.stderr.write("min triangle width = %.16f\n" % hmin)
 
  opref = f"out/{Tag}_ns{Ns:05d}_nr{Nr:05d}_nb{Nb:05d}"
  OBJ_format.write_object(opref + "_orig.obj",   Vlst, Elst, Flst, prec)
  # OBJ_format.write_object(opref + "_triang.obj", Vlst, Etri, Ftri, prec)

  if max(Ns,Nr,Nb) < 20:
    Np = 0; Zp_min = -1; Zp_max = +1
    emag = 1.00
    vmag = 2.50
    POV_format.write_object(opref + "_orig.inc",   Vlst, Elst, Flst, Np, Zp_min, Zp_max, emag, vmag, prec)
    # POV_format.write_object(opref + "_triang.inc", Vlst, Etri, Ftri, Np, Zp_min, Zp_max, emag, vmag, prec)

  return 0

main()  

 
