#! /usr/bin/python3
# Last edited on 2024-09-18 19:16:14 by stolfi
 
from math import sin, cos, tan, sqrt, pi, inf, floor, sqrt
import rn
import sys
import re

from slicing_lib import make_edge, make_face, check_repeated_edge, prte, prtf

def monotonize(Ni, Vind, Vlst, Elst, Flst, prec, verbose):
  # See {slicing_box_example.py} and
  # {slicing_box_vertices.make_vertices} for the description of the
  # object.
  #
  # Subdivides the faces of the original mesh of the object into
  # {Z}-monotone polygons. Assumes that the vertices are given by the
  # structured table {Vind} and flat list {Vlst}, the edges are given by
  # the flat list {Elst}, and the faces are given by the flat list
  # {Flst}. The lists {Vlst,Elst,Flst} are indexed from 1; element 0 is
  # not used.
  #
  # Returns new flat lists {Fmon} and {Emon} of faces and edges of the
  # monotone refinement of the mesh. No new vertices are created. The
  # original edges are preserved with their original indices, and the
  # new diagonal edges are appended after them, with {etype=1}. The
  # faces are replaced by the monotone polygons.
  #
  # Original faces "fi.{ks}{kz}.{ki}" (indentation), "fc.{ks}{kr}"
  # (chamfer), and "fh.{ks}{kz}" (bottom/top horizontal corner) are
  # preserved, with the same labels, since they are monotone. Each plaza
  # "fp.{ks}{kr}" is split into two monotone polygons,
  # with labels "fp.{ks}{kr}.{km}" where {km} is 0 or 1.
  # 
  # The vertical diagonal added across each plaza will have labels
  # "dp.{ks}{kr}".

  Nv = len(Vlst)-1     # Total vertex count (input and output).
  Ne_in = len(Elst)-1  # Total input edge count.
  Nf_in = len(Flst)-1  # Total input faces.
  
  # Output element counts PER WALL:
  Nv_w = 2*2*(2*Ni+1)        # Indentation vertices, inner/outer, lower/upper.

  Nd_w_p = 2                 # Added diagonals in plaza faces, inner/outer
  
  Nf_w_i = 2*(2*Ni)          # Monotone polygons from indentation faces, upper/lower.
  Nf_w_h = 2                 # Monotone polygons from horizontal corner faces, upper/lower.
  Nf_w_c = 2                 # Monotone polygons from chamfer faces, inner/outer.
  Nf_w_p = 2*2               # Monotone polygons from plaza faces, inner/outer, left/right
  
  assert Nv == 4*Nv_w, "{Ni,Vlst} inconsistent"
  Ne_ot = Ne_in + 4*Nd_w_p                       # Expected total output edge count.
  Nf_ot = 4*(Nf_w_i + Nf_w_h + Nf_w_c + Nf_w_p)  # Expected total output face (monotone polygon) count.

  Fmon = [None]*(Nf_ot+1)
  Emon = Elst.copy() + [None]*(Ne_ot-Ne_in)
  
  Eset = set()  # Set of edges as pairs, to check for repetitions.
  for je in range(1,Ne_in+1): 
    e = Elst[je]
    Eset.add((e[0],e[1]))
  
  ne = Ne_in # Number of edges created so far.
  nf = 0     # Number of faces created so far.
 
  def Svdi(kv_org, kv_dst, elab):
    # Adds the diagonal edge from {Vlst[kv_org]} to {Vlst[kv_dst]} to {Emon},
    # reoriented in increasing index sense.
    nonlocal ne, nf
    if kv_org > kv_dst: kv_org, kv_dst = kv_dst, kv_org
    etype = 1
    e = make_edge(Vlst[kv_org], Vlst[kv_dst], ne+1, etype, elab, prec)
    assert e != None
    if verbose: prte(e, prec)
    ne += 1
    Emon[ne] = e
    check_repeated_edge(ne, Emon, Eset, prec)
    return None
    
  def Mon(Miv,flab):
    # Adds to {Emon} the polygonal face with vertices {Vlst[kv]} for
    # {kv} in {Miv}, assumed to be monotone and CCW seen from outside.
    nonlocal ne, nf
    nf += 1
    t = make_face(Miv, nf, flab, Vlst, prec)
    if verbose: prtf(t, Vlst, prec)
    Fmon[nf] = t

  for f in Flst[1:]:
    Nx,Ny,Nz,Fiv,kf,flab = f;
    if verbose: 
      sys.stderr.write("\n")
      sys.stderr.write("  IN: ")
      prtf(f, Vlst, prec)
      sys.stderr.write("\n")
    
    ks = int(flab[3]); assert ks >= 0 and ks < 4
    if flab[0:2] == "fi" or flab[0:2] == "fh" or flab[0:2] == "fc":
      # Indentation, horizontal, or chamfer face.
      assert len(Fiv) == 4
      Mon(Fiv,flab)
    elif flab[0:2] == "fp":
      # Plaza face:
      assert len(Fiv) == 2*(2*Ni+1);
      # Assume that the vertices are lower chain then upper chain.
      jmid0 = Ni
      jend0 = 2*Ni
      jmid1 = jmid0 + 2*Ni+1
      Miv0 = Fiv[:jmid0+1] + Fiv[jmid1:]
      Miv1 = Fiv[jmid0:jmid1+1]
      Mon(Miv0,flab + f".0")
      Mon(Miv1,flab + f".1")
      Svdi(Fiv[Ni],Fiv[3*Ni+1],"d" + flab[1:])
    else:
      assert False, f"invalid face label '{flab}'"

  sys.stderr.write("generated %d faces (expected %d)\n" % (nf, Nf_ot))
  if nf < Nf_ot: sys.stderr.write("!! missing some faces\n")
  assert nf <= Nf_ot

  sys.stderr.write("generated %d edges (expected %d)\n" % (ne, Ne_ot))
  if ne < Ne_ot: sys.stderr.write("!! missing some edges\n")
  assert ne <= Ne_ot

  return Emon, Fmon
