// The DCC abacus

// Assumes defined the following things:
//   These are relative to the slider's overall radius:
//     abacus_slider_width       = thickness of sliders (along hole)
//     abacus_slider_round_min_R = radius of filet around rim
//     abacus_slider_sin_t       = sin of angle between chamfer and base of slider
//     abacus_slider_cos_t       = cos of angle between chamfer and base of slider
//     abacus_slider_hole_R      = radius of slider holes

//   These are absolute:
//     abacus_width              = width (X) of abacus, including frame
//     abacus_height             = height (Z) of abacus, including frame
//     abacus_frame_width        = width of abacus frame, in both directions
//     abacus_frame_thickness    = thickness (Y) of abacus frame
//     abacus_stick_R            = radius of slider-bearing sticks

//     abacus_frame_texture      = texture for frame (long axis is X axis)
//     abacus_slider_{texture,interior} = texture and interior for the sliders
//     abacus_stick_texture      = texture for slider-bearing sticks

// Defines 
//  abacus_slider  = 
//     the slider piece, outer radius = 1, thickness = 2*abacus_slider_HY
//     centered on the origin and threaded on the Y axis.
//  abacus  = 
//     the whole abacus, centered on the origin and perpendicular
//     to the Y axis, with sticks parallel to X axis.

#declare abacus_slider_HY = abacus_slider_width / 2;

#declare abacus_slider_round_maj_R  =  
  1 - abacus_slider_round_min_R;

#declare abacus_slider_round_HY  =  
  abacus_slider_round_min_R * abacus_slider_cos_t;

#declare abacus_slider_cap_base_Y  =  
  abacus_slider_round_HY;

#declare abacus_slider_cap_base_R  =  
  abacus_slider_round_maj_R  + abacus_slider_round_min_R / abacus_slider_sin_t;

#declare abacus_slider_cap_height  =  
  abacus_slider_cap_base_R * (abacus_slider_sin_t / abacus_slider_cos_t);

#declare abacus_slider_cap_join_R  =  
  abacus_slider_round_maj_R  + abacus_slider_round_min_R * abacus_slider_sin_t;

#declare abacus_slider_bounder =
  cylinder{
    <0, -abacus_slider_HY, 0>, 
    <0, +abacus_slider_HY, 0>, 
    1.0
  }

#declare abacus_slider_joiner_cyl =
  cylinder{
    <0, -abacus_slider_HY, 0>, 
    <0, +abacus_slider_HY, 0>, 
    1.00001*abacus_slider_cap_join_R
  }

#declare abacus_slider_cap_cone =
  cone{
    <0, -0.0001*abacus_slider_cap_height, 0>, 1.0001*abacus_slider_cap_base_R 
    <0, abacus_slider_cap_height, 0>, 0
  }

#declare abacus_slider_caps =
  intersection{
    merge{
      object{abacus_slider_cap_cone}
      object{abacus_slider_cap_cone scale <1,-1,1>}
    }
    object{abacus_slider_joiner_cyl}
  }

#declare abacus_slider_round =
  difference{
    torus{abacus_slider_round_maj_R, abacus_slider_round_min_R}
    object{abacus_slider_joiner_cyl scale <0.99999, 1.00001, 0.99999>}
  }
    
#declare abacus_slider_hole =
  cylinder{
    <0, -1.1*abacus_slider_HY, 0>, 
    <0, +1.1*abacus_slider_HY, 0>, 
    abacus_slider_hole_R
  }

#declare abacus_slider =
  difference{
    merge{
      object{abacus_slider_caps}
      abacus_slider_round
    }
    object{abacus_slider_hole}
    bounded_by{object{abacus_slider_bounder scale 1.0001}}
  }    

#declare abacus_space_HX = (abacus_width - 2 * abacus_frame_width) / 2
#declare abacus_space_HZ = (abacus_height - 2 * abacus_frame_width) / 2

#declare abacus_slider_scaled_width = (2 * abacus_space_HX - abacus_frame_width) / 9;

#declare abacus_slider_scale = abacus_slider_scaled_width / abacus_slider_width

// Properly rotated, scaled, and Y-displaced slider:
#declare abacus_scaled_slider = 
  object{abacus_slider
    rotate <0,0,90>
    scale abacus_slider_scale
    translate <0, abacus_stick_R - abacus_slider_scale * abacus_slider_hole_R, 0>
  }

#declare abacus_frame_HY = abacus_frame_thickness / 2;

#declare abacus_frame_long_side =
  intersection{
    box{
      < -(abacus_space_HX + abacus_frame_width), 
        -abacus_frame_HY, 
        -abacus_frame_width
      >, 
      < +(abacus_space_HX + abacus_frame_width), 
        +abacus_frame_HY, 
        0
      >
    }
    plane{<-1,00,+1>, 0 translate <-abacus_space_HX,0,0>}
    plane{<+1,00,+1>, 0 translate <+abacus_space_HX,0,0>}
  }

#declare abacus_frame_short_side =
  intersection{
    box{
      < -(abacus_space_HZ + abacus_frame_width),
        -abacus_frame_HY,
        -abacus_frame_width
      >, 
      < +(abacus_space_HZ + abacus_frame_width),
        +abacus_frame_HY, 
        0
      >
    }
    plane{<-1,00,+1>, 0 translate <-abacus_space_HZ,0,0>}
    plane{<+1,00,+1>, 0 translate <+abacus_space_HZ,0,0>}
   }
  

#declare abacus_frame_bottom_side =
  object{abacus_frame_long_side
    translate <0,0,-abacus_space_HZ>
    texture{abacus_frame_texture}
  }

#declare abacus_frame_top_side =
  object{abacus_frame_long_side 
    scale <1,1,-1>
    translate <0,0,+abacus_space_HZ>
    texture{abacus_frame_texture}
  }

#declare abacus_frame_left_side =
  object{abacus_frame_short_side
    translate <0,0,-abacus_space_HX>
    texture{abacus_frame_texture}
    rotate <0,90,0>
  }

#declare abacus_frame_right_side =
  object{abacus_frame_short_side
    scale <1,1,-1>
    translate <0,0,+abacus_space_HX>
    texture{abacus_frame_texture}
    rotate <0,90,0>
  }

#declare abacus_frame_separator_X = 
  - abacus_space_HX + 6 * abacus_slider_scaled_width + abacus_frame_width/2;

#declare abacus_frame_separator =
  box{
    < -abacus_space_HZ, 
      -abacus_frame_HY, 
      -abacus_frame_width/2
    >, 
    < +abacus_space_HZ,
      +abacus_frame_HY, 
      +abacus_frame_width/2
    >
    translate <0,0,abacus_frame_separator_X>
    texture{abacus_frame_texture}
    rotate <0,90,0>
  }

#declare abacus_frame =
  merge{
    object{abacus_frame_bottom_side}
    object{abacus_frame_top_side}
    object{abacus_frame_left_side}
    object{abacus_frame_right_side}
    object{abacus_frame_separator}
  }
  
#declare abacus_stick =
  cylinder{
    <-(abacus_space_HX + abacus_frame_width/2), 0, 0>,
    <+(abacus_space_HX + abacus_frame_width/2), 0, 0>
    abacus_stick_R
    texture{abacus_stick_texture}
  }
  
#declare abacus =
  union{
    object{abacus_frame}
    // Top stick:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 1.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 1.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      translate <0, 0.5*abacus_frame_HY, +0.6*abacus_space_HZ>
    }
    // Second stick from top:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 4.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      translate <0,0.5*abacus_frame_HY,+0.2*abacus_space_HZ>
    }
    // Third stick from top:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
        translate <-abacus_space_HX + 4.5*abacus_slider_scaled_width, 0, 0>
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      translate <0,0.5*abacus_frame_HY,-0.2*abacus_space_HZ>
    }
    // Bottom stick:
    union{
      object{abacus_stick}
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 2.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 3.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 4.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <-abacus_space_HX + 5.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        translate <+abacus_space_HX - 0.5*abacus_slider_scaled_width, 0, 0>
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
      }
      object{abacus_scaled_slider 
        texture{ abacus_slider_texture } interior{ abacus_slider_interior }
        translate <+abacus_space_HX - 2.5*abacus_slider_scaled_width, 0, 0>
      }
      translate <0,0.5*abacus_frame_HY,-0.6*abacus_space_HZ>
    }
  }
