#define PROG_NAME "pnmdiffop"
#define PROG_DESC "PBM/PGM/PPM differential operatios -- gradient,hessian,laplacian,elongation"
#define PROG_VERS "1.0"

#define pnmdiffop_C_COPYRIGHT \
  "Copyright  2006 by the State University of Campinas (UNICAMP)"

/* Last edited on 2010-08-24 03:00:59 by stolfi */

/* !!! Replace the {RMAG} parameter by -scale and -offset options. !!! */

#define PROG_HELP \
  PROG_NAME " \\\n" \
  "  -op { dx | dy | dxx | dxy | dyy | gradient | laplacian | orthicity | elongation | average | deviation } \\\n" \
  "  [ -squared ] \\\n" \
  "  [ -scale {SCALE} ] \\\n" \
  "  [ -offset {OFFSET} ] \\\n" \
  "  [ -maxval {OMAXVAL} ] \\\n" \
  "  [ -badIn {IBAD} [ -keepBad ] ] \\\n" \
  "  [ -badOut {OBAD} ] \\\n" \
  "  [ -isMaskIn {IMK} ] \\\n" \
  "  [ -isMaskOut {OMK} ] \\\n" \
  "  [ -replicate ] \\\n" \
  "  [ -yup ] \\\n" \
  "  [<] INFILE \\\n" \
  "  [>] OUTFILE "

#define PROG_INFO \
  "NAME\n" \
  "  " PROG_NAME " - " PROG_DESC "\n" \
  "\n" \
  "SYNOPSIS\n" \
  "  " PROG_HELP "\n" \
  "\n" \
  "DESCRIPTION\n" \
  "  This program applies one of several differential operator to an image {U}, read" \
  " from the PBM, PGM or PPM file {INFILE}.  The differential operator is" \
  " selected by the \"-op\" option.  The output is another" \
  " image {V} with the same number of channels and same dimensions, written to the" \
  " file {OUTFILE} with the same sample" \
  " type (PBM, PGM, or PPM) and same encoding (`raw' or `plain').\n" \
  "  For monochromatic" \
  " images, the sample of {V} at any pixel is" \
  " computed from the samples of {U} within a 3 by 3 window centered at the" \
  " same pixel of {U}.  For color (PPM) images," \
  " each channel is processed independently, as a separate monochromatic image.\n" \
  "\n" \
  "IMAGE ENCODING\n" \
  "  The samples in the input and output images are assumed to have linear" \
  " encoding (gamma 1.0).  The input image samples are converted to float values" \
  " in {[0_1]}, as determined by the \"-badIn\" and \"-isMaskIn\" arguments.  Then" \
  " the formula of the requested operator is applied (see the \"-op\" and" \
  " \"-squared\" options), the result is scaled as requested by" \
  " the \"-scale\" and \"-offset\" options, and clipped to the" \
  " interval {[0_1]}.  IN the output images, these results are encoded" \
  " as integers in the range {0..OMAXVAL}, as determined by" \
  " the \"-maxval\", \"-badOut\", and \"-isMaskOut\" arguments.\n" \
  "\n" \
  "INVALID SAMPLE VALUES\n" \
  "  The program allows a specific sample value in the input image to be declared" \
  " as meaning `undefined value'.  See" \
  " the \"-badIn\" option below.  Any output sample that depends" \
  " on `undefined' input samples will itself be `undefined'.  Pixels outside the image domain may or" \
  " may not be considered `undefined' (see \"-replicate\" option below).  Similarly," \
  " a specific output sample value" \
  " may be used to encode `undefined' or `invalid' results.  See" \
  " the \"-badOut\" option below.\n" \
  "\n" \
  "OPTIONS\n" \
  "  -op {OPKIND}.. \n" \
  "    Specifies the operator to use.  The valid alternatives are:\n" \
  "\n" \
  PROG_INFO_OPERATORS "\n" \
  "\n" \
  "  -squared\n" \
  "    This option replaces the output of the operator by its square.\n" \
  "  -scale {SCALE}\n" \
  "  -offset {OFFSET}\n" \
  "    These options specify that the result of the operator (possibly squared) be" \
  " rescaled by an affine function (polinomial of degree 1).  Namely, a raw result {v} (as" \
  " computed by the formulas above) is replaced by {SCALE*v + OFFSET}.  Both" \
  "h coefficients may be fractional numers, zero, or negative.\n" \
  "\n  Certain choices of {SCALE} and {OFFSET} may result in rescaled output" \
  " values that lie outside the range {[0_1]}.  Any such values are" \
  " clipped to {[0_1]} before conversion to integer pixels.  If either" \
  " option is omitted, the program chooses a default value for the" \
  " corresponding coefficient so that clipping does not occur.  Ditto" \
  " if both are omitted.\n" \
  "\n" \
  PROG_INFO_OP_RANGES \
  "\n" \
  "  The default {SCALE} is positive and as large as possible, and the" \
  " default {OFFSET} is such that the rescaled output range is centered in {[0_1]}." \
  "\n" \
  "  -maxval {OMAXVAL} \n" \
  "    Defines the nominal maximum sample value for the output" \
  " image.  Must be an integer between 1" \
  " and " stringify(PNM_MAX_MAXVAL) ".  If not specified," \
  " defaults to the input maximum sample value {IMAXVAL}.\n" \
  "\n" \
  "  -badIn {IBAD} \n" \
  "    If this option is present, and {IBAD} is in the range" \
  " {0..IMAXVAL}, any input sample with value {IBAD} is assumed" \
  " to be `undefined', and its weight is set to 0.  In that" \
  " case, the program implicitly subtracts 1 from any input" \
  " sample value in the range {IBAD+1..IMAXVAL}, so that the" \
  " nominal input range becomes {0..IMAXVAL-1}.  Thus, for" \
  " example, given \"-badIn 2\" and {IMAXVAL = 6}, input" \
  " sample values 0 through 6 are interpreted" \
  " as 0, 1, undefined, 2, 3, 4, and 5, respectively.\n" \
  "\n" \
  "  -keepBad \n" \
  "    This option forces an invalid output sample value" \
  " whenever the corresponding input sample is invalid.  It is effective" \
  " only if \"-badIn\" is specified, and only for some operators, such" \
  " as \"dx\", \"dy\", \"dxy\", and \"gradient\"," \
  " whose formulas do not use directly the center sample in the window.\n" \
  "\n" \
  "  -badOut {OBAD} \n" \
  "    If this option is present, and {OBAD} is in" \
  " the range {0..OMAXVAL}, the output sample value {OBAD}" \
  " is reserved to represent `undefined' or `invalid' operator results.  In" \
  " that case, valid operator results will be scaled to the range {0..OMAXVAL-1}" \
  " instead of {0..OMAXVAL}, and then results in the range {OBAD..OMAXVAL-1} will" \
  " be incremented by 1. Thus, for example, given \"-badOut 2\" and" \
  " {OMAXVAL = 6}, the valid operator results will be scaled so as to range" \
  " from 0 through 5, and output values 0 through 6 will mean results" \
  " 0, 1, undefined, 2, 3, 4, and 5, respectively.\n" \
  "\n" \
  "    If \"-badOut\" is omitted but \"-badIn\" is given," \
  " and the input and output ranges are the same ({OMAXVAL=IMAXVAL})," \
  " then {OBAD} defaults to {IBAD} --- that is, the" \
  " same `undefined value' convention will be used for" \
  " the input and output images.  Otherwise," \
  " if \"-badOut\" is omitted, the output" \
  " will have no special `undefined' value." \
  " Valid operator results will be scaled to the" \
  " range {0..OMAXVAL}, and  invalid results will be mapped" \
  " to {OMAXVAL/2}, without any adjustment of higher values.\n" \
  "\n" \
  "  -isMaskIn {IMK} \n" \
  "    This Boolean option specifies the method for conversion of input image" \
  " samples to float values in the real interval {[0_1]}.  Let {0..MAXVAL} be" \
  " the effective range of the integer samples from the input image, after" \
  " excluding the invalid value {IBAD} if given.  If the argument {IMK} is true (\"T\" or 1)," \
  " " sample_conv_0_1_isMask_true_INFO "  If the argument {IMK} is false (\"F\" or 0)," \
  " " sample_conv_0_1_isMask_false_INFO "  The default is \"-isMaskIn F\".\n" \
  "\n" \
  "  -isMaskOut {OMK} \n" \
  "    This Boolean option specifies the method for rounding computed float" \
  " samples, nominally in the real interval {[0_1]}, to integer samples" \
  " in the output image.  Let {0..MAXVAL} being the effective range of" \
  " the integer samples in the output image, namely {0..OMAXVAL} but" \
  " excluding the invalid value {OBAD} if given.  The two possibilities" \
  " selected by {OMK} are those described under the \"-isMaskIn\" option.  The" \
  " default is {OMK=IMK}.\n" \
  "\n" \
  "  -replicate\n" \
  "    This option specifies that whenever the operator needs the value of" \
  " a pixel just outside the image domain, the nearest pixel inside the" \
  " domain should be used instead.  Othwerwise, by default, those pixels" \
  " are assumed to have `undefined' value, so the formula's will be `undefined' too.\n" \
  "\n" \
  argparser_help_info_HELP_INFO "\n" \
  "SEE ALSO\n" \
  "  pnmnlfilt(1), pnmconvol(1), pgmenhance(1)\n" \
  "\n" \
  "AUTHOR\n" \
  "  Created on 2010-08-19 by J. Stolfi (IC-UNICAMP), based on Jef Poskanzer Netpbm image file" \
  " formats and earlier PGM filter programs.\n" \
  "\n" \
  "MODIFICATION HISTORY\n" \
  "  2010-08-19 created.\n" \
  "\n" \
  "WARRANTY\n" \
  argparser_help_info_NO_WARRANTY "\n" \
  "\n" \
  "RIGHTS\n" \
  "  " pnmdiffop_C_COPYRIGHT ".\n" \
  "\n" \
  argparser_help_info_STANDARD_RIGHTS

#define PROG_INFO_OPERATORS \
  "  The operators \"dx\", \"dy\", \"dxx\", \"dxy\", \"dyy\" are the spatial" \
  " derivatives of the input image.  Each derivative are computed" \
  " by the simplest unbiased finite difference\n" \
  " formula.  Specifically, the sample {V[x,y]}, on column {x} and row {y} of {V}, is" \
  " computed from the input image {U} as follows:\n" \
  "\n" \
  "    {dx}         {(U[x+1,y] - U[x-1,y])/2}\n" \
  "    {dy}         {(U[x,y+1] - U[x,y-1])/2}\n" \
  "    {dxx}        {U[x-1,y] + U[x+1,y] - 2*U[x,y]}\n" \
  "    {dyy}        {U[x,y-1] + U[x,y+1] - 2*U[x,y]}\n" \
  "    {dxy}        {(U[x+1,y+1] - U[x-1,y+1] - U[x+1,y-1] + U[x-1,y-1])/4}\n" \
  "    {gradient}   {hypot(dx,dy)}\n" \
  "    {laplacian}  {dxx+dyy}\n" \
  "    {orthicity}  {dxx-dyy}\n" \
  "    {elongation} {hypot(2*dxy,dxx-dxy)}\n" \
  "    {average}    {weighted average of 3x3 window}\n" \
  "    {deviation}  {root weighted mean square deviation from average}\n" \
  "\n" \
  "  The names \"orthicity\" and \"elongation\" are made up; the" \
  " Laplacian and the elongation modulus are the two second-order" \
  " rotational invariants.  The average {avg} uses weights 4 for the" \
  " center pixel, 2 for its four nearest neighbors, and 1 for the corner" \
  " pixels.  The variance {var} is the weighted mean of {(U[x+r,y+s]-avg)^2}, using" \
  " the same weights; the deviation is the square root of the variance."

#define PROG_INFO_OP_RANGES \
  "       The natural ranges (before rescaling) for the operators, normal and squared, are:\n" \
  "\n" \
  "    Operator     Plain                Squared\n" \
  "\n" \
  "    {dx}         {[-1/2 _ +1/2]}      {[0 _ 1/4]}\n" \
  "    {dy}         {[-1/2 _ +1/2]}      {[0 _ 1/4]}\n" \
  "    {dxx}        {[-2 _ +2]}          {[0 _ 4]}  \n" \
  "    {dyy}        {[-2 _ +2]}          {[0 _ 4]}  \n" \
  "    {dxy}        {[-1/2 _ +1/2]}      {[0 _ 1/2]}\n" \
  "    {gradient}   {[0 _ +sqrt(1/2)]}   {[0 _ 1/2]}\n" \
  "    {laplacian}  {[-4 _ +4]}          {[0 _ 16]} \n" \
  "    {orthicity}  {[-2 _ +2]}          {[0 _ 4]} \n" \
  "    {elongation} {[0 _ sqrt(5)]}      {[0 _ 5]}  \n" \
  "    {average}    {[0 _ 1]}            {[0 _ 1]}  \n" \
  "    {deviation}  {[0 _ 1/2]}          {[0 _ 1/4]}\n" \

#define stringify(x) #x

#define _GNU_SOURCE
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include <assert.h>
#include <limits.h>
#include <values.h>

#include <jspnm.h> 
#include <jsfile.h> 
#include <indexing.h> 
#include <jspnm_image.h>
#include <float_pnm_image_stream.h>
#include <float_image_buffer.h>
#include <sample_conv.h>
#include <affirm.h> 
#include <argparser.h> 

#define INF INFINITY

/* DATA TYPES */

typedef enum
  { OP_KIND_DX,         /* Horizontal derivative. */
    OP_KIND_DY,         /* Vertical derivative. */
    OP_KIND_DXX,        /* Second derivative wrt {x}. */
    OP_KIND_DXY,        /* Mixed second derivative. */
    OP_KIND_DYY,        /* Second derivative wrt {y}. */
    OP_KIND_GRADIENT,   /* Gradient modulus. */
    OP_KIND_LAPLACIAN,  /* Laplacian {dxx+dyy}. */
    OP_KIND_ORTHICITY,  /* Orthicity {dxx-dyy}. */
    OP_KIND_ELONGATION, /* Elongation modulus {hypot(2*dxy,dxx-dyy)}. */
    OP_KIND_AVERAGE,    /* Weighted average. */
    OP_KIND_DEVIATION   /* Weighted deviation from average. */
  } op_kind_t;
  /* Kind of operator. */

typedef struct options_t 
  { char *iname;         /* Input filename ("-" for stdin). */
    char *mname;         /* Mask image name ("-" for stdin,{NULL} if not given). */ 
    char *wname;         /* Window/weight file name ("-" for stdin). */ 
    char *oname;         /* Output filename ("-" for stdout). */
    bool_t verbose;      /* TRUE says to mumble while working. */
    /* Input/output encoding and undefined value handling: */
    pnm_sample_t maxval; /* Output maxval (0 if not specified). */
    uint32_t badIn;      /* Input sample value that means `undefined' ({>maxval} if none). */
    bool_t keepBad;      /* TRUE forces undef output when input is undef. */
    bool_t isMaskIn;     /* TRUE if input 0 and {maxval} are to be mapped to 0.0 and 1.0. */
    uint32_t badOut;     /* Output sample value that means `undefined' ({>maxval} if none). */
    bool_t isMaskOut;    /* TRUE if output 0 and {maxval} are to mean 0.0 and 1,0. */
    bool_t replicate;    /* TRUE specifies edge replication. */
    /* Operator parameters (may be {NAN} if not applicable): */
    op_kind_t op_kind;   /* Kind of operator to apply. */
    bool_t squared;      /* TRUE to square the operator's result. */
    double scale;        /* Affine scaling coefficient for the result, or {NAN} if not given. */
    double offset;       /* Affine offset for the result, or {NAN} if not given. */
  } options_t;
  /* Arguments parsed from the command line. */

#define MAX_CHNS 3
  /* Max channels in a PNM file. */

/* INTERNAL PROTOTYPES */

int main(int argc, char* argv[]);

options_t *parse_options(int argc, char **argv);

void filter_image_file
  ( FILE *ifile,
    FILE *ofile,
    options_t *o
  );
  /* Reads a PBM, PGM, or PPM image from file {ifile}, applies the operator described
    by {o}, and writes the result to {ofile}. */

void grab_input_samples
  ( int c,                      /* Channel of input image. */
    int x,                      /* Column of center pixel in input image. */
    int y,                      /* Row of center pixel in input image. */
    float_image_buffer_t *ibuf, /* Circular row buffer of input image. */
    bool_t replicate,           /* If TRUE, extends image by edge replication. */
    int wrx,                    /* Half-width of window. */
    int wry,                    /* Half-height of window. */
    double smp[]                /* OUT: {wcols*wrows} array of pixel values in window. */
  );
  /* Stores into {smp[0..nw-1]} the samples taken from channel {c} of
    the input image, within the window centered at {x,y}. The window
    dimensions are assumed to be {2*wrx+1} columns by {2*wry+1}
    rows, and it is linearized by rows. If {replicate} is FALSE,
    window samples that fall outside the image's domain are set to
    {NAN}; otherwise they are copied from the nearest edge samples.
    
    Assumes that the {wrows} rows of the input image that are spanned by the
    window are stored in {buf}. */

double apply_dx_operator(int wctr, int wcols, double smp[]);
double apply_dy_operator(int wctr, int wcols, double smp[]);
double apply_dxx_operator(int wctr, int wcols, double smp[]);
double apply_dxy_operator(int wctr, int wcols, double smp[]);
double apply_dyy_operator(int wctr, int wcols, double smp[]);
double apply_gradient_operator(int wctr, int wcols, double smp[]);
double apply_gradient_squared_operator(int wctr, int wcols, double smp[]);
double apply_laplacian_operator(int wctr, int wcols, double smp[]);
double apply_orthicity_operator(int wctr, int wcols, double smp[]);
double apply_elongation_operator(int wctr, int wcols, double smp[]);
double apply_elongation_squared_operator(int wctr, int wcols, double smp[]);
double apply_average_operator(int wctr, int wcols, double smp[]);
double apply_deviation_operator(int wctr, int wcols, double smp[]);
double apply_deviation_squared_operator(int wctr, int wcols, double smp[]);
  /* These procedures apply the relevant operator to the current
    window samples {smp[k]}, and return the output sample value for
    that pixel.
    
    These procedures assume that {smp} is linearized by rows, that
    {smp[wctr]} is the sample at the window center, and that each row
    has {wcols} samples. They also assume that no sample {smp[k]} is
    infinite. If any {smp[k]} used by the operator is {NAN}, the
    result may be {NAN}. */
    
void provide_scale_offset_defaults(op_kind_t op_kind, bool_t squared, bool_t verbose, double *scaleP, double *offsetP);
  /* If any of {*scaleP} and {*offsetP} is {NAN}, sets it to the default most appropriate
    for the given operator {op_kind} and the given {squared} option. */

void get_output_range(op_kind_t op_kind, bool_t squared, bool_t verbose, double *loP, double *hiP);
  /* Sets {*loP} and {*hiP} to the range of the operator {op_kind} with the given {squared} option,
    assuming inputs in {[0_1]}. */

/* ROUTINES */

int main(int argc, char* argv[])
  { /* Command line arguments: */
    options_t *o = parse_options(argc, argv);
    
    /* Fix defaults for scale and offset: */
    provide_scale_offset_defaults(o->op_kind, o->squared, o->verbose, &(o->scale), &(o->offset));
    
    /* Consistency check: */
    assert((int)PNM_MAX_SAMPLE < INT_MAX - 1);

    /* Read input header and get image attributes: */
    FILE *ifile = open_read(o->iname, o->verbose);
    
    /* Write output file header: */
    FILE *ofile = open_write(o->oname, o->verbose);

    filter_image_file(ifile, ofile, o);
    
    if (ifile != stdin) { fclose(ifile); }
    if ((ofile != stdout) && (ofile != stderr)) { fclose(ofile); }
    return 0;
  }
  
void provide_scale_offset_defaults(op_kind_t op_kind, bool_t squared, bool_t verbose, double *scaleP, double *offsetP)
  { 
    /* Determine the natural range {[lo_hi]} of the operator: */
    double lo, hi;
    get_output_range(op_kind, squared, verbose, &lo, &hi);
    assert(lo < hi);
    
    double scale = (*scaleP);
    double offset = (*offsetP);
    if (isnan(scale) && isnan(offset))
      { /* Map range to {[0_1]}: */
        scale = 1.0/(hi - lo);
        offset = 0.5 - scale*(hi + lo)/2;
      }
    else if (isnan(scale))
      { double rad = fmax(hi, -lo);
        scale = 0.5/rad;
      }
    else if (isnan(offset))
      { double ctr = scale*(hi + lo)/2;
        offset = 0.5 - ctr;
      }
    if (verbose)
      { fprintf(stderr, "using scale = %24.16f  offset = %24.16f\n", scale, offset); } 
    
    (*scaleP) = scale;
    (*offsetP) = offset;
  }
    
void get_output_range(op_kind_t op_kind, bool_t squared, bool_t verbose, double *loP, double *hiP)
  { 
    /* Determine the natural range {[lo_hi]} of the operator, without squaring: */
    double lo = NAN, hi = NAN;
    if (! squared)
      {
        switch(op_kind)
          {
            case OP_KIND_DX:         lo = -0.5; hi = +0.5;       break;
            case OP_KIND_DY:         lo = -0.5; hi = +0.5;       break;
            case OP_KIND_DXX:        lo = -2.0; hi = +2.0;       break;
            case OP_KIND_DXY:        lo = -0.5; hi = +0.5;       break;
            case OP_KIND_DYY:        lo = -0.5; hi = +0.5;       break;
            case OP_KIND_GRADIENT:   lo = 00.0; hi = +sqrt(0.5); break;
            case OP_KIND_LAPLACIAN:  lo = -4.0; hi = +4.0;       break;
            case OP_KIND_ORTHICITY:  lo = -2.0; hi = +2.0;       break;
            case OP_KIND_ELONGATION: lo = 00.0; hi = +sqrt(5.0); break;
            case OP_KIND_AVERAGE:    lo = 00.0; hi = +1.0;       break;
            case OP_KIND_DEVIATION:  lo = 00.0; hi = +0.5;       break;
          }
      }
    else
      { lo = 0.0;
        switch(op_kind)
          {
            case OP_KIND_DX:         hi = +0.25; break;
            case OP_KIND_DY:         hi = +0.25; break;
            case OP_KIND_DXX:        hi = +4.00; break;
            case OP_KIND_DXY:        hi = +0.25; break;
            case OP_KIND_DYY:        hi = +0.50; break;
            case OP_KIND_GRADIENT:   hi = +0.50; break;
            case OP_KIND_LAPLACIAN:  hi = +16.0; break;
            case OP_KIND_ORTHICITY:  hi = +4.00; break;
            case OP_KIND_ELONGATION: hi = +5.00; break;
            case OP_KIND_AVERAGE:    hi = +1.00; break;
            case OP_KIND_DEVIATION:  hi = +0.25; break;
          }
      }
    if (verbose)
      { fprintf(stderr, "unscaled operator range = [ %24.16f  _ %24.16f]\n", lo, hi); } 
    assert(! isnan(lo));
    assert(! isnan(hi));
    /* Return results: */
    (*loP) = lo;
    (*hiP) = hi;
  }

void filter_image_file
  ( FILE *ifile,
    FILE *ofile,
    options_t *o
  )
  { 
    /* Choose the window size: */
    int wcols = 0; /* Widnow width */
    int wrows = 0; /* Window height */
    switch(o->op_kind)
      { 
        case OP_KIND_DX:         wcols = 3; wrows = 1; break;
        case OP_KIND_DY:         wcols = 1; wrows = 3; break;
        case OP_KIND_DXX:        wcols = 3; wrows = 1; break;
        case OP_KIND_DXY:        wcols = 3; wrows = 3; break;
        case OP_KIND_DYY:        wcols = 1; wrows = 3; break;
        case OP_KIND_GRADIENT:   wcols = 3; wrows = 3; break;
        case OP_KIND_LAPLACIAN:  wcols = 3; wrows = 3; break;
        case OP_KIND_ORTHICITY:  wcols = 3; wrows = 3; break;
        case OP_KIND_ELONGATION: wcols = 3; wrows = 3; break;
        case OP_KIND_AVERAGE:    wcols = 3; wrows = 3; break;
        case OP_KIND_DEVIATION:  wcols = 3; wrows = 3; break;
      }
    assert(wcols % 2 == 1);
    assert(wrows % 2 == 1);
    int wrx = wcols/2, wry = wrows/2; /* Half-dimensions of window. */
    int wctr = wry*wcols + wrx; /* Index of center pixel in array. */
    
    if (o->verbose)
      { fprintf(stderr, "using a window with %d columns and %d rows\n", wcols, wrows); }
    
    /* Read the input file header, create a buffered read stream {istr} for it: */
    fpnm_image_stream_t *istr = fpnm_read_stream_new(ifile, o->isMaskIn, o->badIn, wrows);
    int rows = istr->rows;
    int cols = istr->cols;
    int chns = istr->chns;
    pnm_sample_t imaxval = istr->maxval;  /* Input image's {maxval}. */
    assert(imaxval > 0);
    assert((chns > 0) && (chns <= MAX_CHNS));
    float_image_buffer_t *ibuf = istr->buf;

    if (o->verbose)
      { fprintf(stderr, "image has %d channels %d columns %d rows\n", chns, cols, rows);
        fprintf(stderr, "input maxval = %d\n", imaxval);
        if (o->badIn <= imaxval) 
          { fprintf(stderr, "input samples with value %d assumed invalid\n", o->badIn); }
      }
    
    /* Choose the output maxval: */
    pnm_sample_t omaxval = (o->maxval != 0 ? o->maxval : imaxval);
    assert(omaxval > 0);
    
    /* Choose the output file parameters and write the output file header: */
    bool_t forceplain = FALSE;  /* TRUE to force the `plain' (ascii) format variant. */
    pnm_format_t oformat;       /* Format of output PNM file. */
    bool_t oraw;                /* TRUE if `raw' format variant. */
    bool_t obits;               /* TRUE for PBM format (`raw' or `plain'). */
    pnm_choose_output_format(imaxval, chns, forceplain,  &oformat, &oraw, &obits);
    pnm_write_header(ofile, cols, rows, omaxval, oformat);
    
    if (o->verbose)
      { fprintf(stderr, "output maxval = %d\n", omaxval);
        if (o->badOut <= imaxval) 
          { fprintf(stderr, "output samples with value %d represent invalid values\n", o->badOut); }
      }

    /* Allocate the output buffer for a single-row of quantized samples: */
    pnm_sample_t *osmp = pnm_image_alloc_pixel_row(cols, chns);
    
    /* Loop on output image rows: */
    double smp[wcols*wrows];  /* Linearized window sample array. */
    int y;
    for (y = 0; y < rows; y++)
      { if (o->verbose) { fputc('.', stderr); }
        int yimin = y - wry; /* Min input {y} needed to compute this row. */
        if (yimin < 0) { yimin = 0; }
        int yimax = y + wry; /* Max input {y} needed to compute this row. */
        if (yimax >= rows) { yimax = rows-1; }
        
        /* Make sure that rows {yimin..yimax} are in the input buffer: */
        (void)fpnm_read_stream_get_row(ifile, istr, yimax);
        assert(ibuf->yini <= yimin);
        assert(ibuf->ylim > yimax);
        
        /* Compute row {y} of output image: */
        int x;
        for (x = 0; x < cols; x++)
          { /* Now loop on channels: */
            int c;
            for (c = 0; c < chns; c++)
              { /* Get input pixel values and mask (if any) for the new window placement: */
                grab_input_samples(c, x, y, ibuf, o->replicate, wrx, wry, smp);
                
                /* Compute the floated output pixel value: */
                double oval = NAN;
                if ((! o->keepBad) || (! isnan(smp[wctr])))
                  {
                    switch(o->op_kind)
                      { case OP_KIND_DX:
                          oval = apply_dx_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_DY:
                          oval = apply_dy_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_DXX:
                          oval = apply_dxx_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_DXY:
                          oval = apply_dxy_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_DYY:
                          oval = apply_dyy_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_GRADIENT:
                          if (o->squared) 
                            { oval = apply_gradient_squared_operator(wctr, wcols, smp); }
                          else
                            { oval = apply_gradient_operator(wctr, wcols, smp); }
                          break;
                        case OP_KIND_LAPLACIAN:
                          oval = apply_laplacian_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_ORTHICITY:
                          oval = apply_orthicity_operator(wctr, wcols, smp);
                          if (o->squared) { oval = oval*oval; }
                          break;
                        case OP_KIND_ELONGATION:
                          if (o->squared) 
                            { oval = apply_elongation_squared_operator(wctr, wcols, smp); }
                          else
                            { oval = apply_elongation_operator(wctr, wcols, smp); }
                          break;
                        case OP_KIND_AVERAGE:
                          oval = apply_average_operator(wctr, wcols, smp);
                          break;
                        case OP_KIND_DEVIATION:
                          if (o->squared) 
                            { oval = apply_deviation_squared_operator(wctr, wcols, smp); }
                          else
                            { oval = apply_deviation_operator(wctr, wcols, smp); }
                          break;
                      }
                    /* Apply the affine map and clip: */
                    oval = o->scale*oval + o->offset;
                  }
                /* Quantize and store the output pixel value: */
                pnm_sample_t oqts = pnm_quantize(oval, omaxval, o->isMaskOut, o->badOut);
                assert(oqts <= omaxval);
                osmp[x*chns + c] = oqts;
              }
          }
        /* Write row {y} of the output image: */
        pnm_write_pixels(ofile, osmp, cols, chns, omaxval, oraw, obits);
      } 

    /* Release working storage: */
    fpnm_stream_free(istr);
    free(osmp);
  }

void grab_input_samples
  ( int c,                      /* Channel of input image. */
    int x,                      /* Column of center pixel in input image. */
    int y,                      /* Row of center pixel in input image. */
    float_image_buffer_t *ibuf, /* Circular row buffer of input image. */
    bool_t replicate,           /* If TRUE, extends image by edge replication. */
    int wrx,                    /* Half-width of window. */
    int wry,                    /* Half-height of window. */
    double smp[]                /* OUT: {wcols*wrows} array of pixel values in window. */
  )
  { int NC = ibuf->sz[0];
    int NX = ibuf->sz[1];
    int NY = ibuf->sz[2];
    
    /* Scan the window pixels: */
    int yw;
    int kw = 0;
    for (yw = -wry; yw <= wry; yw++)
      { int xw;
        for (xw = -wrx; xw <= wrx; xw++)
          { /* Compute the sample's indices {xp,yp} in input img, or -1 if non-existant: */
            int xp = x + xw;
            int yp = y + yw;
            if (replicate)
              { /* Map invalid {xp,yp} to nearest pixel in domain: */
                if (yp < 0) { yp = 0; } else if (yp >= NY) { yp = NY - 1; }
                if (xp < 0) { xp = 0; } else if (xp >= NX) { xp = NX - 1; }
              }
            else
              { /* Set invalid {xp,yp} to -1: */
                if ((xp < 0) || (xp >= NX) || (yp < 0) || (yp >= NY)) { xp = yp = -1; }
              }
            /* Get and set the value: */  
            if (yp < 0)
              { /* Pixel doesn't exist: */
                smp[kw] = NAN;
              }
            else
              { /* Pixel exists, get sample values from input image: */
                double *ismp = float_image_buffer_get_row(ibuf, yp);
                smp[kw] = ismp[xp*NC + c];
              }
            kw++;
          }
      }
  }

/* OPERATORS */

double apply_dx_operator(int wctr, int wcols, double smp[])
  { return (smp[wctr+1] - smp[wctr-1])/2; }

double apply_dy_operator(int wctr, int wcols, double smp[])
  { return (smp[wctr+wcols] - smp[wctr-wcols])/2; }

double apply_dxx_operator(int wctr, int wcols, double smp[])
  { return smp[wctr+1] + smp[wctr-1] - 2*smp[wctr]; }

double apply_dxy_operator(int wctr, int wcols, double smp[])
{ return (smp[wctr+wcols+1] - smp[wctr+wcols-1] - smp[wctr-wcols+1] + smp[wctr-wcols-1])/4; }

double apply_dyy_operator(int wctr, int wcols, double smp[])
  { return smp[wctr+wcols] + smp[wctr-wcols] - 2*smp[wctr]; }

double apply_gradient_operator(int wctr, int wcols, double smp[])
  { double dx = (smp[wctr+1] - smp[wctr-1])/2;
    double dy = (smp[wctr+wcols] - smp[wctr-wcols])/2;
    return hypot(dx,dy);
  }

double apply_gradient_squared_operator(int wctr, int wcols, double smp[])
  { double dx = (smp[wctr+1] - smp[wctr-1])/2;
    double dy = (smp[wctr+wcols] - smp[wctr-wcols])/2;
    return dx*dx + dy*dy;
  }

double apply_laplacian_operator(int wctr, int wcols, double smp[])
  { return smp[wctr+1] + smp[wctr-1] + smp[wctr+wcols] + smp[wctr-wcols] - 4*smp[wctr]; }

double apply_orthicity_operator(int wctr, int wcols, double smp[])
  { return smp[wctr+1] + smp[wctr-1] - smp[wctr+wcols] - smp[wctr-wcols]; }

double apply_elongation_operator(int wctr, int wcols, double smp[])
  { double ort = smp[wctr+1] + smp[wctr-1] - smp[wctr+wcols] - smp[wctr-wcols]; /* Orthicity. */
    double wrp = (smp[wctr+wcols+1] - smp[wctr+wcols-1] - smp[wctr-wcols+1] + smp[wctr-wcols-1])/2; /* Warp = {2*dxy}. */
    return hypot(wrp, ort);
  }

double apply_elongation_squared_operator(int wctr, int wcols, double smp[])
  { double ort = smp[wctr+1] + smp[wctr-1] - smp[wctr+wcols] - smp[wctr-wcols]; /* Orthicity. */
    double wrp = (smp[wctr+wcols+1] - smp[wctr+wcols-1] - smp[wctr-wcols+1] + smp[wctr-wcols-1])/2; /* Warp = {2*dxy}. */
    return wrp*wrp + ort*ort;
  }

double apply_average_operator(int wctr, int wcols, double smp[])
  { return 
      ( 4.0 * smp[wctr] + 
        2.0 * (smp[wctr+1] + smp[wctr-1] + smp[wctr+wcols] + smp[wctr-wcols]) +  
        smp[wctr+1+wcols] + smp[wctr+1-wcols] + smp[wctr-1+wcols] + smp[wctr-1-wcols]
      ) / 16.0;
  }

double apply_deviation_squared_operator(int wctr, int wcols, double smp[])
  { double avg = apply_average_operator(wctr, wcols, smp);
    double dmm = smp[wctr-1-wcols] - avg;
    double dmo = smp[wctr-1] - avg;
    double dmp = smp[wctr-1+wcols] - avg;
    double dom = smp[wctr-wcols] - avg;
    double doo = smp[wctr] - avg;
    double dop = smp[wctr+wcols] - avg;
    double dpm = smp[wctr+1-wcols] - avg;
    double dpo = smp[wctr+1] - avg;
    double dpp = smp[wctr+1+wcols] - avg;
    
    return 
      ( 4.0 * doo*doo + 
        2.0 * (dmo*dmo + dpo*dpo + dom*dom + dop*dop) +  
        dmm*dmm + dmp*dmp + dpm*dpm + dpp*dpp
      ) / 16.0;
  }

double apply_deviation_operator(int wctr, int wcols, double smp[])
  { double var = apply_deviation_squared_operator(wctr, wcols, smp);
    return sqrt(var);
  }

/* AUXILIARY PROCS */

options_t *parse_options(int argc, char **argv)
  {
    argparser_t *pp = argparser_new(stderr, argc, argv);
    argparser_set_help(pp, PROG_NAME " version " PROG_VERS ", usage:\n" PROG_HELP);
    argparser_set_info(pp, PROG_INFO);
    argparser_process_help_info_options(pp);

    options_t *o = (options_t *)notnull(malloc(sizeof(options_t)), "no mem"); 
    
    /* Get keyword arguments: */

    /* Operator kind and parameters: */
    if (argparser_keyword_present(pp, "-operator"))
      { 
        if (argparser_keyword_present_next(pp, "dx"))
          { o->op_kind = OP_KIND_DX; }
        else if (argparser_keyword_present_next(pp, "dy"))
          { o->op_kind = OP_KIND_DY; }
        else if (argparser_keyword_present_next(pp, "dxx"))
          { o->op_kind = OP_KIND_DXX; }
        else if (argparser_keyword_present_next(pp, "dxy"))
          { o->op_kind = OP_KIND_DXY; }
        else if (argparser_keyword_present_next(pp, "dyy"))
          { o->op_kind = OP_KIND_DYY; }
        else if (argparser_keyword_present_next(pp, "gradient"))
          { o->op_kind = OP_KIND_GRADIENT; }
        else if (argparser_keyword_present_next(pp, "laplacian"))
          { o->op_kind = OP_KIND_LAPLACIAN; }
        else if (argparser_keyword_present_next(pp, "orthicity"))
          { o->op_kind = OP_KIND_ORTHICITY; }
        else if (argparser_keyword_present_next(pp, "elongation"))
          { o->op_kind = OP_KIND_ELONGATION; }
        else if (argparser_keyword_present_next(pp, "average"))
          { o->op_kind = OP_KIND_AVERAGE; }
        else if (argparser_keyword_present_next(pp, "deviation"))
          { o->op_kind = OP_KIND_DEVIATION; }
        else 
          { argparser_error(pp, "invalid operator kind"); }
      }
    else
      { argparser_error(pp, "missing \"-operator\" option"); }

    o->squared = argparser_keyword_present(pp, "-squared");
    
    if (argparser_keyword_present(pp, "-scale"))
      { o->scale = argparser_get_next_double(pp, -1.0e+20, +1.0e+20); }
    else
      { /* Choose a good scale: */
        o->scale = NAN;
      }
    
    if (argparser_keyword_present(pp, "-offset"))
      { o->offset = argparser_get_next_double(pp, -1.0e+20, +1.0e+20); }
    else
      { /* Choose a good offset: */
        o->offset = NAN;
      }
    
    if (argparser_keyword_present(pp, "-maxval"))
      { o->maxval = argparser_get_next_int(pp, 0, PNM_FILE_MAX_MAXVAL); }
    else
      { /* Use input maxval: */
        o->maxval = 0;
      }
 
    if (argparser_keyword_present(pp, "-badIn"))
      { o->badIn = argparser_get_next_int(pp, 0, UINT32_MAX); }
    else
      { /* No undef input value: */
        o->badIn = PNM_NO_BADVAL;
      }
 
    o->keepBad = argparser_keyword_present(pp, "-keepBad");

    if (argparser_keyword_present(pp, "-badOut"))
      { o->badOut = argparser_get_next_int(pp, 0, UINT32_MAX); }
    else
      { /* No undef output value: */
        o->badOut = PNM_NO_BADVAL;
      }
 
    if (argparser_keyword_present(pp, "-isMaskIn"))
      { o->isMaskIn = argparser_get_next_bool(pp); }
    else
      { /* No undef input value: */
        o->isMaskIn = FALSE;
      }
 
    if (argparser_keyword_present(pp, "-isMaskOut"))
      { o->isMaskOut = argparser_get_next_bool(pp); }
    else
      { /* No undef input value: */
        o->isMaskOut = o->isMaskIn;
      }
 
    o->replicate = argparser_keyword_present(pp, "-replicate");
    
    o->verbose = argparser_keyword_present(pp, "-verbose");
    
   /* Skip to positional arguments: */
    argparser_skip_parsed(pp);
    
    /* Get optional input file name: */
    if (argparser_next(pp) != NULL)
      { o->iname = argparser_get_next(pp); }
    else
      { o->iname = "-"; }
 
    /* Get optional output file name: */
    if (argparser_next(pp) != NULL)
      { o->oname = argparser_get_next(pp); }
    else
      { o->oname = "-"; }
    
    /* Check for spurious args: */
    argparser_finish(pp);

    return o;
  }
