#ifndef frb_candidate_H
#define frb_candidate_H

/* A `candidate' is a pair of segments from two different curves */
/* Last edited on 2006-02-28 11:49:37 by stolfi */

#include <frb_segment.h>
#include <frb_match.h>
#include <frb_signal.h>
#include <frb_types.h>

typedef
  struct frb_candidate_t
    { frb_segment_t seg[2];    /* Segments being matched. */
      double mismatch;         /* Mismatch function, or 0 if not available. */
      double length;           /* Mean length of the two segments. */
      double matchedLength;    /* Length effectively matched. */
      frb_match_packed_t *pm;  /* Matching for this candidate, or {NULL} */
    } frb_candidate_t;
    /* A {frb_candidate_t} consists of two segments of curve,
      and an optional matching between them. */

/* CANDIDATE LISTS */

vec_typedef(frb_candidate_vec_t,frb_candidate_vec,frb_candidate_t);
  /* A vector of candidate matches ({frb_candidate_t}s). */

typedef 
  struct frb_candidate_read_data_t
    { char *cmt;
      double lambda;
      frb_candidate_vec_t c;
    } frb_candidate_read_data_t;

#define frb_candidate_empty \
  (frb_candidate_t) \
    { /* seg */     { frb_segment_empty, frb_segment_empty }, \
      /* mismatch */ 0.0, \
      /* length */  0.0, \
      /* matchedLength */ 0.0,  \
      /* pm */ NULL \
    }
  /* An `empty' candidate (NOT the only one!).  May be handy. */

bool_t frb_candidate_is_empty(frb_candidate_t *c);
  /* TRUE if the candidate is empty (either segment has zero samples). */

frb_candidate_t frb_candidate_expand(frb_candidate_t *c, int *iniEx, int *finEx);
  /* Expands segment {i} of candidate {c}, for {i = 0,1}, by
    {iniEx[i]} steps at the beginning and {finEx[i]} steps at
    the end, preserving its matching (if any). If either argument is
    negative, removes that many steps from the respective end, and
    trims the matching {c.pm} accordingly. */

void frb_candidate_write
  ( FILE *wr,
    char *cmt,
    frb_candidate_vec_t *c,
    double lambda
  );
  /* Writes the list of candidates {c} and the parameter {lambda}
    to the writer {wr}. */

frb_candidate_read_data_t frb_candidate_read (FILE *rd);
  /* Reads from {rd} a list of candidates that was written by {Write}. */

frb_candidate_read_data_t frb_candidate_read_old
  ( FILE *rd, 
    int_vec_t *m, 
    bool_vec_t rev
  );
  /* Reads from {rd} a list of candidates that was written by the old
    version of {Write}. The {s.tot} field of each segment {s} is taken
    from the array {m}, indexed by the corresponding curve number;
    and the {s.rev} flag is set to {rev[i]}, where {i} is the
    candidate's side. */

bool_vec_t frb_candidate_curves_used (frb_candidate_vec_t *cand);
  /* Returns a vector {used} such that {used[k]} is TRUE
    iff curve {k} occurs in some candidate of {cand[i]}. */

void frb_candidate_print(FILE *wr, frb_candidate_t *cand, bool_t pairing);
  /* Prints the data of candidate {cand} to {wr}, in a multiline
    readable format.  If {pairing} is TRUE, prints also the sample 
    pairing. */



#endif
