#! /usr/bin/gawk -f
# Last edited on 2015-11-17 17:44:59 by stolfilocal

BEGIN \
  { 
    # Writes an STL description of a self-intersecting semi-everted torus,
    # modeled by a 2D array of triangles with hex topology. 
    
    # The surface is rotationally symmetric about the {Z}-axis and
    # mirror-symmetric about the {XY} plane. Its intersection with a meridional half-plane
    # (a vertical half-plane bounded by the {Z} axis)
    # is a figure of eight on that plane.  
    
    pi = 3.141592653589793238462;
    
    # User must define (with "-v") the number {N} of parallel bands of triangles.
    # and the number {M} of meridian bands.
    if (N =="") { printf "** must define {N}\n" > "/dev/stderr"; exit(1); }
    N = N+0;
    if (N < 6) { printf "** invalid {N}\n" > "/dev/stderr"; exit(1); }
    if ((N % 6) != 0) { printf "** {N} must be multiple of 6\n" > "/dev/stderr"; exit(1); }
    if (M =="") { printf "** must define {M}\n" > "/dev/stderr"; exit(1); }
    if (M < 4) { printf "** invalid {M}\n" > "/dev/stderr"; exit(1); }
    
    rad = 2.718281828;   # Radius at {Z=0} (self-intersection curve).
    hH = 3.1415925;      # Half-height of surface at distance {rad} from {Z}-axis.

    printf "mesh will have %d bands and %d sectors (total %d triangles)", N, M, 2*N*M > "/dev/stderr"; 
    
    # The surface consists of {N} "longitudinal bands" of triangles. Each band 
    # is limited by two "parallels" (horizontal circles on the surface, 
    # parallel to the midline circle).  The parallels are
    # numbered from 0 to {N}, where parallel {N} coincides with parallel 0.
    # Each band contains {2*M} triangles. Each triangle in a band has two vertices
    # on one parallel and the opposite vertex on the other parallel. The triangles 
    # alternate in orientation.
    
    # The triangles can also be seen as {M} "meridional bands" of triangles.
    # Each meridional band has {2*N} triangles, forming a zigzag band
    # that traces a figure of eight.
    
    printf "solid\n";
    
    # Points on generating eight-curve. 
    # Indexed {[k,c]}, where {k} is the point index in {0..N}, {c} is {0,1} for {R,Z}. 
    split("", pp);
    compute_parallels(M,N,rad,hH,pp);
    
    # We generate {M} transversal bands.
    for (i = 0; i < M; i++)
      { 
        # Output the triangles:
        for (j = 0; j < N; j++)
          { # Find the indices {j0,j1} of the two parallels, {j0} being base of first triangle:
            if ((j%2) == 0)
              { j0 = j; j1 = j+1; }
            else
              { j0 = j+1; j1 = j; }
            # Write the two triangles between parallels {j0} and {j1}
            write_triangle(M, i-0.5,pp[j0,0],pp[j0,1], i+0.0,pp[j1,0],pp[j1,1], i+0.5,pp[j0,0],pp[j0,1]);
            write_triangle(M, i+0.0,pp[j1,0],pp[j1,1], i+0.5,pp[j0,0],pp[j0,1], i+1.0,pp[j1,0],pp[j1,1]);
          }
      }
     
    printf "\n";
    printf "endsolid\n";
    
  }
        
function compute_parallels \
  ( M,N,rad,hH,p, \
    t,j,u,rj,zj,wt,rt,zt,Q,C,E,T,S \
  ) 
  {
    # Stores into {p[0..N,0..2]} the 3D coordinates of the {N} parallels
    # of the generator (a figure of eight curve).
    
    # Global variables used: {pi}

    # Parameters for the morphing projective map of the unit circle to itself:
    Q = hH/rad;
    C = (Q*Q - 1)/(Q*Q + 1);
    E = sqrt(1 - C*C);
    
    # Parameters of Euclidean mapping of morphed circle to the final circle:
    T = 1;       # Radial translation.
    S = hH/E;    # Final scaling.
    
    for (j = 0; j <= N; j++)
      { # Compute equally spaced {r,z} points on the prototypical eight-curve.
        # It is centered at the origin, and osculates the unit circle at top 
        # and bottom:
        t = 2*pi*(j + 0.5)/N
        rj = sqrt(0.5)*sin(2*t);
        zj = sin(t);
        
        printf "  %3d ( %+7.4f %+7.4f )", j, rj, zj > "/dev/stderr";

        # Apply a projective map of the {r,z} plane
        # that maps the the unit circle to a circle centered on the {Z=0} axis,
        # maps origin {(0,0)} to {(rad,0)}, and maps {(0,1)} to {(rad,hH)}:
        
        wt = 1 - C*rj;
        rt = S*((rj - C)/wt + T);
        zt = S*(E*zj/wt);
        
        p[j,0] = rt;
        p[j,1] = zt;
        
        printf " --> ( %+7.4f %+7.4f )\n", p[j,0], p[j,1] > "/dev/stderr";
      }
  }

function write_triangle \
  ( M, ia,ra,za, ib,rb,zb, ic,rc,zc, \
    ta,xa,ya, tb,xb,yb, tc,xc,yc, xu,yu,zu, xv,yv,zv, xn,yn,zn, dn \
  )        
  {
    # Writes a triangle with vertices on the meridians {ia,ib,ic} 
    # (integers or half-integers), out of {M} meridians. The radial coordinates on those
    # meridional half-planes are {ra,rb,rc}, and the corresponding
    # {Z} coordinates are {za,zb,zc}.
    
    # Global variables used: {pi}

    # Convert from {i,r,z} to {x,y,z}:
    ta = ia*2*pi/M
    xa = ra*cos(ta);
    ya = ra*sin(ta);
    
    tb = ib*2*pi/M
    xb = rb*cos(tb);
    yb = rb*sin(tb);
    
    tc = ic*2*pi/M
    xc = rc*cos(tc);
    yc = rc*sin(tc);
    
    # xa = ra;
    # ya = ta;
    # 
    # xb = rb;
    # yb = tb;
    # 
    # xc = rc;
    # yc = tc;
    
    # Compute normal direction: 
    xu = xc - xb; yu = yc - yb; zu = zc - zb;
    xv = xa - xb; yv = ya - yb; zv = za - zb;
    xn = zu*yv - zv*yu;
    yn = xu*zv - xv*zu;
    zn = yu*xv - yv*xu;
    dn = sqrt(xn*xn + yn*yn + zn*zn);
    if (dn <= 1.0e-8) 
      { xn = 0; yn = 0; zn = 0; }
    else
      { xn /= dn; yn /= dn; zn /= dn; }
    
    printf "\n";
    printf "facet\n"
    printf "normal %+.6f %+.6f %+.6f\n", xn, yn, zn;
    printf "outer loop\n";
    printf "  vertex %10.5f %10.5f %10.5f\n", xa, ya, za;
    printf "  vertex %10.5f %10.5f %10.5f\n", xb, yb, zb;
    printf "  vertex %10.5f %10.5f %10.5f\n", xc, yc, zc;
    printf "endloop\n";
    printf "endfacet\n";
  }
    
    
