#! /usr/bin/gawk -f
# Last edited on 2015-11-17 23:12:27 by stolfilocal

BEGIN \
  { 
    # Writes an STL description of a toroidal helix knot
    # with {K} twists and {L} apparent strands, 
    # modeled by a 2D array of triangles with hex topology. 
    
    # The intersection of the surface with a meridional half-plane
    # (a vertical half-plane bounded by the {Z} axis)
    # is a set of {L} ellipses arranged symmetrically around a center.
    # As the meridional plane makes {L} full turns
    # around the {Z}-axis, that curve makes {K} turns around its center.
    # If {L} and {K} are relatively prime, there will be one actual strand only;
    # in general there will be {gcd(L,K)} strands.
    
    pi = 3.141592653589793238462;
    
    # User must define (with "-v") the number {N} of parallel bands of triangles.
    # and the two orders {K} and {L}, where {L} must be positive.
    # Optionally, also define the strand radius {radS}.
    
    if (N == "") { printf "** must define {N}\n" > "/dev/stderr"; exit(1); }
    N += 0;
    if (N < 3) { printf "** invalid {N}\n" > "/dev/stderr"; exit(1); }

    if (K == "") { printf "** must define {K}\n" > "/dev/stderr"; exit(1); }
    K += 0;
    if (K < 0) { printf "** invalid {K}\n" > "/dev/stderr"; exit(1); }

    if (L == "") { printf "** must define {L}\n" > "/dev/stderr"; exit(1); }
    L += 0;
    if (L < 1) { printf "** invalid {L}\n" > "/dev/stderr"; exit(1); }

    printf "there will be %d apparent strands making %d/%d twists every turns\n", L, K, L > "/dev/stderr"; 
    printf "each strand will make %d twists every %d turns\n", K, L > "/dev/stderr"; 

    radM = 4.18;  # Radius of midline.
    radB = 2.15;  # Distance of midline of bundle to midline of strands.
    
    if (radS != "") 
      { radS += 0.0; 
        if (radS <= 0) { printf "** invalid {L}\n" > "/dev/stderr"; exit(1); }
      }
    else
      { # Compute the max tilt {angS} and radius {radS} of the strands:
        # Assume at least 6 strands so that they don't get too thick.
        Lref = (L < 4 ? 4 : L);
        angS = atan2(K*radB, Lref*(radM-radB));
        radS = 0.75*radB*cos(angS)*sin(pi/Lref);
      }
    printf "radius of midline = %.4f  of bundle = %.4f  of strand = %.4f\n", radM, radB, radS > "/dev/stderr";

    # Choose the number {M} of meridional bands so triangles are almost equilateral: 
    len = 2*pi*sqrt(K*K*radB*radB + L*L*radM*radM)/L; # Approx total length of 1 strand over 1 turn.
    cir = 2*pi*radS; # Circumference of each strand.
    M = int(N*len/cir*sqrt(0.75));
    printf "each apparent strand will have %d bands and %d sectors (total %d triangles)\n", N, M, 2*N*M*L > "/dev/stderr"; 
    
    # Any meridional half-plane (vertical, bounded by the {Z}-axis) cuts the 
    # knot in {L} separate strand sections. These {L} apparent strands 
    # twist by {K/L} turns around the midline while the meridional plane
    # makes one turn around the {Z}-axis.  
    
    # Each strand of the knot consists of {N} "longitudinal bands" of triangles. Each band 
    # is limited by two "parallels" roughly parallel to the midline of the strand.  The parallels are
    # numbered from 0 to {N}, where parallel {N} coincides with parallel 0 and is the point of the 
    # strand farther away from the bundle midline. Each triangle in a band has two vertices
    # on one parallel and the opposite vertex on the other parallel. The triangles 
    # alternate in orientation.
    
    # Each band contains {2*M} triangles before connecting to the same band of the same 
    # or another strand after one full turn of the meridian halfplane.
    
    # The triangles of the mesh can also be seen as {M} "meridional sets",
    # each containing {2*N*L} triangles forming {L} separate zigzag closed bands of 
    # {2*N} triangles each.
    
    printf "solid\n";
    
    # Nominal intersections of strand parallels at various longitudes. 
    # Indexed {[s,k,c]}, where {s} is the strand index in {0..L-1},
    # {k} is the parallel index in {0..N}, {c} is {0,1,2} for {X,Y,Z}. 
    split("", pa); # Parallel points at longitude step {i-0.5}.
    split("", pb); # Parallel points at longitude step {i}.
    split("", pc); # Parallel points at longitude step {i+0.5}.
    split("", pd); # Parallel points at longitude step {i+1}.
    
    # We generate {M} transversal cuts:
    # The transverse band index {i} starts at {-1} to compute {pa,pb} computed for band 0.
    for (i = -1; i < M; i++)
      { 
        # Compute the endpoints of the generating segment:
        compute_parallels(i+0.5, K,L,M,N,radM,radB,radS, pc);
        compute_parallels(i+1.0, K,L,M,N,radM,radB,radS, pd);
        if (i >= 0)
          { # Arrays {pa,pb} were set on previous iteration.
            # Output the triangles:
            for (s = 0; s < L; s++)
              { for (j = 0; j < N; j++)
                  { # Find the indices {j0,j1} of the two parallels, {j0} being base of first triangle:
                    if ((j%2) == 0)
                      { j0 = j; j1 = j+1; }
                    else
                      { j0 = j+1; j1 = j; }
                    # Write the two triangles between parallels {j0} and {j1}
                    write_triangle( \
                        pa[s,j0,0],pa[s,j0,1],pa[s,j0,2], \
                        pb[s,j1,0],pb[s,j1,1],pb[s,j1,2], \
                        pc[s,j0,0],pc[s,j0,1],pc[s,j0,2] \
                      );
                    write_triangle( \
                        pb[s,j1,0],pb[s,j1,1],pb[s,j1,2], \
                        pc[s,j0,0],pc[s,j0,1],pc[s,j0,2], \
                        pd[s,j1,0],pd[s,j1,1],pd[s,j1,2] \
                      );
                  }
              }
          }
        # Prepare for the next transversal band:
        assign_parallels(L,N, pa, pc);
        assign_parallels(L,N, pb, pd);
      }
     
    printf "\n";
    printf "endsolid\n";
    
  }
        
function compute_parallels \
  ( i,K,L,M,N,radM,radB,radS,p, \
    fi,ti,tsi,vlon,vtan,ob,a,s,j,u,rj,fj,zj,ro,fo,zo,rm,zm,wt,rt,zt \
  ) 
  {
    # Stores into {p[0..L-1,0..N,0..2]} the 3D coordinates of the {N} parallels
    # of the generator (a figure of eight curve) for longitude {i*2*pi/M)}.
    
    # Global variables used: {pi}

    # Compute the longitude {fi} and the accumulated twist {ti} of the bunndle:
    fi = 2*pi*i/M;
    ti = 2*pi*i*K/L/M;

    for (s = 0; s < L; s++)
      { # Angular position of strand {s} in bundle at this longitude:
        tsi = 2*pi*s/L + ti;
    
        # Compute the obliquity of the strand relative to the bundle's parallel:
        vlon = radM + radB*cos(tsi);
        vtan = radB*K/L
        ob = atan2(vtan, vlon);

        for (j = 0; j <= N; j++)
          { # latitude on strand rel to outermost point:
            a = 2*pi*j/N;
            
            # Point on circle on {R,Z} plane ({fj} is zero).
            rj = radS*cos(a);
            zj = radS*sin(a);
            
            if ((i == 0) || (i == 0.5*M))
              { printf "  %3d ( %+7.4f %+7.4f )", j, rj, zj > "/dev/stderr"; }

            # Tilt circle around the {R} axis by {ob}:
            ro = rj;
            fo = -zj*sin(ob);
            zo = zj*cos(ob);
            
            # Displace by {radB} along {R} and rotate to strand position on {R,Z} plane:
            ro = ro + radB;
            rm = ro*cos(tsi) - zo*sin(tsi);
            fm = fo;
            zm = ro*sin(tsi) + zo*cos(tsi);
            
            # Displace by {radM} along {R} and convert to Cartesian coordinates:
            rm = rm + radM;
            p[s,j,0] = rm*cos(fi+fm/rm);
            p[s,j,1] = rm*sin(fi+fm/rm);
            p[s,j,2] = zm;

            if ((i == 0) || (i == 0.5*M))
              { printf " --> ( %+7.4f %+7.4f %+7.4f )\n", p[s,j,0], p[s,j,1], p[s,j,2] > "/dev/stderr"; }
          }
      }
  }

function assign_parallels(L,N, dst,src,  s,j,c)
  { 
    # Sets {dst[j,c] = src[j,c]} for {j} in {0..N} and {c} in {0..2}.
    
    for (s = 0; s < L; s++)
      { for (j = 0; j <= N; j++)
          { for (c = 0; c < 3; c ++)
              { dst[s,j,c] = src[s,j,c]; }
          }
      }
  }

function write_triangle \
  ( xa,ya,za, xb,yb,zb, xc,yc,zc, \
    xu,yu,zu, xv,yv,zv, xn,yn,zn, dn \
  )        
  {
    # Compute normal direction: 
    xu = xc - xb; yu = yc - yb; zu = zc - zb;
    xv = xa - xb; yv = ya - yb; zv = za - zb;
    xn = zu*yv - zv*yu;
    yn = xu*zv - xv*zu;
    zn = yu*xv - yv*xu;
    dn = sqrt(xn*xn + yn*yn + zn*zn);
    if (dn <= 1.0e-8) 
      { xn = 0; yn = 0; zn = 0; }
    else
      { xn /= dn; yn /= dn; zn /= dn; }
    
    printf "\n";
    printf "facet\n"
    printf "normal %+.6f %+.6f %+.6f\n", xn, yn, zn;
    printf "outer loop\n";
    printf "  vertex %10.5f %10.5f %10.5f\n", xa, ya, za;
    printf "  vertex %10.5f %10.5f %10.5f\n", xb, yb, zb;
    printf "  vertex %10.5f %10.5f %10.5f\n", xc, yc, zc;
    printf "endloop\n";
    printf "endfacet\n";
  }
    
