#! /usr/bin/gawk -f
# Last edited on 2015-11-17 17:45:47 by stolfilocal

BEGIN \
  { 
    # Writes an STL description of a Moebius strip,
    # modeled by a 2D array of triangles with hex topology. 
    
    # The strip midline is a circle on the {Z=0} plane.  The intersection 
    # of the strip with a meridional half-plane 
    # (a vertical half-plane bounded by the {Z} axis)
    # is a straight segment on that plane, that turns at constant speed.
    
    pi = 3.141592653589793238462;
    
    # User must define (with "-v") the number {N} of triangle layers.
    if (N =="") { printf "** must define {N}\n" > "/dev/stderr"; exit(1); }
    N = N+0;
    if (N <= 0) { printf "** invalid {N}\n" > "/dev/stderr"; exit(1); }
    if ((N % 2) != 0) { printf "** {N} must be even\n" > "/dev/stderr"; exit(1); }
    
    rad = 2.718281828;   # Radius of midline.
    hW = rad/3.0;        # Half-width of strip.
    dW = 2*hW/N;         # Width of each triangle band when tape is vertical.
    dL = dW/(sqrt(3)/2); # Approx side of each triangle when tape is vertical.
    
    # Choose the number {M} of meridional bands so triangles are almost equilateral: 
    L = 2*pi*rad; # Length of midline circle.
    M = int(L/dL);
    printf "mesh will have %d bands and %d sectors (total %d triangles)", N, M, 2*N*M > "/dev/stderr"; 
    
    # The strip consists of {N} "longitudinal bands" of triangles. Each band 
    # is limited by two "parallels" (polygonal lines on the strip, roughly
    # parallel to the midline circle).  At each longitude, the parallels are
    # numbered from 0 to {N}.  Each band contains {2*M} triangles.
    # Each triangle in a band has one side on one parallel and the opposite vertex 
    # on the other parallel. The triangles alternate in orientation.
    
    # The triangles can also be seen as {M} "transversal bands" of triangles.
    # Each transversal band has {2*N} triangles, forming a zigzag band
    # that goes from edge to edge of the strip.
    
    printf "solid\n";
    
    # Points where the parallels cross the generating segments: 
    # Indexed {[k,c]}, where {k} is the point index in {0..N}, {c} is {0,1,2} for {X,Y,Z}. 
    split("", pa); # Parallel points on generating segment at {i-0.5}.
    split("", pb); # Parallel points on generating segment at {i}.
    split("", pc); # Parallel points on generating segment at {i+0.5}.
    split("", pd); # Parallel points on generating segment at {i+1}.
    
    # We generate {M} transversal bands.
    # The transverse band index {i} starts at {-1} to compute {pa,pb} computed for band 0.
    for (i = -1; i < M; i++)
      { 
        # Compute the endpoints of the generating segment:
        compute_parallels(i+0.5, pc);
        compute_parallels(i+1.0, pd);
        if (i >= 0)
          { # Arrays {pa,pb} were set on previous iteration.
            # Output the triangles:
            for (j = 0; j < N; j++)
              { # Find the indices {j0,j1} of the two parallels, {j0} being base of first triangle:
                if ((j%2) == 0)
                  { j0 = j; j1 = j+1; }
                else
                  { j0 = j+1; j1 = j; }
                # Write the two triangles between parallels {j0} and {j1}
                write_triangle(pa[j0,0],pa[j0,1],pa[j0,2], pb[j1,0],pb[j1,1],pb[j1,2], pc[j0,0],pc[j0,1],pc[j0,2]);
                write_triangle(pb[j1,0],pb[j1,1],pb[j1,2], pc[j0,0],pc[j0,1],pc[j0,2], pd[j1,0],pd[j1,1],pd[j1,2]);
              }
          }
        # Prepare for the next transversal band:
        assign_parallels(pa, pc);
        assign_parallels(pb, pd);
      }
     
    printf "\n";
    printf "endsolid\n";
    
  }
        
function compute_parallels \
  ( i,p, \
    t,f,j,u,rj,zj,wt,rt,zt,Q,C,E,T,S \
  ) 
  {
    # Stores into {p[0..N,0..2]} the 3D coordinates of the {N} parallels
    # for longitude {i*2*pi/M}.
    
    # The coordinates are such that increasing {i} by {M}
    # produces the same points but in reverse order.
    
    # Global variables used: {M,N,pi,hW,rad}

    # Parameters for the morphing projective map of the unit circle to itself:
    Q = hW/rad;
    E = 1.0/sqrt(1 + Q*Q);
    C = Q*E;
    
    # Parameters of morphed circle to the final circle:
    T = 1/C;   # Radial translation.
    S = hW/E;  # Final scaling.
    
    # Compute the longitude {f} and the tilt {t} of the generating segment:
    f = i*2*pi/M;
    t = i*pi/M;
    
    for (j = 0; j <= N; j++)
      { # Compute equally spaced {r,z} points on the segment of length 2 centered at the origin:
        u = 2*(j + 0.0)/N - 1;
        rj = +u*sin(t);
        zj = -u*cos(t);

        # Apply a projective map of the {r,z} plane
        # that maps the origin to {(rad,0)} 
        # and the circle with center {(0,0)} and radius 1
        # to a circle, so that the band is {2*hW}
        # wide for {f=0}:
        
        wt = 1 - C*rj;
        rt = S*((rj - C)/wt + T);
        zt = S*(E*zj/wt);
        
        p[j,0] = rt*cos(f);
        p[j,1] = rt*sin(f);
        p[j,2] = zt;
        
        if (i == 0) 
          { printf " %3d ( %+7.4f %+7.4f %+7.4f )\n", j, p[j,0], p[j,1], p[j,2] > "/dev/stderr"; }
      }
  }

function assign_parallels(dst,src,  j,c)
  { 
    # Sets {dst[j,c] = src[j,c]} for {j} in {0..N} and {c} in {0..2}.
    
    for (j=0; j <= N; j++)
      { for (c = 0; c < 3; c ++)
          { dst[j,c] = src[j,c]; }
      }
  }

function write_triangle \
  ( xa,ya,za, xb,yb,zb, xc,yc,zc, \
    xu,yu,zu, xv,yv,zv, xn,yn,zn, dn \
  )        
  {
    # Compute normal direction: 
    xu = xc - xb; yu = yc - yb; zu = zc - zb;
    xv = xa - xb; yv = ya - yb; zv = za - zb;
    xn = zu*yv - zv*yu;
    yn = xu*zv - xv*zu;
    zn = yu*xv - yv*xu;
    dn = sqrt(xn*xn + yn*yn + zn*zn);
    if (dn <= 1.0e-8) 
      { xn = 0; yn = 0; zn = 0; }
    else
      { xn /= dn; yn /= dn; zn /= dn; }
    
    printf "\n";
    printf "facet\n"
    printf "normal %+.6f %+.6f %+.6f\n", xn, yn, zn;
    printf "outer loop\n";
    printf "  vertex %10.5f %10.5f %10.5f\n", xa, ya, za;
    printf "  vertex %10.5f %10.5f %10.5f\n", xb, yb, zb;
    printf "  vertex %10.5f %10.5f %10.5f\n", xc, yc, zc;
    printf "endloop\n";
    printf "endfacet\n";
  }
    
    
