#! /usr/bin/gawk -f
# Last edited on 2015-10-03 23:23:39 by stolfilocal

BEGIN \
  { 
    # Writes an STL description of a sphere 
    # centered at the origin optionally hollow.
    # The sphere is modeled as an icosahedron with
    # triangles divided into 4, recursively, {N} times.
    
    # User must define (with "-v") the depth {N} of recurence.  
    # Each sphere will have {20*4^N} triangles.
    if (N =="") { printf "** must define {N}\n" > "/dev/stderr"; exit(1); }
    
    Ro = 3.14; # Outer radius.
    Ri = 2.72; # Inner radius.
    
    pi = 3.141592653589793238462;
    
    printf "solid\n";
    
    write_sphere_shell(N, Ro);
    write_sphere_shell(N, Ri);
    
    printf "\n";
    printf "endsolid\n";
    
  }
        
function write_sphere_shell(N,R,  a,b,c,F,r,s,t)
  {
    F = (sqrt(5) - 1)/2; # Golden ratio.
    split("", a);
    split("", b);
    split("", c);
    
    for (r = -1; r <= +1; r = r + 2)
      { for (s = -1; s <= +1; s = s + 2)
          { set_unit_vector(a, +F, s, 0); 
            set_unit_vector(b, -F, s, 0); 
            set_unit_vector(c,  0, s*F, r);
            write_axis_permuted_triangles(N,R,a,b,c);
            for (t = -1; t <= +1; t = t + 2)
              { set_unit_vector(a, r*F,   s,   0); 
                set_unit_vector(b,   0, s*F,   t); 
                set_unit_vector(c,   r,   0, t*F);
                write_subdivided_triangle(N,R,a,b,c);
              }
          }
      }
  }
  
function set_unit_vector(v,x,y,z,  m)
  { # Sets {v[0..2]} to {(x,y,z)} normalized to unit length.
    m = sqrt(x*x + y*y + z*z);
    v[0] = x/m;
    v[1] = y/m;
    v[2] = z/m;
  }

function write_axis_permuted_triangles(N,R,a,b,c,  k,x)
  { 
    for (k = 0; k < 3; k++)
      { write_subdivided_triangle(N,R,a,b,c);
        permute_axes(a);
        permute_axes(b);
        permute_axes(c);
      }
  }

function permute_axes(v,   t)
  { # Cyclic permutation of coordinates {v[0..2]}.
    t = v[0];
    v[0] = v[1];
    v[1] = v[2];
    v[2] = t;
  }

function write_subdivided_triangle(N,R,a,b,c, p,q,r)
  { if (N == 0)
      { write_spherical_triangle(R,a,b,c); }
    else
      { split("", p);
        split("", q);
        split("", r);
        set_unit_vector(p, (a[0]+b[0])/2, (a[1]+b[1])/2, (a[2]+b[2])/2);
        set_unit_vector(q, (b[0]+c[0])/2, (b[1]+c[1])/2, (b[2]+c[2])/2);
        set_unit_vector(r, (c[0]+a[0])/2, (c[1]+a[1])/2, (c[2]+a[2])/2);
        write_subdivided_triangle(N-1,R,a,r,p);
        write_subdivided_triangle(N-1,R,b,p,q);
        write_subdivided_triangle(N-1,R,c,q,r);
        write_subdivided_triangle(N-1,R,p,q,r);
      }
  }
    
function write_spherical_triangle(R,a,b,c)
  { 
    write_triangle(R*a[0],R*a[1],R*a[2], R*b[0],R*b[1],R*b[2], R*c[0],R*c[1],R*c[2]);
  }

function write_triangle \
  ( xa,ya,za, xb,yb,zb, xc,yc,zc, \
    xu,yu,zu, xv,yv,zv, xn,yn,zn, dn \
  )        
  {
    # Compute normal direction: 
    xu = xc - xb; yu = yc - yb; zu = zc - zb;
    xv = xa - xb; yv = ya - yb; zv = za - zb;
    xn = zu*yv - zv*yu;
    yn = xu*zv - xv*zu;
    zn = yu*xv - yv*xu;
    dn = sqrt(xn*xn + yn*yn + zn*zn);
    if (dn <= 1.0e-8) 
      { xn = 0; yn = 0; zn = 0; }
    else
      { xn /= dn; yn /= dn; zn /= dn; }
    
    printf "\n";
    printf "facet\n"
    printf "normal %+.6f %+.6f %+.6f\n", xn, yn, zn;
    printf "outer loop\n";
    printf "  vertex %10.5f %10.5f %10.5f\n", xa, ya, za;
    printf "  vertex %10.5f %10.5f %10.5f\n", xb, yb, zb;
    printf "  vertex %10.5f %10.5f %10.5f\n", xc, yc, zc;
    printf "endloop\n";
    printf "endfacet\n";
  }
    
    
