#ifndef MWPLOT_H
#define MWPLOT_H

/* Last edited on 2007-01-08 02:35:27 by stolfi */ 

/* Plots fields and meshes (using Gnuplot "splot"). */

typedef struct mwplot_t mwplot_t;
  /* A {mwplot_t} object contains a connection to an active
    {gnuplot} process, which may plot data on the screen or
    write it to a file. */

mwplot_t *mwplot_start_gnuplot(int debug);
  /* Starts a {gnuplot} process and returns a handle to it. 
  
    If {debug} is nonzero, the {gnuplot} is not started, and all plot
    commands will be written to a file called "mwplot.plt". */

void mwplot_stop_gnuplot(mwplot_t *wp);
  /* Kills the {gnuplot} process attached to {wp}, and reclaims
    its storage. */

/* PLOTTING OPTIONS
  
  An {mwplot_t} object also contains several plot parameters, such as
  terminal type, plot styles, data ranges, etc.. The client can use
  the various {mwplot_set_XXX} functions below to change those
  parameters.  Those changes will affect all subsequent {mwplot_plot_XXX}
  calls, until overriden by other {mwplot_set_XXX} calls. */

void mwplot_set_terminal(mwplot_t *wp, int terminalKind);
  /* This function sets gnuplot's ``terminal'' (output device type),
    according to {terminalKind}: 0 -> x11, 10 -> eps grayscale, 
    11 -> eps color, 101 -> png grayscale, 111 -> png color. */

void mwplot_set_palette(mwplot_t *wp, int paletteCode);
  /* This function sets the color palette for certain {plot} commands,
    as specified by the {paletteCode}: 10 -> yellow--red, 15 ->
    red--yellow, 20 -> blue--red. 
    
    The special {paletteCode} 100 chooses a palette 
    suitable for showing 2D data as a grid of points
    (see {mwplot_set_splot_options}, style 100). */
 
void mwplot_set_bivariate_plot_style(mwplot_t *wp, int splotStyle);
  /* This function sets the {gnuplot} options that are relevant
    for plotting 2D data (matrices or bivariate functions),
    as specified by {splotStyle}: 
    
       0 -> graph as a surface in 3D, with hidden parts removed.
      10 -> graph as a point cloud in 3D.
      20 -> graph as a wire mesh 3D.
    
    The special {splotStyle} 100 causes 2D data to be shown as a 2D
    array of colored dots. */
   
void mwplot_set_axis_labels(mwplot_t *wp, char *xLabel, char *yLabel, char *zLabel);
  /* This function specifies the labels for the X, Y, and Z axes
    of the plot. If any of them is NULL, the corresponding axis
    is omitted. */
    
void mwplot_set_ranges
  ( mwplot_t *wp, 
    double xMin, double xMax, 
    double yMin, double yMax,
    double zMin, double zMax
  );
  /* This function sets the data ranges, that wil define the plot
    scales for subsequent plot commands.
    
    If any of these parameters is NaN, the corresponding limit will be
    left uspecified, so that {gnuplot} will choose it based on the
    data values to be provided. */

/* PLOTTING COMMANDS

  The commands {mwplot_plot_XXX} functions below will 
  sed the data to {gnuplot}, and cause it to generate
  the desired plot (displayed or written to disk). */

void mwplot_plot_array
  ( mwplot_t *wp,            
    char *fileName,  
    int nx,        /* Number of rows of {A} */
    int ny,        /* Number of cols of {A} */
    double dx,     /* Sampling step in X. */
    double dy,     /* Sampling step in Y. */
    double **A,    /* {A[ix][iy]} is the field value at {(ix*dx,iy*dy)}. */
    char* title    /* Name of the function being plotted. */
  );
  /* This function plots the elements of array {A} as a 2D function.
    The output device and plot style options should have been set
    previosly with the {mwplot_set_XXX} functions.
    
    The array {A} is assumed to have {nx} rows, each with {ny}
    elements {A[ix][iy]}. The first index {ix} increases along X, the
    second {iy} increases along Y. The parameters {dx} and {dy} define
    the spacing of the elements along each axis.
    
    The {fileName} is used only if the current terminal type
    requires it.  If the current terminal is a window
    (e.g., type "x11"), the {fileName} is ignored. */
 
#endif
