/* See {R3_IncrementalSlicing.h} */

using namespace std;

#include <math.h>
#include <stdlib.h>
#include <assert.h>

#include "R3.h"
#include "R2.h"
#include "R3_Mesh.h"
#include "R3_Slicer.h"
#include "R3_IncrementalSlicing.h"

int R3_IncrementalSlicing_binary_search (float zMin, vector<float> P);
/* Assumes that {P} is a list of {k} strictly increasing {Z} coordinates. Returns an integer 
  {p} such that {P[p-1] < zMin < P[p]}.  As special cases, if {zMin < P[0]} returns
  0; if {zMin > P[k-1]} returns {k}. */


R3_Mesh_Triangle_List_t* R3_IncrementalSlicing_buildLists (bool srt, double delta, R3_Mesh_t *mesh, vector<float> P);
/* Assumes that {P[0..k-1]} is a list of {k} strictly increasing {Z} coordinates. Returns a 
  vector of {k+1} lists {L[0..k]} such that {L[p]} contains all triangles of the {mesh}
  that have {zMin} between {P[p-1]} and {P[p]}, assuming that {P[-1] = -oo} and {P[k] = +oo}.
  If {delta > 0}, assumes that {P[p]-P[p-1] = delta} for {p} in {1..k-1}. If {srt} is true, assumes
  that the triangles are already sorted by increasing {zMin}. */

int R3_IncrementalSlicing_binary_search (float zMin, vector<float> P) {
  int k = P.size();
  assert(k >= 1);
  if (zMin >= P[k-1]) { return k; }
  /* Binary search: */
  int l = -1; /* Inferior Z index. */
  int r = k; /* Superior Z index. .*/
  while (r - l > 1) {
    /* At this point, {zMin} is between {P[l]} and {P[r]}. */
    int m = (l + r)/2; 
    assert((0 <= m) && (m < k));
    if (zMin >= P[m]) {
      l = m;
    } else {
      r = m;
    }
  }
  return r;
}

R3_Mesh_Triangle_List_t* R3_IncrementalSlicing_buildLists (bool srt, double delta, R3_Mesh_t *mesh, vector<float> P) {

  int k = P.size(); /* Number of planes. */

  R3_Mesh_Triangle_List_t* L = (R3_Mesh_Triangle_List_t *)malloc((k+1)*sizeof(R3_Mesh_Triangle_List_t));
  for (size_t p = 0; p <= k; p++) { L[p] = R3_Mesh_Triangle_List_create(); }

  R3_Triangle_Vector_t *T = &(mesh->T);
  int n = T->size(); /* Number of triangles. */

  if (delta > 0.0) { 
    /* Uniform slicing - compute list index: */ 
    for (R3_Triangle_Vector_t::iterator it = T->begin(), itEnd = T->end(); it != itEnd; ++it) {
      R3_Triangle_t *t = &(*it);
      int p;
      if (t->zMin < P[0]) {
	p = 0;
      }
      else if (t->zMin > P[k-1]) {
	p = k;
      }
      else {
	p = floor((t->zMin - P[0])/delta) + 1; 
      }
      R3_Mesh_Triangle_List_insert(t, &(L[p]));
    }
  } else if (srt) {
    /* Slicing of a pre-sorted mesh - merge {zMin}s and {P}: */ 
    R3_Triangle_Vector_t::iterator it = T->begin();
    R3_Triangle_Vector_t::iterator itEnd = T->end();
    double zprev = -INFINITY;
    for (int p = 0; p <= k; p++) {
      float Zp = (p < k ? P[k] : +INFINITY);
      R3_Triangle_t *t = &(*it);
      assert(t->zMin >= zprev);
      while ((it != itEnd) && (t->zMin < Zp)) {
	R3_Mesh_Triangle_List_insert (t, &(L[p]));
	zprev = t->zMin;
	it++;
      }
    }
  } else {
    /* General case: */ 
    for (R3_Triangle_Vector_t::iterator it = T->begin(), itEnd = T->end(); it != itEnd; ++it) {
      R3_Triangle_t *t = &(*it);
      int p = R3_IncrementalSlicing_binary_search(t->zMin, P); 
      assert((p >= 0) && (p <= k));      
      R3_Mesh_Triangle_List_insert (t, &(L[p]));
    }
  }
  return L;
}

void R3_IncrementalSlicing_slice (R3_Mesh_t *mesh, vector<float> P, float delta, bool srt, R3_Closure_Proc_t *closer) {

  int k = P.size(); 
 
  vector<R2_Segment_t> segs;

  //int miss = 0;

  /* Classify triangles by the plane gaps that contain their {zMin}: */
  R3_Mesh_Triangle_List_t* L = R3_IncrementalSlicing_buildLists(srt, delta, mesh, P); 
  /* Now perform a plane sweep from bottom to top: */

  R3_Mesh_Triangle_List_t A = R3_Mesh_Triangle_List_create(); /* Active triangle list. */
  for (int p = 0; p < k; p++) {   
    /* Add triangles that start between {P[p-1]} and {P[p]}: */            
    R3_Mesh_Triangle_List_union (&A, &(L[p]));
    /* Scan the active triangles: */             
    segs.clear(); 
    R3_Mesh_Triangle_Node_t *aux = A.head;
    while (aux != NULL) {
      assert(aux->t->zMin < P[p]);
      R3_Mesh_Triangle_Node_t *next = aux->next;
      if (aux->t->zMax < P[p]) {
	/* Triangle is exhausted: */
	R3_Mesh_Triangle_List_remove(&A, aux);
      } else {
	/* Compute intersection: */
	R2_Segment_t seg = R3_Mesh_Triangle_slice(aux->t, P[p]);
	segs.push_back(seg);
      }
      aux = next;
    }
    if (closer != NULL) { 
      /* Process the segments: */
      closer(mesh, P[p], &segs);
    }
  }
  free(L);
}
