/* See {R3_Mesh.h} */

#include <cmath>
#include <fstream>
#include <string>
#include <vector>

#include <stdlib.h>
#include <assert.h>

/*
#include "glm/glm.hpp"
#include "glm/gtc/matrix_transform.hpp"
*/

#include "R3.h"
#include "R3_Mesh.h"

void R3_Triangle_fixZMinZmax(R3_Triangle_t *t);
/* Recomputes {t->zMin,t->zMax} from {t->v}. */

void R3_Triangle_setZMinZMax(R3_Triangle_t *t, float z);
/* Updates {t->zMin,t->zMax} to include the given {z} value. */

R3_Triangle_t R3_Triangle_create(R3_t normal, R3_t v0, R3_t v1, R3_t v2) {
  R3_Triangle_t t;
  t.v[0] = v0; t.v[1] = v1; t.v[2] = v2;
  t.normal = normal;
  R3_Triangle_fixZMinZmax(&t);
  return t;
}

void R3_Triangle_fixZMinZmax(R3_Triangle_t *t) {	
  t->zMin = +INFINITY; t->zMax = -INFINITY;
  R3_Triangle_setZMinZMax(t, t->v[0].z);
  R3_Triangle_setZMinZMax(t, t->v[1].z);
  R3_Triangle_setZMinZMax(t, t->v[2].z);
}

void R3_Triangle_setZMinZMax(R3_Triangle_t *t, float z) { 
  if (z < t->zMin) t->zMin = z; 
  if (z > t->zMax) t->zMax = z;
}

R3_Mesh_t R3_Mesh_create(void) {
  R3_Mesh_t mesh;
  mesh.vMin = (R3_t){ .x = +INFINITY, .y = +INFINITY, .z = +INFINITY};
  mesh.vMax = (R3_t){ .x = -INFINITY, .y = -INFINITY, .z = -INFINITY};
  mesh.meshSize = 0;
  mesh.T.clear();
  mesh.sorted = false;
  return mesh;
}

//      // move 3D Model coordinates to be center around COG(0,0,0)
//      void normalize() {
//          R3_t halfBbox = (vMax - vMin)/2.0f;
//          //R3_t start = vMin + halfBbox;
//  		//for (const R3_Triangle_t &t : mesh) {
//  			//R3_Triangle_t t1;
//  			//t1 = t - start;
//          //}
//          vMin = halfBbox*-1.0f;
//          vMax = halfBbox;
//      }
//      void push_back(R3_Triangle_t &t) {
//          meshSize++;
//          if (ordered) {
//              orderedMesh[t.zMin][t.zMax].push_back(t);
//          } else {
//              long sortIni = Timer::getTimeMs();
//              if (sort_triangles) {
//                 mesh.insert(t);
//              }
//              else {
//                 T.push_back(t);
//              } 
//              long sortEnd = Timer::getTimeMs();
//              timeSort += (sortEnd - sortIni);
//          }
//          for (size_t i = 0; i < 3; ++i) {
//              if (t.v[i].x < vMin.x) vMin.x = t.v[i].x;
//              if (t.v[i].y < vMin.y) vMin.y = t.v[i].y;
//              if (t.v[i].z < vMin.z) vMin.z = t.v[i].z;
//              if (t.v[i].x > vMax.x) vMax.x = t.v[i].x;
//              if (t.v[i].y > vMax.y) vMax.y = t.v[i].y;
//              if (t.v[i].z > vMax.z) vMax.z = t.v[i].z;
//          }
//      }
//      R3_t meshAABBSize() const {
//          return R3_t(vMax.x-vMin.x, vMax.y-vMin.y, vMax.z-vMin.z);
//      }
//      multiset<R3_Triangle_t, R3_Triangle_Comparer_t>& getMesh() { return mesh; }
//      zMinTriangleMap& getOrderedMesh() { return orderedMesh; }
//      const zMinTriangleMap& getOrderedMesh() const { return orderedMesh; }
//      const multiset<R3_Triangle_t, R3_Triangle_Comparer_t>& getMesh() const { return mesh; }
//      const vector<R3_Triangle_t>& getT() const { return T; }
//      // Mesh COG point should be at (0,0,0)
//      int transform(const glm::mat4 &mat) {
//          //for (const R3_Triangle_t &t : mesh) {
//              //t.transform(mat);
//          //}
//          return 0;
//      }


//  int amfToMeshInMemory(const char *amfFile, TriangleMesh *mesh) {
//  	//long timeIni, timeEnd;
//      //clock_t timeIni, timeEnd;
//      long timeIni = Timer::getTimeMs();
//      XMLDocument doc;
//  	std::vector<R3_t> points;
//  	doc.LoadFile(amfFile);
//  	
//  	XMLElement *m = doc.FirstChildElement()->FirstChildElement("object")->FirstChildElement("mesh");
//  	XMLElement *vertices = m->FirstChildElement("vertices");
//  	
//  	for (XMLElement *vertex = vertices->FirstChildElement("vertex"); vertex; vertex = vertex->NextSiblingElement())
//  	{
//  		XMLElement *coords = vertex->FirstChildElement("coordinates");
//  		float x = atof(coords->FirstChildElement("x")->GetText());
//  		float y = atof(coords->FirstChildElement("y")->GetText());
//  		float z = atof(coords->FirstChildElement("z")->GetText());
//  		
//          //long timeIni = Timer::getTimeMs();
//          //timeIni = clock();
//          //auto timeIni = chrono::high_resolution_clock::now();
//          
//  		points.push_back(R3_t(x,y,z));
//          
//          //timeEnd = clock();
//          //auto timeEnd = chrono::high_resolution_clock::now();
//          //long timeEnd = Timer::getTimeMs();
//          //timeBuild += timeEnd - timeIni;
//          
//          //timeBuild += chrono::duration_cast<chrono::milliseconds>(timeEnd - timeIni);
//  	}
//  	
//  	XMLElement *volume = m->FirstChildElement("volume");
//  	//int count=0;
//  	for (XMLElement *triangle = volume->FirstChildElement("triangle"); triangle; triangle = triangle->NextSiblingElement())
//  	{
//  		//std::cout << "R3_Triangle_t " << count++ << ":" << std::endl;
//  		R3_t v_1 = points[atoi(triangle->FirstChildElement("v1")->GetText())];
//  		R3_t v_2 = points[atoi(triangle->FirstChildElement("v2")->GetText())];
//  		R3_t v_3 = points[atoi(triangle->FirstChildElement("R3_t")->GetText())];
//  		//long timeIni = Timer::getTimeMs();
//          //timeIni = clock();
//          //auto timeIni = chrono::high_resolution_clock::now();
//  		R3_Triangle_t t(R3_t(0,0,0), v_1, v_2, v_3);
//  		
//  		mesh->push_back(t);
//          //long timeEnd = Timer::getTimeMs();
//          //timeBuild += timeEnd - timeIni;
//          //timeEnd = clock();
//          //auto timeEnd = chrono::high_resolution_clock::now();
//          
//          //timeBuild += chrono::duration_cast<chrono::milliseconds>(timeEnd - timeIni);
//  		//std::cout << "\tv1: (x: " << v1.x << "; y: " << v1.y << "; z: " << v1.z << ")" << std::endl;
//  		//std::cout << "\tv2: (x: " << v2.x << "; y: " << v2.y << "; z: " << v2.z << ")" << std::endl;
//  		//std::cout << "\tv3: (x: " << R3_t.x << "; y: " << R3_t.y << "; z: " << R3_t.z << ")" << std::endl;
//  		//std::cout << "-----------------------" << std::endl;
//  	}
//      
//      long timeEnd = Timer::getTimeMs();
//      timeBuild += timeEnd - timeIni;
//  	
//  	return 0;
//  }

void R3_Mesh_recompute_box(R3_Mesh_t *mesh);
/* Recomputes the bounding box of {mesh} from the triangle vertices. */

void R3_Mesh_box_include(R3_t *vMin, R3_t *vMax, R3_t *v);
/* Updates the bounding bbox corners {vMin,vMax} to include the point {v}. */

int stlToMeshInMemory(const char *stlFile, R3_Mesh_t *mesh, bool isBinaryFormat) {
  if (!isBinaryFormat) {
    ifstream in(stlFile);
    if (!in.good()) return 1;
    std::string s0, s1;
    float n0, n1, n2, f0, f1, f2, f3, f4, f5, f6, f7, f8;
    while (!in.eof()) {
      in >> s0;                                // "facet" || "endsolid"
      if (s0=="facet") {
	in >> s1 >> n0 >> n1 >> n2;            // normal x y z
	in >> s0 >> s1;                        // outer loop
	in >> s0 >> f0 >> f1 >> f2;         // vertex x y z
	in >> s0 >> f3 >> f4 >> f5;         // vertex x y z
	in >> s0 >> f6 >> f7 >> f8;         // vertex x y z
	in >> s0;                            // endloop
	in >> s0;                            // endfacet
	// Generate a new R3_Triangle_t with Normal as 3 Vertices
	R3_t normal = (R3_t){ .x = n0, .y = n1, .z = n2 };

	R3_t v0 = (R3_t){ .x = f0, .y = f1, .z = f2 };
	R3_t v1 = (R3_t){ .x = f3, .y = f4, .z = f5 };
	R3_t v2 = (R3_t){ .x = f6, .y = f7, .z = f8 };
	R3_Triangle_t t = R3_Triangle_create(normal, v0, v1, v2); 
	mesh->T.push_back(t);
        mesh->meshSize++;
      }
      else if (s0=="endsolid") {
	break;
      }
    }
    in.close();
  }
  else {
    FILE *f = fopen(stlFile, "rb");
    if (f == NULL) return 1;
    char title[80];
    int nFaces;
    int err;
    err = fread(title, 80, 1, f);
    err = fread((void*)&nFaces, 4, 1, f);
    float v[12]; // normal=3, vertices=3*3 = 12
    unsigned short uint16;
    // Every Face is 50 Bytes: Normal(3*float), Vertices(9*float), 2 Bytes Spacer
    for (size_t i=0; i<nFaces; ++i) {
      for (size_t j=0; j<12; ++j) {
	err = fread((void*)&v[j], sizeof(float), 1, f);
      }
      err = fread((void*)&uint16, sizeof(unsigned short), 1, f); // spacer between successive faces
      // Generate a new R3_Triangle_t with Normal as 3 Vertices

      R3_t normal = (R3_t){ .x = v[0], .y = v[1], .z = v[2] };

      R3_t v0 = (R3_t){ .x = v[3], .y = v[4], .z = v[5] };
      R3_t v1 = (R3_t){ .x = v[6], .y = v[7], .z = v[8] };
      R3_t v2 = (R3_t){ .x = v[9], .y = v[10], .z = v[11] };
      R3_Triangle_t t = R3_Triangle_create(normal, v0, v1, v2); 
      mesh->T.push_back(t);
      mesh->meshSize++;
    }
    fclose(f);
  }
  R3_Mesh_recompute_box(mesh);
  return 0;
}

void R3_Mesh_recompute_box(R3_Mesh_t *mesh) {
  for (size_t i = 0; i < mesh->meshSize; i++) {
    R3_Triangle_t *t = &(mesh->T.at(i));
    R3_Mesh_box_include(&(mesh->vMin), &(mesh->vMax), &(t->v[0]));
    R3_Mesh_box_include(&(mesh->vMin), &(mesh->vMax), &(t->v[1]));
    R3_Mesh_box_include(&(mesh->vMin), &(mesh->vMax), &(t->v[2]));
  }
}

void R3_Mesh_box_include(R3_t *vMin, R3_t *vMax, R3_t *v) {
  if (v->x < vMin->x) { vMin->x = v->x; }
  if (v->x > vMax->x) { vMax->x = v->x; }

  if (v->y < vMin->y) { vMin->y = v->y; }
  if (v->y > vMax->y) { vMax->y = v->y; }

  if (v->z < vMin->z) { vMin->z = v->z; }
  if (v->z > vMax->z) { vMax->z = v->z; }
}

R2_t R3_Mesh_Side_slice(R3_t *vi, R3_t *vj, float Z) {
  double dx = vj->x - vi->x;
  double dy = vj->y - vi->x;
  double dz = vj->z - vi->z;
  assert(dz != 0);
  double frac = (Z - vi->z)/dz;
  float xint = (float)(frac*dx + (double)vi->x);
  float yint = (float)(frac*dy + (double)vi->y);
  return (R2_t){ .x = xint, .y = yint };
}

R2_Segment_t R3_Mesh_Triangle_slice(R3_Triangle_t *t, float Z) {
  assert((t->zMin < Z) && (t->zMax > Z));
  int np = 0; /* Number of segment endpoints found */
  R2_Segment_t seg;
  for (int i = 0; i < 3; i++) {
    /* Get side {i} of triangle: */
    int j = (i == 2 ? 0 : i+1);
    R3_t *vi = &(t->v[i]);
    R3_t *vj = &(t->v[j]);
    /* Check for intersection of plane with {vi--vj}. */
    /* Must consider segment closed at bottom and open at top in case {Z} goes through a vertex. */
    float vzMin = (vi->z < vj->z ? vi->z : vj->z);
    float vzMax = (vi->z > vj->z ? vi->z : vj->z);
    if ((vzMin <= Z) && (vzMax > Z)) {
      R2_t p = R3_Mesh_Side_slice(vi, vj, Z);
      assert(np < 2);
      seg.v[np] = p;
      np++;
    }
  }
  assert(np == 2);
  return seg;
}

R3_Mesh_Triangle_List_t R3_Mesh_Triangle_List_create (void) {
  R3_Mesh_Triangle_List_t list;
  list.head = NULL;
  list.tail = NULL;
  return list;
}

void R3_Mesh_Triangle_List_insert (R3_Triangle_t *t, R3_Mesh_Triangle_List_t *l) {
  R3_Mesh_Triangle_Node_t *node = (R3_Mesh_Triangle_Node_t *)malloc(sizeof(R3_Mesh_Triangle_Node_t));
  node->t = t;
  node->next = l->head;
  node->prev = NULL;
  if (l->head == NULL) {
    /*New head*/
    l->head = l->tail = node;
  }
  else {
    l->head->prev = node;
    l->head = node;
  }
}

void R3_Mesh_Triangle_List_union (R3_Mesh_Triangle_List_t *l1, R3_Mesh_Triangle_List_t *l2) {
  if ((l1->head != NULL) && (l2->head != NULL)) {
    l1->tail->next = l2->head;
    l2->head->prev = l1->tail;
    l1->tail = l2->tail;
  }
  else if (l2->head != NULL) {
    l1->head = l2->head;
    l1->tail = l2->tail;
  }
  l2->head = NULL;
  l2->tail = NULL;
}

void R3_Mesh_Triangle_List_remove (R3_Mesh_Triangle_List_t *l, R3_Mesh_Triangle_Node_t *node) {
  if ((node->prev == NULL) && (node->next == NULL)) {
    l->head = NULL;
    l->tail = NULL;
  } else if (node->prev == NULL) {
    node->next->prev = NULL;
    l->head = node->next;
  } else if (node->next == NULL) {
    node->prev->next = NULL;
    l->tail = node->prev;
  } else {
    R3_Mesh_Triangle_Node_t *next = node->next;
    R3_Mesh_Triangle_Node_t *prev = node->prev;
    next->prev = prev;
    prev->next = next;
  }
  free(node);
}

// void R3_Trangle_transform(R3_Triangle_t *t, const glm::mat4 *mat) {
//   R3_transform(&(t->v[0]), mat); R3_transform(&(t->v[1]), mat); R3_transform(&(t->v[2]), mat);
//   R3_Triangle_fixZMinZmax(t);
// }

/*
    R3_Triangle_t& operator-=(const R3_t &pt) { v[0]-=pt; v[1]-=pt; v[2]-=pt; return *this;}
	bool operator<(const R3_Triangle_t &t) { return zMin < t.zMin; }
	friend ostream& operator<<(ostream& os, const R3_Triangle_t& t) {
		os << "V0: (" << t.v[0] << "); V1: (" << t.v[1] << "); V2: (" << t.v[2] << ")";
		return os;
	}

    // @return -1 = all triangle is on plane back side
    //          0 = plane intersects the triangle
    //          1 = all triangle is on plane front side
    //         -2 = error in function
	int checkInterception(const R3_Plane_t &plane, LineMesh &lm) const {
		R3_Segment_t ls;
		vector<R3_Segment_t> segments;
		size_t cntFront=0, cntBack=0;
		for (size_t j=0; j<3; ++j) {
			float distance = plane.distanceToPoint(v[j]);
			if (distance<0) ++cntBack;
			else ++cntFront;
		}
		if (3 == cntBack) return -1;
		else if (3 == cntFront) return 1;
		
		size_t lines[] = {0,1,1,2,2,0};	
		
		for (size_t i=0; i<3; ++i) {
			R3_t a = v[lines[i*2+0]];
			R3_t b = v[lines[i*2+1]];
			if (a>b) swap(a,b);
			
			const float da = plane.distanceToPoint(a);
            const float db = plane.distanceToPoint(b);
			
			if ((da*db < 0 || 0 == da || 0 == db) && (da != db)) {
				ls.v[0] = a;
				ls.v[1] = b;			
				
				segments.push_back(ls);
			}
		}
		
		if (2 == segments.size()) {
			//R3_t a = segments[0].v[0];
			//R3_t b = segments[0].v[1];
			//cout << "A1: " << a.x << "|" << a.y << "|" << a.z << " / " << b.x << "|" << b.y << "|" << b.z << endl;
			lm[segments[0]].push_back(segments[1]);
			lm[segments[1]].push_back(segments[0]);
			
			//a = segments[1].v[0];
			//b = segments[1].v[1];
			//cout << "A2: " << a.x << "|" << a.y << "|" << a.z << " / " << b.x << "|" << b.y << "|" << b.z << endl;

		} else return -2;
		return 0;
	}

int R3_Trangle_getSegment (const R3_Plane_t *plane, R3_Segment_t *ls) {
        //CCW triangle: 
        size_t lines[] = {0,1,1,2,2,0};
        R3_t intersectPoints[3];
        int size = 0;
        for (size_t i = 0; i < 3; i++) {
            R3_t a = v[lines[i*2+0]];
            R3_t b = v[lines[i*2+1]];
            if (a>b) swap(a,b);
            const float da = plane.distanceToPoint(a);
            const float db = plane.distanceToPoint(b);
            if ((da * db) <= 0) {
                //Intersection factor between 0 and 1:
                const float s = da/(da-db);
                R3_t bMinusa = b - a;
                R3_t r = a + bMinusa * s;
                r.x = round (r.x * 100.0) / 100.0;
                r.y = round (r.y * 100.0) / 100.0;
                r.z = round (r.z * 100.0) / 100.0;
                intersectPoints[i] = r;
                size++;
            }
        }
        ls.v[0] = intersectPoints[0];
        for (int i = 1; i < size; i++) {
            if ((intersectPoints[0] != intersectPoints[i]) || (size == 2)) {
                ls.v[1] = intersectPoints[i];
                return 0;
            }
        }
        ls.v[1] = intersectPoints[1];
        return 0;
    }

	
    // @return -1 = all triangle is on plane front/back side
    //          0 = plane intersects the triangle
    int intersectPlane(const R3_Plane_t &plane, R3_Segment_t &ls) const {

        size_t front = 0, back = 0;
        visits++;
        for (size_t i = 0; i < 3; i++) {
            float distance = plane.distanceToPoint(v[i]);
            if (distance < 0) { back++; }
            else { front++; }
        }

        if ( (back == 3) || (front == 3) ) {
            //cout << "sem interseccao" << endl; 
            return -1;
        } 

        //CCW R3_Triangle_t: 
        size_t lines[] = {0,1,1,2,2,0};

        R3_t intersectPoints[3];
        //std::vector<R3_t> intersectPoints;

        int size = 0;
        for (size_t i = 0; i < 3; i++) {

            R3_t a = v[lines[i*2+0]];
            R3_t b = v[lines[i*2+1]];
            if (a>b) swap(a,b);
            const float da = plane.distanceToPoint(a);
            const float db = plane.distanceToPoint(b);

            if ((da * db) <= 0) {
                //Intersection factor between 0 and 1:
                const float s = da/(da-db);
                R3_t bMinusa = b - a;
                R3_t r = a + bMinusa * s;
                //r.x = roundIt(r.x, 2);
                //r.y = roundIt(r.y, 2);
                //r.z = roundIt(r.z, 2);
                //CUIDADO COMENTEI O CODIGO ABAIXO
                r.x = round (r.x * 100.0) / 100.0;
                r.y = round (r.y * 100.0) / 100.0;
                r.z = round (r.z * 100.0) / 100.0;
                intersectPoints[i] = r;
                size++;
                //intersectPoints.push_back(r);
            }
        }

        ls.v[0] = intersectPoints[0];
        //for (int i = 1; i < intersectPoints.size(); i++) {
        for (int i = 1; i < size; i++) {
            if ((intersectPoints[0] != intersectPoints[i]) || (size == 2)) {
                ls.v[1] = intersectPoints[i];
                return 0;
            }
        }

        //throw logic_error("Error in interceptPlane");
        //return -2;
        ls.v[1] = intersectPoints[1];
        return 0;

       //return 0;
    }
};  

class R3_Triangle_Comparer_t {
public:
	bool operator() (const R3_Triangle_t &t1, const R3_Triangle_t &t2) const {
		if (t1.zMin == t2.zMin)
			return t1.zMax < t2.zMax;
		else
			return t1.zMin < t2.zMin;
	}
};

bool R3_Triangle_compare (R3_Triangle_t t1, R3_Triangle_t t2) {
   if (t1.zMin == t2.zMin) {
      //cout << "t1: " << t1.zMax << ", t2: " << t2.zMax << endl;
      //if (t1.zMax < t2.zMax) { cout << "t1 menor" << endl; } 
      return t1.zMax < t2.zMax;
   }
   else
      return t1.zMin < t2.zMin;
}
*/

