/* See {R2_LoopClosure.h}. */

#include "R2.h"
#include "R2_LoopClosure.h"

struct R2_Vertex_Hasher_t {
  int operator() (const R2_t *v) const {
    int h = R2_hash (*v);
    return h;
  }
}

typedef struct R2_Vertex_Pair_t {
  R2_t v, w;  /* Two neighbors of the key point, or {+INFINITY} if not defined. */
} R2_Vertex_Pair;

typedef unordered_map<R2_t,R2_Vertex_Pair_t,R2_Vertex_Hasher_t> NeighborTable_t; 
/* The neighbor hash table for {LoopClosureIterative}. */

void AddNeighborToTable(NeighborTable_t *H, R2_t *u, R2_t *v);
/* Adds to the neighbor tale {H} the point {v} as a neighbor of {u}. 
   There must be at most one neighbor of {u} already in the table. */

/* IMPLEMENTATIONS */

void AddNeighborToTable(NeighborTable_t *H, R2_t *u, R2_t *v) {
  R2_Vertex_Pair vw = H[*u];
  NeighborTable_t::const_iterator entry = H->find(h);
  if (entry == H->end()) {
    // No neighbors yet:
    R2_Vertex_Pair_t value = (R2_Vertex_Pair_t){ .v = *v, .w = (R2_t){ +INFINITY, +INFINITY } };
    H[*u] = value;
  } else {
    // Some neighbors already -- must be only one.
    R2_t key = entry->first;
    assert((key.x == u->x) && (key.y == u->y));
    R2_Vertex_Pair_t value = entry->second;
    assert((value.w.x == +INFINITY) && (value.w.y == +INFINITY)); 
    value.w = (*v);
    H[u] = value;
  }
}

void R2_LoopClosure_close (vector<R2_Segment_t> segs, vector<R2_Polygon_t> &loops) {

  /*Creating the hash table of proper size.*/
  NeighborTable_t H;

  /*Filling the hash table.*/
  for (std::vector<R2_Segment_t>::iterator i = segs.begin(); i != segs.end(); i++) {
    R2_Segment_t *q = i;
    AddNeighborToTable(H, &(q->v[0]), &(q->v[1]));
    AddNeighborToTable(H, &(q->v[1]), &(q->v[0]));
  }

  /*Loop-closure.*/
  int numContours = 0;
  R3_t v, tail;
  while (! H.empty()) {
    Polygon_t P;
    R2_t prev;
    R2_t current;
    R2_t last;
    int j;
    { 
      auto it = H.begin();
      /* First vertex: */
      R2_t u = it->first;
      P.push_back(u); j = 1;
      /* Second and last vertices: */
      R2_Vertex_Pair vw = it->second;
      P.push_back(vw->v); j = 2;
      prev = u;
      current = vw->v;
      last = vw->w;
      H.erase(u);
    }
    /* Collect other vertices: */
    do {
      R2_t next;
      {
	it = H.find(current);
	assert (it != H.end());
        R2_t key = it->first;
	assert((key.x == current->x) && (key.y == current->y));
        R2_Vertex_Pair vw = it->second;
        if ((vw.v.x == prev.x) && (vw.v.y == prev.y)) {
	  next = vw->w;
	} else if ((vw.w.x == prev.x) && (vw.w.y == prev.y)) {
	  next = vw.v;
	} else {
	  assert(false);
	}
        H.erase(key);
      }
      P.push_back(next);
      prev = current;
      current = next;
    }
    while ((current.x != last.x) || (current.y != last.y));
    numContours++;
    loops.push_back(P);
  }
} 
