/* See hr2.h */
/* Last edited on 2012-01-03 22:21:58 by stolfi */ 

/* Based on HR2.m3, created 1994-05-04 by J. Stolfi. */

#define _GNU_SOURCE
#include <math.h>
#include <assert.h>

#include <r2.h>
#include <r3.h>
#include <rn.h>
#include <r3x3.h>
#include <r2x2.h>
#include <affirm.h>
#include <sign.h>
#include <sign_get.h>

#include <hr2.h>

#define NH 3
  /* Number of homogeneous coordinates in a point. */

#define NC 2
  /* Number of Cartesian coordinates in a point. */

hr2_point_t hr2_from_r2(r2_t *c)
  {
    return (hr2_point_t){{{1.0, c->c[0], c->c[1]}}};
  }

r2_t r2_from_hr2(hr2_point_t *p)
  {
    double w = p->c.c[0];
    demand(w != 0.0, "point at infinity");
    return (r2_t){{p->c.c[1]/w, p->c.c[2]/w}};
  }

double hr2_pt_pt_diff(hr2_point_t *p, hr2_point_t *q)
  {
    return r3_angle(&(p->c), &(q->c));
  }

sign_t hr2_side(hr2_point_t *p, hr2_line_t *L)
  {
    double dd = r3_dot(&(p->c), &(L->f));
    return sign_double(dd);
  }

sign_t hr2_orient(hr2_point_t *p, hr2_point_t *q, hr2_point_t *r)
  {
    double dd = r3_det(&(p->c), &(q->c), &(r->c));
    return sign_double(dd);
  }

hr2_line_t hr2_join(hr2_point_t *p, hr2_point_t *q)
  {
    r3_t f;
    r3_cross(&(p->c), &(q->c), &f);
    return (hr2_line_t){f};
  }

hr2_point_t hr2_meet(hr2_line_t *K, hr2_line_t *L)
  {
    r3_t c;
    r3_cross(&(K->f), &(L->f), &c);
    return (hr2_point_t){c};
  }

bool_t hr2_pmap_is_identity(hr2_pmap_t *m)
  {
    return r3x3_is_unif_scaling(&(m->dir), m->dir.c[0][0]); 
  }

hr2_point_t hr2_pmap_point(hr2_point_t *p, hr2_pmap_t *m)
  {
    r3_t q; 
    r3x3_map_row(&(p->c), &(m->dir), &q);
    return (hr2_point_t){q};
  }

hr2_line_t hr2_pmap_line(hr2_line_t *L, hr2_pmap_t *m)
  {
    r3_t f;
    r3x3_map_col(&(m->inv), &(L->f), &f);
    return (hr2_line_t){f};
  }

hr2_pmap_t hr2_pmap_translation(r2_t *vec)
  { hr2_pmap_t m;
    r3x3_ident(&(m.dir));
    r3x3_ident(&(m.inv));

    m.dir.c[0][1] = +vec->c[0];
    m.dir.c[0][2] = +vec->c[1];

    m.inv.c[0][1] = -vec->c[0];
    m.inv.c[0][2] = -vec->c[1];
    return m;
  }

hr2_pmap_t hr2_pmap_rotation(double ang)
  {
    double c = cos(ang);
    double s = sin(ang);

    hr2_pmap_t m;
    r3x3_ident(&(m.dir));
    r3x3_ident(&(m.inv));

    m.dir.c[1][1] = +c;
    m.dir.c[1][2] = +s;
    m.dir.c[2][1] = -s;
    m.dir.c[2][2] = +c;

    m.inv.c[1][1] = +c;
    m.inv.c[1][2] = -s;
    m.inv.c[2][1] = +s;
    m.inv.c[2][2] = +c;
    return m;
  }

hr2_pmap_t hr2_pmap_comp(hr2_pmap_t *m, hr2_pmap_t *n)
  {
    r3x3_t dir, inv;
    r3x3_mul(&(m->dir), &(n->dir), &dir);
    r3x3_mul(&(n->inv), &(m->inv), &inv);
    return (hr2_pmap_t){dir, inv};
  }

hr2_pmap_t hr2_pmap_inv(hr2_pmap_t *m)
  {
    return (hr2_pmap_t){m->inv, m->dir};
  }

r2_t hr2_line_dir(hr2_line_t *L)
  {
    double dx = L->f.c[2];
    double dy = -L->f.c[1];
    double length = hypot(dx, dy);
    return (r2_t){{dx/length, dy/length}};
  }

r2_t hr2_line_normal(hr2_line_t *L)
  {
    double nx = L->f.c[1];
    double ny = L->f.c[2];
    double length = hypot(nx, ny);
    return (r2_t){{nx/length, ny/length}};
  }

r2_t hr2_point_point_dir(hr2_point_t *frm, hr2_point_t *tto)
  {
    double fw = frm->c.c[0];
    double tw = tto->c.c[0];
    
    double fx = frm->c.c[1];
    double tx = tto->c.c[1];
    double dx = fw * tx - tw * fx;
    
    double fy = frm->c.c[2];
    double ty = tto->c.c[2];
    double dy = fw * ty - tw * fy;
    
    double length = hypot(dx, dy);
    return (r2_t){{dx/length, dy/length}};
  }

hr2_pmap_t hr2_pmap_from_points(hr2_point_t *p, hr2_point_t *q, hr2_point_t *r, hr2_point_t *u)
  {
    int i;
    hr2_pmap_t m; /* The resulting map. */
    
    /* Compute weights {(a,b,c)=w.c[0..2]} for rows of {Q} that map {[1,1,1]} to {u}: */
    r3_t w;
    { /* Compute a matrix {Q} that maps the cardinal points to {p,q,r} as given: */
      r3x3_t Q;
      Q.c[0][0] = p->c.c[0]; Q.c[0][1] = p->c.c[1]; Q.c[0][2] = p->c.c[2];
      Q.c[1][0] = q->c.c[0]; Q.c[1][1] = q->c.c[1]; Q.c[1][2] = q->c.c[2];
      Q.c[2][0] = r->c.c[0]; Q.c[2][1] = r->c.c[1]; Q.c[2][2] = r->c.c[2];
      r3x3_inv(&Q, &Q);
      w.c[0] = u->c.c[0];  w.c[1] = u->c.c[1];  w.c[2] = u->c.c[2];
      r3x3_map_row(&w, &Q, &w);
    }
    
    /* Make the weights positive, so that {p,q,r} are strictly honored: */
    for (i = 0; i < NH; i++) { w.c[i] = fabs(w.c[i]); }

    /* Ensure that {m.dir} maps the cardinal points to {p,q,r} and some unit point to {u}: */
    r3x3_t *P = &(m.dir);
    P->c[0][0] = w.c[0]*p->c.c[0]; P->c[0][1] = w.c[0]*p->c.c[1]; P->c[0][2] = w.c[0]*p->c.c[2];
    P->c[1][0] = w.c[1]*q->c.c[0]; P->c[1][1] = w.c[1]*q->c.c[1]; P->c[1][2] = w.c[1]*q->c.c[2];
    P->c[2][0] = w.c[2]*r->c.c[0]; P->c[2][1] = w.c[2]*r->c.c[1]; P->c[2][2] = w.c[2]*r->c.c[2];

    /* Compute the inverse map: */
    r3x3_inv(&(m.dir), &(m.inv));

    return m;

  }

hr2_pmap_t hr2_affmap_from_point_pairs(r2_vec_t *p1, r2_vec_t *p2)
  {
    int np = p1->ne;
    assert(np == p2->ne);
    int k;
    double debug = FALSE;
    
    if (debug) { fprintf(stderr, "--- computing the affine matrix ---\n"); }
    
    /* Compute the linear matrix {S} and the translation vector {v}: */
    r2x2_t S; r2x2_ident(&S);
    r2_t v = (r2_t){{ 0,0 }};
    if (np > 0)
      { 
        /* Compute the barycenters of {p1} and {p2}: */
        r2_t bar1; r2_vec_barycenter(p1, &bar1);
        if (debug) { r2_gen_print(stderr, &bar1, "%10.4f", "  bar1 = ( ", " ", " )\n"); }
        demand(r2_is_finite(&bar1), "barycenter of {p1} is undefined");

        r2_t bar2; r2_vec_barycenter(p2, &bar2);
        if (debug) { r2_gen_print(stderr, &bar2, "%10.4f", "  bar2 = ( ", " ", " )\n"); }
        demand(r2_is_finite(&bar2), "barycenter of {p2} is undefined");

        if (np > 1)
          { /* Compute mean 2×2 linear transformation {S} from {p1-bar1} to {p2-bar2}: */
            if (np == 2)
              { /* Compute rotation & scale matrix: */
                r2_t q1; r2_sub(&(p1->e[0]), &bar1, &q1);
                r2x2_t R1; r2x2_rot_and_scale(&q1, &R1);
                
                r2_t q2; r2_sub(&(p2->e[0]), &bar2, &q2);
                r2x2_t R2; r2x2_rot_and_scale(&q2, &R2);
                
                r2x2_inv(&R1, &R1); r2x2_mul(&R1, &R2, &S);
              }
            else
              { /* Compute general 2×2 linear transformation by least squares: */
            
                r2x2_t A; r2x2_zero(&A); /* Moment matrix. */
                r2x2_t B; r2x2_zero(&B); /* Projection matrix. */
                for (k = 0; k < np; k++)
                  { 
                    /* Reduce points relative to barycenter: */
                    r2_t q1k, q2k;
                    r2_sub(&(p1->e[k]), &bar1, &q1k);
                    r2_sub(&(p2->e[k]), &bar2, &q2k);
                    /* Accumulate moments and projections: */
                    int i,j;
                    for(i = 0; i < 2; i ++)
                      { for (j = 0; j < 2; j++)
                          { A.c[i][j] += q1k.c[i]*q1k.c[j];
                            B.c[i][j] += q1k.c[i]*q2k.c[j];
                          }
                      }
                  }
                r2x2_t Z; r2x2_inv(&A, &Z);
                r2x2_mul(&Z, &B, &S);
              }
            if (debug) 
              { fprintf(stderr, "  linear matrix:\n");
                r2x2_gen_print(stderr, &S, "%13.6e", "", "\n", "\n", "    [ ", " ", " ]");
              }
          }
          
        r2x2_map_row(&bar1, &S, &v);
        r2_sub(&bar2, &v, &v);
      }

    /* Assemble the map {P.dir}: */
    hr2_pmap_t P;
    P.dir.c[0][0] = 1;
    P.dir.c[1][0] = 0;
    P.dir.c[2][0] = 0;

    P.dir.c[0][1] = v.c[0];
    P.dir.c[0][2] = v.c[1];

    P.dir.c[1][1] = S.c[0][0];
    P.dir.c[1][2] = S.c[0][1];
    P.dir.c[2][1] = S.c[1][0];
    P.dir.c[2][2] = S.c[1][1];

    /* Compute the inverse map and return: */
    r3x3_inv(&(P.dir), &(P.inv));
    return P;
  }
        
