/* See {image_window_ops.h}. */
/* Last edited on 2012-10-27 20:15:21 by stolfilocal */

#define _GNU_SOURCE
#include <math.h>
#include <string.h>
#include <assert.h>

#include <affirm.h>
#include <bool.h>

#include <image_window_ops.h>

void iwop_get_window_size(iwop_kind_t op, int *wcolsP, int *wrowsP, int *wctrP)
  {
    int wcols = 0; /* Window width */
    int wrows = 0; /* Window height */
    switch(op)
      { 
        case iwop_kind_IDENT:      wcols = 1; wrows = 1; break;
        case iwop_kind_DX:         wcols = 3; wrows = 1; break;
        case iwop_kind_DY:         wcols = 1; wrows = 3; break;
        case iwop_kind_DXX:        wcols = 3; wrows = 1; break;
        case iwop_kind_DXY:        wcols = 3; wrows = 3; break;
        case iwop_kind_DYY:        wcols = 1; wrows = 3; break;
        case iwop_kind_GRADIENT:   wcols = 3; wrows = 3; break;
        case iwop_kind_LAPLACIAN:  wcols = 3; wrows = 3; break;
        case iwop_kind_ORTHICITY:  wcols = 3; wrows = 3; break;
        case iwop_kind_ELONGATION: wcols = 3; wrows = 3; break;
        case iwop_kind_AVERAGE:    wcols = 3; wrows = 3; break;
        case iwop_kind_DEVIATION:  wcols = 3; wrows = 3; break;
        default: demand(FALSE, "invalid {op}"); 
      }
    (*wcolsP) = wcols;
    (*wrowsP) = wrows;
    assert(wcols % 2 == 1);
    assert(wrows % 2 == 1);
    (*wctrP) = (wrows/2)*wcols + (wcols/2);
  }

void iwop_get_range(iwop_kind_t op, bool_t squared, double *loP, double *hiP)
  { 
    /* Determine the natural range {[lo_hi]} of the operator, without squaring: */
    double lo = NAN, hi = NAN;
    if (! squared)
      {
        switch(op)
          {
            case iwop_kind_IDENT:      lo = 00.0; hi = +1.0;       break;
            case iwop_kind_DX:         lo = -0.5; hi = +0.5;       break;
            case iwop_kind_DY:         lo = -0.5; hi = +0.5;       break;
            case iwop_kind_DXX:        lo = -2.0; hi = +2.0;       break;
            case iwop_kind_DXY:        lo = -0.5; hi = +0.5;       break;
            case iwop_kind_DYY:        lo = -0.5; hi = +0.5;       break;
            case iwop_kind_GRADIENT:   lo = 00.0; hi = +sqrt(0.5); break;
            case iwop_kind_LAPLACIAN:  lo = -4.0; hi = +4.0;       break;
            case iwop_kind_ORTHICITY:  lo = -2.0; hi = +2.0;       break;
            case iwop_kind_ELONGATION: lo = 00.0; hi = +sqrt(5.0); break;
            case iwop_kind_AVERAGE:    lo = 00.0; hi = +1.0;       break;
            case iwop_kind_DEVIATION:  lo = 00.0; hi = +0.5;       break;
            default: demand(FALSE, "invalid {op}"); 
          }
      }
    else
      { lo = 0.0;
        switch(op)
          {
            case iwop_kind_IDENT:      hi = +1.00; break;
            case iwop_kind_DX:         hi = +0.25; break;
            case iwop_kind_DY:         hi = +0.25; break;
            case iwop_kind_DXX:        hi = +4.00; break;
            case iwop_kind_DXY:        hi = +0.25; break;
            case iwop_kind_DYY:        hi = +0.50; break;
            case iwop_kind_GRADIENT:   hi = +0.50; break;
            case iwop_kind_LAPLACIAN:  hi = +16.0; break;
            case iwop_kind_ORTHICITY:  hi = +4.00; break;
            case iwop_kind_ELONGATION: hi = +5.00; break;
            case iwop_kind_AVERAGE:    hi = +1.00; break;
            case iwop_kind_DEVIATION:  hi = +0.25; break;
            default: demand(FALSE, "invalid {op}"); 
          }
      }
    assert(! isnan(lo));
    assert(! isnan(hi));
    /* Return results: */
    (*loP) = lo;
    (*hiP) = hi;
  }

double iwop_apply(iwop_kind_t op, bool_t squared, int wctr, int wcols, double smp[])
  {
    double res = 0;
    switch(op)
      { case iwop_kind_IDENT:
          res = smp[wctr];
          if (squared) { res = res*res; }
          break;
        case iwop_kind_DX:
          res = iwop_dx(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_DY:
          res = iwop_dy(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_DXX:
          res = iwop_dxx(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_DXY:
          res = iwop_dxy(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_DYY:
          res = iwop_dyy(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_GRADIENT:
          if (squared) 
            { res = iwop_gradient_squared(wctr, wcols, smp); }
          else
            { res = iwop_gradient(wctr, wcols, smp); }
          break;
        case iwop_kind_LAPLACIAN:
          res = iwop_laplacian(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_ORTHICITY:
          res = iwop_orthicity(wctr, wcols, smp);
          if (squared) { res = res*res; }
          break;
        case iwop_kind_ELONGATION:
          if (squared) 
            { res = iwop_elongation_squared(wctr, wcols, smp); }
          else
            { res = iwop_elongation(wctr, wcols, smp); }
          break;
        case iwop_kind_AVERAGE:
          res = iwop_average(wctr, wcols, smp);
          break;
        case iwop_kind_DEVIATION:
          if (squared) 
            { res = iwop_deviation_squared(wctr, wcols, smp); }
          else
            { res = iwop_deviation(wctr, wcols, smp); }
          break;
        default: demand(FALSE, "invalid {op}"); 
      }
    return res;
  }

iwop_kind_t iwop_kind_from_string(const char *chop)
  {
    if (strcmp(chop, "ident") == 0)
      { return iwop_kind_IDENT; }
    else if (strcmp(chop, "dx") == 0)
      { return iwop_kind_DX; }
    else if (strcmp(chop, "dy") == 0)
      { return iwop_kind_DY; }
    else if (strcmp(chop, "dxx") == 0)
      { return iwop_kind_DXX; }
    else if (strcmp(chop, "dxy") == 0)
      { return iwop_kind_DXY; }
    else if (strcmp(chop, "dyy") == 0)
      { return iwop_kind_DYY; }
    else if (strcmp(chop, "gradient") == 0)
      { return iwop_kind_GRADIENT; }
    else if (strcmp(chop, "laplacian") == 0)
      { return iwop_kind_LAPLACIAN; }
    else if (strcmp(chop, "orthicity") == 0)
      { return iwop_kind_ORTHICITY; }
    else if (strcmp(chop, "elongation") == 0)
      { return iwop_kind_ELONGATION; }
    else if (strcmp(chop, "average") == 0)
      { return iwop_kind_AVERAGE; }
    else if (strcmp(chop, "deviation") == 0)
      { return iwop_kind_DEVIATION; }
    else 
      { /* Invalid op name: */
        return iwop_kind_NUM_VALUES;
      }
  }

const char *iwop_kind_to_string(iwop_kind_t op)
  {
    switch(op)
      { 
        case iwop_kind_IDENT:      return "ident";
        case iwop_kind_DX:         return "dx";
        case iwop_kind_DY:         return "dy";
        case iwop_kind_DXX:        return "dxx";
        case iwop_kind_DXY:        return "dxy";
        case iwop_kind_DYY:        return "dyy";
        case iwop_kind_GRADIENT:   return "gradient";
        case iwop_kind_LAPLACIAN:  return "laplacian";
        case iwop_kind_ORTHICITY:  return "orthicity";
        case iwop_kind_ELONGATION: return "elongation";
        case iwop_kind_AVERAGE:    return "average";
        case iwop_kind_DEVIATION:  return "deviation";
        default: demand(FALSE, "invalid {op}");
      }
  }

/* SPECIFIC OPERATORS */

double iwop_ident(int wctr, int wcols, double smp[])
  { return smp[wctr]; }

double iwop_dx(int wctr, int wcols, double smp[])
  { return (smp[wctr+1] - smp[wctr-1])/2; }

double iwop_dy(int wctr, int wcols, double smp[])
  { return (smp[wctr+wcols] - smp[wctr-wcols])/2; }

double iwop_dxx(int wctr, int wcols, double smp[])
  { return smp[wctr+1] + smp[wctr-1] - 2*smp[wctr]; }

double iwop_dxy(int wctr, int wcols, double smp[])
{ return (smp[wctr+wcols+1] - smp[wctr+wcols-1] - smp[wctr-wcols+1] + smp[wctr-wcols-1])/4; }

double iwop_dyy(int wctr, int wcols, double smp[])
  { return smp[wctr+wcols] + smp[wctr-wcols] - 2*smp[wctr]; }

double iwop_gradient(int wctr, int wcols, double smp[])
  { double dx = (smp[wctr+1] - smp[wctr-1])/2;
    double dy = (smp[wctr+wcols] - smp[wctr-wcols])/2;
    return hypot(dx,dy);
  }

double iwop_gradient_squared(int wctr, int wcols, double smp[])
  { double dx = (smp[wctr+1] - smp[wctr-1])/2;
    double dy = (smp[wctr+wcols] - smp[wctr-wcols])/2;
    return dx*dx + dy*dy;
  }

double iwop_laplacian(int wctr, int wcols, double smp[])
  { return smp[wctr+1] + smp[wctr-1] + smp[wctr+wcols] + smp[wctr-wcols] - 4*smp[wctr]; }

double iwop_orthicity(int wctr, int wcols, double smp[])
  { return smp[wctr+1] + smp[wctr-1] - smp[wctr+wcols] - smp[wctr-wcols]; }

double iwop_elongation(int wctr, int wcols, double smp[])
  { double ort = smp[wctr+1] + smp[wctr-1] - smp[wctr+wcols] - smp[wctr-wcols]; /* Orthicity. */
    double wrp = (smp[wctr+wcols+1] - smp[wctr+wcols-1] - smp[wctr-wcols+1] + smp[wctr-wcols-1])/2; /* Warp = {2*dxy}. */
    return hypot(wrp, ort);
  }

double iwop_elongation_squared(int wctr, int wcols, double smp[])
  { double ort = smp[wctr+1] + smp[wctr-1] - smp[wctr+wcols] - smp[wctr-wcols]; /* Orthicity. */
    double wrp = (smp[wctr+wcols+1] - smp[wctr+wcols-1] - smp[wctr-wcols+1] + smp[wctr-wcols-1])/2; /* Warp = {2*dxy}. */
    return wrp*wrp + ort*ort;
  }

double iwop_average(int wctr, int wcols, double smp[])
  { return 
      ( 4.0 * smp[wctr] + 
        2.0 * (smp[wctr+1] + smp[wctr-1] + smp[wctr+wcols] + smp[wctr-wcols]) +  
        smp[wctr+1+wcols] + smp[wctr+1-wcols] + smp[wctr-1+wcols] + smp[wctr-1-wcols]
      ) / 16.0;
  }

double iwop_deviation_squared(int wctr, int wcols, double smp[])
  { double avg = iwop_average(wctr, wcols, smp);
    double dmm = smp[wctr-1-wcols] - avg;
    double dmo = smp[wctr-1] - avg;
    double dmp = smp[wctr-1+wcols] - avg;
    double dom = smp[wctr-wcols] - avg;
    double doo = smp[wctr] - avg;
    double dop = smp[wctr+wcols] - avg;
    double dpm = smp[wctr+1-wcols] - avg;
    double dpo = smp[wctr+1] - avg;
    double dpp = smp[wctr+1+wcols] - avg;
    
    return 
      ( 4.0 * doo*doo + 
        2.0 * (dmo*dmo + dpo*dpo + dom*dom + dop*dop) +  
        dmm*dmm + dmp*dmp + dpm*dpm + dpp*dpp
      ) / 16.0;
  }

double iwop_deviation(int wctr, int wcols, double smp[])
  { double var = iwop_deviation_squared(wctr, wcols, smp);
    return sqrt(var);
  }

