/* Types and tools for STL files. */
/* Last edited on 2016-04-18 19:45:01 by stolfilocal */

#ifndef stmesh_STL_H
#define stmesh_STL_H

#define _GNU_SOURCE
#include <stdio.h>

#include <vec.h>
#include <bool.h>
#include <i3.h>

typedef struct stmesh_STL_r3_t { float c[3]; } stmesh_STL_r3_t;
  /* A point of {\RR^3}, with single-precision coordinates, as in and STL file. */

typedef struct stmesh_STL_face_t 
  { stmesh_STL_r3_t v[3];    /* Triangle vertices. */
    stmesh_STL_r3_t normal;  /* Triangle normal. */
  } stmesh_STL_face_t;
  /* A triangular face of the mesh, as represented in an STL file. */  

typedef void stmesh_STL_face_proc_t(int line, stmesh_STL_face_t *face);
  /* Type of a procedure that is called by {stmesh_STL_read} to process
    each triangle {*face} read from the STL file. The {line} argument is
    a line number to be used in error messages. Note: the face record
    {*face} is reused. */

void stmesh_STL_read(char *fileName, bool_t binary, stmesh_STL_face_proc_t *process_face);
  /* Reads the STL file {fileName} and calls {process_face(line,face)} for each
    face {face} read from it.  
    
    If {binary} is false, assumes ASCII STL format; in that case {line}
    is the line number in the file (counting from 1). If {binary} is
    true, assumes binary STL format; in that case, {line} is 1 fr the
    header, 2 for the number of faces, and is incremented by 1 for
    each face read. */

void stmesh_STL_print_triangle(FILE *wr, stmesh_STL_face_t *f); 
  /* Prints the STL face {f} to {wr} for debugging purposes. */

/* VERTEX ROUNDING */

i3_t stmesh_STL_round_point(stmesh_STL_r3_t *p, float eps, bool_t even);
  /* Converts a float-valued point {p} of {\RR^3}, as read from an STL
    file, to an integer point {q} of {\RZ^3}. Namely, rounds each coordinate
    {p.c[k]} of {p} to an integer multiple {eps*q.c[k]} of {eps},
    and returns the integer vector {q}.  If {even} is true,
    the coordinates will be rounded to even integers. */

#endif
