#! /usr/bin/gawk -f
# Last edited on 2015-11-17 19:42:45 by stolfilocal

BEGIN \
  { 
    # Writes an STL description of a symmetric Klein bottle,
    # modeled by a 2D array of triangles with hex topology. 
    
    # The intersection of the surface with a meridional half-plane
    # (a vertical half-plane bounded by the {Z} axis)
    # is a figure of eight on that plane, with the self-crossing point 
    # on the {XY} plane.  As the meridional plane makes a full turn
    # around the {Z}-axis, that curve rotates by 180 degrees aroud its
    # self-crossing point, so the surface gets a Moebius-like twist.
    # The surface self-intersects along a circle on the {XY}-plane.
    
    pi = 3.141592653589793238462;
    
    # User must define (with "-v") the number {N} of parallel bands of triangles.
    # and the number {M} of meridian bands.
    if (N =="") { printf "** must define {N}\n" > "/dev/stderr"; exit(1); }
    N = N+0;
    if (N < 6) { printf "** invalid {N}\n" > "/dev/stderr"; exit(1); }
    if ((N % 6) != 0) { printf "** {N} must be multiple of 6\n" > "/dev/stderr"; exit(1); }
    if (M =="") { printf "** must define {M}\n" > "/dev/stderr"; exit(1); }
    if (M < 8) { printf "** invalid {M}\n" > "/dev/stderr"; exit(1); }
    
    rad = 4.18;   # Radius of midline.
    hH = 2.15;   # Half-height of bottle at radius {rad}.

    printf "mesh will have %d bands and %d sectors (total %d triangles)", N, M, 2*N*M > "/dev/stderr"; 
    
    # The bottle consists of {N} "longitudinal bands" of triangles. Each band 
    # is limited by two "parallels" (curves on the bottle that spiral around 
    # the midline circle).  The parallels are
    # numbered from 0 to {N}, where parallel {N} coincides with parallel 0.
    # After a full turn, parallel {j} connects to parallel {N-j}.
    # Each band contains {2*M} triangles. Each triangle in a band has two vertices
    # on one parallel and the opposite vertex on the other parallel. The triangles 
    # alternate in orientation.
    
    # The triangles can also be seen as {M} "meridional bands" of triangles.
    # Each meridional band has {2*N} triangles, forming a zigzag band
    # that traces a figure of eight.
    
    printf "solid\n";
    
    # Points where the parallels intersect eight-curves at various longitudes. 
    # Indexed {[k,c]}, where {k} is the point index in {0..N}, {c} is {0,1,2} for {X,Y,Z}. 
    split("", pa); # Parallel points on generating segment at {i-0.5}.
    split("", pb); # Parallel points on generating segment at {i}.
    split("", pc); # Parallel points on generating segment at {i+0.5}.
    split("", pd); # Parallel points on generating segment at {i+1}.
    
    # We generate {M} transversal bands.
    # The transverse band index {i} starts at {-1} to compute {pa,pb} computed for band 0.
    for (i = -1; i < M; i++)
      { 
        # Compute the endpoints of the generating segment:
        compute_parallels(i+0.5, M,N,rad,hH, pc);
        compute_parallels(i+1.0, M,N,rad,hH, pd);
        if (i >= 0)
          { # Arrays {pa,pb} were set on previous iteration.
            # Output the triangles:
            for (j = 0; j < N; j++)
              { # Find the indices {j0,j1} of the two parallels, {j0} being base of first triangle:
                if ((j%2) == 0)
                  { j0 = j; j1 = j+1; }
                else
                  { j0 = j+1; j1 = j; }
                # Write the two triangles between parallels {j0} and {j1}
                write_triangle(pa[j0,0],pa[j0,1],pa[j0,2], pb[j1,0],pb[j1,1],pb[j1,2], pc[j0,0],pc[j0,1],pc[j0,2]);
                write_triangle(pb[j1,0],pb[j1,1],pb[j1,2], pc[j0,0],pc[j0,1],pc[j0,2], pd[j1,0],pd[j1,1],pd[j1,2]);
              }
          }
        # Prepare for the next transversal band:
        assign_parallels(pa, pc);
        assign_parallels(pb, pd);
      }
     
    printf "\n";
    printf "endsolid\n";
    
  }
        
function compute_parallels \
  ( i,M,N,rad,hH,p, \
    f,t,a,j,u,rj,zj,rm,zm,wt,rt,zt,Q,A,C,E,T,S \
  ) 
  {
    # Stores into {p[0..N,0..2]} the 3D coordinates of the {N} parallels
    # of the generator (a figure of eight curve) for longitude {i*2*pi/M}.
    
    # Global variables used: {pi}

    # Parameters for the morphing projective map of the unit circle to itself:
    Q = hH/rad;
    A = Q*Q + 1;
    E = Q*(2 + sqrt(4 - 3*A))/A;
    C = sqrt(1 - E*E);
    
    # Parameters of Euclidean mapping of morphed circle to the final circle:
    T = 2;       # Radial translation.
    S = hH/E;    # Final scaling.
    
    # Compute the longitude {f} and the twist {t} of the generating curve:
    f = i*2*pi/M;
    t = i*pi/M;

    for (j = 0; j <= N; j++)
      { # Compute equally spaced {r,z} points on the prototypical eight-curve.
        # It is centered at the origin, and osculates the unit circle at top 
        # and bottom:
        a = 2*pi*j/N
        rj = sqrt(0.5)*sin(2*a);
        zj = sin(a);
        
        if ((i == 0) || (i == 0.5*M))
          { printf "  %3d ( %+7.4f %+7.4f )", j, rj, zj > "/dev/stderr"; }

        # Rotate the figure of eight curve by angle {t} about the origin:
        rm = rj*cos(t) - zj*sin(t);
        zm = rj*sin(t) + zj*cos(t);

        # Apply a projective map of the {r,z} plane
        # that maps the the unit circle to a circle centered on the {Z=0} axis,
        # maps origin {(0,0)} to {(rad,0)}, and maps {(0,1)} to {(rad,hH)}:
        
        wt = 1 - C*rm;
        rt = S*((rm - C)/wt + T);
        zt = S*(E*zm/wt);
        
        # Convert to Cartesian coordinates:
        p[j,0] = rt*cos(f);
        p[j,1] = rt*sin(f);
        p[j,2] = zt;
        
        if ((i == 0) || (i == 0.5*M))
          { printf " --> ( %+7.4f %+7.4f %+7.4f )\n", p[j,0], p[j,1], p[j,2] > "/dev/stderr"; }
      }
  }

function assign_parallels(dst,src,  j,c)
  { 
    # Sets {dst[j,c] = src[j,c]} for {j} in {0..N} and {c} in {0..2}.
    
    for (j=0; j <= N; j++)
      { for (c = 0; c < 3; c ++)
          { dst[j,c] = src[j,c]; }
      }
  }

function write_triangle \
  ( xa,ya,za, xb,yb,zb, xc,yc,zc, \
    xu,yu,zu, xv,yv,zv, xn,yn,zn, dn \
  )        
  {
    # Compute normal direction: 
    xu = xc - xb; yu = yc - yb; zu = zc - zb;
    xv = xa - xb; yv = ya - yb; zv = za - zb;
    xn = zu*yv - zv*yu;
    yn = xu*zv - xv*zu;
    zn = yu*xv - yv*xu;
    dn = sqrt(xn*xn + yn*yn + zn*zn);
    if (dn <= 1.0e-8) 
      { xn = 0; yn = 0; zn = 0; }
    else
      { xn /= dn; yn /= dn; zn /= dn; }
    
    printf "\n";
    printf "facet\n"
    printf "normal %+.6f %+.6f %+.6f\n", xn, yn, zn;
    printf "outer loop\n";
    printf "  vertex %10.5f %10.5f %10.5f\n", xa, ya, za;
    printf "  vertex %10.5f %10.5f %10.5f\n", xb, yb, zb;
    printf "  vertex %10.5f %10.5f %10.5f\n", xc, yc, zc;
    printf "endloop\n";
    printf "endfacet\n";
  }
    
