#! /usr/bin/gawk -f
# Last edited on 2015-11-16 15:59:44 by stolfilocal

BEGIN \
  { 
    # Writes an STL description of a cube with each face
    # capped by a series of teeth with many 
    # faces sharing the same edge.
    
    pi = 3.141592653589793238462;
    
    # User must define (with "-v") the number {N} of teeth on each face.
    if (N =="") { printf "** must define {N}\n" > "/dev/stderr"; exit(1); }
    
    Sh = 3.1415926; # Half-side of cube.
    Th = 0.075*Sh;  # Thickness of the walls.
    Rt = 0.8 * Sh;  # Radius of teeth crown.
    Ht = 0.6 * Rt;  # Height of teeth.
    Dt = Ht/3;      # Displacement of base of teeth at center.
    
    if (Sh-2*Th <= Rt) { printf "** bug\n" > "/dev/stderr"; exit(1); }
    
    outer = 1; # True to generate the outer surface.
    inner = 1; # True to generate the inner surface.
    
    printf "solid\n";
    
    split("", M); # Orthonormal matrix with axis vectors. 
    
    # Top teeth bulge out of the box:
    fill_axis_matrix(M, +1,0,0, 0,+1,0, 0,0,+1); write_cube_face(M, Sh, Rt, Ht, Dt, N, Sh, Th);
    
    #Bottom teeth are recessed into the box:
    fill_axis_matrix(M, -1,0,0, 0,-1,0, 0,0,-1); write_cube_face(M, Sh, Rt, Ht, -Ht-Dt, N, Sh, Th);
    
    # Side teeth are half-in, half-out:
    fill_axis_matrix(M, 0,+1,0, 0,0,+1, +1,0,0); write_cube_face(M, Sh, Rt, Ht, -Ht/2, N, Sh, Th);
    fill_axis_matrix(M, 0,-1,0, 0,0,-1, -1,0,0); write_cube_face(M, Sh, Rt, Ht, -Ht/2, N, Sh, Th);
    
    fill_axis_matrix(M, 0,0,+1, +1,0,0, 0,+1,0); write_cube_face(M, Sh, Rt, Ht, -Ht/2, N, Sh, Th);
    fill_axis_matrix(M, 0,0,-1, -1,0,0, 0,-1,0); write_cube_face(M, Sh, Rt, Ht, -Ht/2, N, Sh, Th);
    
    printf "\n";
    printf "endsolid\n";
    
  }
        
function fill_axis_matrix(M, ux,uy,uz, vx,vy,vz, wx,wy,wz)
  {
    M[0,0] = ux;
    M[0,1] = uy;
    M[0,2] = uz;

    M[1,0] = vx;
    M[1,1] = vy;
    M[1,2] = vz;

    M[2,0] = wx;
    M[2,1] = wy;
    M[2,2] = wz;
  }
        
function write_cube_face \
  ( M, Sh, Rt, Ht, Dt, N, Sw, Th, \
    i, wn,w0,w1,wp,wq, at,au,av, bt,bu,bv, ct,cu,cv, et \
  )
  {
    # Generates one face of the cube, parallel to axis vectors
    # {u=M[0,*]} and {v=M[1,*]}, facing out towards {w=M[2,*]},
    # at coordinate {w=Sw}. Also generates the corresponding 
    # inner surface.
    
    # The outer surface has {N} teeth separated by {N} interteeth. 
    # The teeth come together at the center on a single edge 
    # parallel to the {w} direction.  The edge starts at 
    # {w=w0+Dt} and extends to {w=w0+Dt+Ht}. 

    # The inner surface has no teeth, just a pyramid 
    # displaced {Th} inwards relative to the inner surface of the teeth.
    
    # The {w} coordinates of the outer face and of the tooth top/bottom:
    w0 = Sw;
    wn = w0 + Dt;
    wp = w0 + Dt + Ht;
    
    # The {w} coordinates of the inner face and of the pyramid's apex:
    w1 = w0 - Th;
    wq = w1 + Dt;
    
    # The angular argument {at} and coords {au,av} of previous corner of the tooth:
    at = 0; au = Rt; av = 0;
    
    # Angle of previous corner of the face:
    et = -pi/4;
    
    for (i = 0; i < N; i++)
      { 
        # The angle and coords of the forward corner of tooth {i}:
        bt = 2 * pi * ((i + 0.5) / N);
        bu = Rt * cos(bt);
        bv = Rt * sin(bt);
        
        # The angle and coords of the forward corner of intertooth {i}:
        ct = 2 * pi * ((i + 1.0) / N);
        cu = Rt * cos(ct);
        cv = Rt * sin(ct);
        
        if (outer)
          { # Generate the outer surface of the tooth:
            write_uvw_triangle(M, 0,0,wn, 0,0,wp, au,av,w0); # One side.
            write_uvw_triangle(M, 0,0,wn, 0,0,wp, bu,bv,w0); # Other side.
            write_uvw_triangle(M, 0,0,wp, au,av,w0, bu,bv,w0); # Tooth face.
            write_uvw_triangle(M, 0,0,wn, bu,bv,w0, cu,cv,w0); # Gum face.
          }
        
        if (inner)
          { # Generate the faces of the inner pyramid:
            write_uvw_triangle(M, 0,0,wq, au,av,w1, bu,bv,w1); # Below tooth face.
            write_uvw_triangle(M, 0,0,wq, bu,bv,w1, cu,cv,w1); # Below gum face.
          }
        
        # Generate the outer and inner filler triangles:
        et = write_uvw_fillers(M, w0, w1, at, bt, et, Sh, Rt, Th);
        et = write_uvw_fillers(M, w0, w1, bt, ct, et, Sh, Rt, Th);
        
        # Prepare for next iteration: 
        at = ct; au = cu; av = cv;
      }
  }
  
function write_uvw_fillers \
  ( M,w0,w1,at,bt,et,Sh,Rt,Th, \
    au,av,bu,bv, eu0,ev0,eu1,ev1, mt,ft, fu0,fv0,fu1,fv1 \
  )
  { 
    # Writes a filler triangle on the outer face with {w}-coordinate {w0}
    # that spans two successive corners {a,b} on the tooth circle 
    # and a corner {f} of the face nearest to their midpoint.
    # If that corner is different from the previous face corner
    # {e}, also outputs the triangle {aef}. 
    #
    # Also writes the corresponding triangles on the inner face,
    # assumed to have {w}-coord {w1} and half-side {Sh-Th}.
    #
    # The points {a,b,e,f} are identified by their argument angles
    # {at,bt,et,ft}. The procedure returns {ft} as the result.
    
    # Compute the {u,v} coords of the two circle points:
    au = Rt * cos(at);
    av = Rt * sin(at);

    bu = Rt * cos(bt);
    bv = Rt * sin(bt);

    # Find the argument angle {ft} of the nearest face corner:
    mt = (at + bt)/2;
    ft = (int(mt/(pi/2)) + 0.5)*(pi/2);
    
    # Get the {u,v} coords of that corner on the outer face:
    fu0 = Sh * sqrt(2) * cos(ft);
    fv0 = Sh * sqrt(2) * sin(ft);
    
    # Get the {u,v} coords of that corner on the inner face:
    fu1 = (Sh-Th) * sqrt(2) * cos(ft);
    fv1 = (Sh-Th) * sqrt(2) * sin(ft);
    
    if (ft != et)
      { # Get the {u,v} coords of previous corner on outer face: 
        eu0 = Sh * sqrt(2) * cos(et);
        ev0 = Sh * sqrt(2) * sin(et);
        
        # Get the {u,v} coords of previous corner on inner face: 
        eu1 = (Sh-Th) * sqrt(2) * cos(et);
        ev1 = (Sh-Th) * sqrt(2) * sin(et);
        
        # Generate the {aef} triangle: 
        
        if (outer) { write_uvw_triangle(M, au,av,w0, eu0,ev0,w0, fu0,fv0,w0); }
        if (inner) { write_uvw_triangle(M, au,av,w1, eu1,ev1,w1, fu1,fv1,w1); }
      }
    
    # Generate the {abf} triangle and its counterpart on the inner face:
    if (outer) { write_uvw_triangle(M, au,av,w0, bu,bv,w0, fu0,fv0,w0); }
    if (inner) { write_uvw_triangle(M, au,av,w1, bu,bv,w1, fu1,fv1,w1); }
    
    return ft;
  }
        
function write_uvw_triangle\
  ( M, au,av,aw, bu,bv,bw, cu,cv,cw, \
    a,b,c \
  )
  {
    # Writes the triangle with corners {(au,av,aw)},
    # {(bu,bv,bw)}, and {(cu,cv,cw)}, mapped by the 
    # axis matrix {M}.
    
    split("", a); map_point(M, au,av,aw, a);
    split("", b); map_point(M, bu,bv,bw, b); 
    split("", c); map_point(M, cu,cv,cw, c); 
    write_triangle(a[0],a[1],a[2], b[0],b[1],b[2], c[0],c[1],c[2]);
  }

function map_point(M, au,av,aw, a)
  { 
    # Maps {(au,av,aw)} by the matrix {M} and stores in {a[0..2]}.
    # Assumes {a} is an array.
    a[0] = au*M[0,0] + av*M[1,0] + aw*M[2,0];
    a[1] = au*M[0,1] + av*M[1,1] + aw*M[2,1];
    a[2] = au*M[0,2] + av*M[1,2] + aw*M[2,2];
  }

function write_triangle \
  ( xa,ya,za, xb,yb,zb, xc,yc,zc, \
    xu,yu,zu, xv,yv,zv, xn,yn,zn, dn \
  )        
  {
    # Compute normal direction: 
    xu = xc - xb; yu = yc - yb; zu = zc - zb;
    xv = xa - xb; yv = ya - yb; zv = za - zb;
    xn = zu*yv - zv*yu;
    yn = xu*zv - xv*zu;
    zn = yu*xv - yv*xu;
    dn = sqrt(xn*xn + yn*yn + zn*zn);
    if (dn <= 1.0e-8) 
      { xn = 0; yn = 0; zn = 0; }
    else
      { xn /= dn; yn /= dn; zn /= dn; }
    
    printf "\n";
    printf "facet\n"
    printf "normal %+.6f %+.6f %+.6f\n", xn, yn, zn;
    printf "outer loop\n";
    printf "  vertex %10.5f %10.5f %10.5f\n", xa, ya, za;
    printf "  vertex %10.5f %10.5f %10.5f\n", xb, yb, zb;
    printf "  vertex %10.5f %10.5f %10.5f\n", xc, yc, zc;
    printf "endloop\n";
    printf "endfacet\n";
  }
    
    
