# Generic Makefile for one or more executable programs
# Last edited on 2015-10-01 16:03:03 by stolfilocal

######################################################################
# To be included by the top-level Makefile in the program's directory.
# The caller must define
#   
#   ${PROG} or ${PROGS}  the programs (whose main C cources are in current directory). 
#   
# and may also define
#   
#   ${IGNORE}         list of ".h" or ".c" files that should not be compiled or installed.
#   ${JS_LIBS}        list of Stolfi's libraries to bind with (WITHOUT directories).
#   ${USE_GL}         define as `YES' to bind with the GL headers and libraries.
#   ${USE_X11}        define as `YES' to bind with the X11 headers and libraries.
#   ${OTHER_LIBS}     list of additional libraries (WITH directories).
#   ${OTHER_OFILES}   list of additional .o files (WITH directories).
#   ${OTHER_I_FLAGS}  additional -I flags for C compilation and dependency checks.
#   ${OTHER_C_FLAGS}  additional flags for C compilation.
#   ${OTHER_LD_FLAGS} additional flags for linking.
#   
# All library names must include the extension (".a", ".so", etc.)
# The client must define the "all" target.
# The client may extend the "clean" target with double-colon rules.

include GENERIC.make

ifeq "/${PROGS}" "/"
PROGS := ${PROG}
endif

ifneq "/${PROGS}" "/"

# ----------------------------------------------------------------------
# Package components and directories

# Get list of all local ".h" files, and their derived objects:
HFILES := ${filter-out ${IGNORE}, ${wildcard *.h}}
HOFILES := ${subst .h,.ho,${HFILES}}

# Get list of all local ".c" files (including the main sources of all ${PROGS}):
CFILES := ${filter-out ${IGNORE}, ${wildcard *.c}}
OFILES := ${subst .c,.o,${CFILES}}

# Get list of ".c" files of good main programs and their objects:
PROG_CFILES := ${addsuffix .c,${PROGS}}
PROG_OFILES := ${subst .c,.o,${PROG_CFILES}}

# Get list of non-main ".c" files and objects:
LIB_CFILES := ${filter-out ${PROG_CFILES} ${IGNORE}, ${CFILES}}
LIB_OFILES := ${subst .c,.o,${LIB_CFILES}}

# Directory with test runs:
TEST_DIR := tests

# ----------------------------------------------------------------------
# Installation directories

# Where to install the program:
INST_LIB_DIR := ${INST_LIB_DIR}/${PLATFORM}
INST_BIN_DIR := ${INST_BIN_DIR}/${PLATFORM}
INST_MAN_DIR := ${INST_MAN_DIR}/${PLATFORM}
INST_CAT_DIR := ${INST_CAT_DIR}/${PLATFORM}

# ----------------------------------------------------------------------
# Locations of imported packages

# Location of Stolfi's headers and libraries:
JS_PUB_DIR := ./
JS_LIB_DIR := ./
JS_INC_DIR := ./

# Location of X11 headers and libraries (including ICE):
X11_INC_DIR := /usr/include/X11
X11_LIB_DIR := /usr/lib/x86_64-linux-gnu/

# ----------------------------------------------------------------------
# Optional libraries and include directories

# Get OpenGL libraries and headers if requested:
ifeq "/${USE_GL}" "/YES" 
  GL_LIBRARIES := \
    /usr/lib/x86_64-linux-gnu/libglut.so.3 \
    /usr/lib/x86_64-linux-gnu/libGLU.so \
    /usr/lib/x86_64-linux-gnu/libGL.so
  GL_I_FLAGS := \
    -I${JS_INC_DIR}/GL \
    -I/usr/include/GL
else
  GL_LIBRARIES :=
  GL_I_FLAGS := 
endif

# Get X11 libraries and headers if requested:
ifeq "/${USE_X11}" "/YES" 
  X11_LIBRARIES := \
    ${X11_LIB_DIR}/libX11.so \
    ${X11_LIB_DIR}/libXext.so \
    ${X11_LIB_DIR}/libXmu.so.6 \
    ${X11_LIB_DIR}/libXt.so.6 \
    ${X11_LIB_DIR}/libXi.so.6 \
    ${X11_LIB_DIR}/libSM.so.6 \
    ${X11_LIB_DIR}/libICE.so.6 
  X11_I_FLAGS := -I${X11_INC_DIR}
else
  X11_LIBRARIES := 
  X11_I_FLAGS := 
endif

#---------------------------------------------------------------------
# Default target for "make" with no target: 

all: uninstall build-progs install

#---------------------------------------------------------------------
# "make actions" performs the ${ACTIONS} locally:

actions:
ifeq "/${ACTIONS}" "/"
	@echo '$${ACTIONS} is empty'
else
	${MAKE} ${ACTIONS}
endif

#---------------------------------------------------------------------
# Directories to search for #include files:

I_FLAGS := \
  -I. \
  -I.. \
  -I../.. \
  ${GL_I_FLAGS} \
  ${X11_I_FLAGS} \
  ${OTHER_I_FLAGS}

#---------------------------------------------------------------------
# "make depend" recreates the source dependency file ${DEPFILE}.

DEPFILE := Deps.make

depend: ${DEPFILE}

${DEPFILE}: ${CFILES} ${HFILES}
	@/bin/rm -f ${DEPFILE}
	@./extract-ho-deps ${I_FLAGS} -I/usr/include \
            ${HFILES} ${CFILES} \
          | egrep -v ': /usr/include' \
          > ${DEPFILE}

#---------------------------------------------------------------------
# "make build-libs" is void because any local modules and libs will be built 
# together with the programs:

build-lib:
build-libs:

#---------------------------------------------------------------------
# "make build" and "make build-progs" assumes that the dependencies in ${DEPFILE}
# are up-to-date.

build: build-progs
build-prog: build-progs
  
build-progs: ${DEPFILE}
	${MAKE} debug do-build-progs

do-build-progs: ${PROGS}

LIBRARIES := \
  ${addprefix ${JS_LIB_DIR}/, ${JS_LIBS}} \
  ${GL_LIBRARIES} \
  ${X11_LIBRARIES} \
  ${OTHER_LIBS}

CC := /usr/bin/gcc

C_FLAGS := \
  -Wall -Wundef \
  -Wpointer-arith \
  -Wmissing-prototypes \
  -Wstrict-prototypes \
  -Wconversion \
  -Wno-sign-conversion \
  -Wclobbered \
  -Wcast-align \
  -Wtype-limits \
  -Wmissing-field-initializers \
  -ggdb \
  -ffloat-store \
  -frounding-math \
  -fpcc-struct-return \
  -std=c99
  
LD_FLAGS := \
  -ggdb

%.o: %.c
	${CC} -c ${C_FLAGS} ${OTHER_C_FLAGS} ${I_FLAGS} $*.c
   
%.ho: %.h
	${CC} -o $*.ho -c ${C_FLAGS} ${OTHER_C_FLAGS} ${I_FLAGS} -x c $*.h \
          || /bin/rm -f $*.ho
  
${PROGS}: ${PROG_OFILES} ${LIB_OFILES} ${LIBRARIES}
	@echo '# - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'
	@echo building $@ ...
	-rm -f $@
	${CC} -o $@ ${LD_FLAGS} ${OTHER_LD_FLAGS} $@.o \
          ${LIB_OFILES} ${OTHER_OFILES} \
          ${LIBRARIES} -lm -lrt
	@echo '# - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'

ifneq "/${wildcard ${DEPFILE}}" "/"
  # Include specific dependencies extracted by "make depend"
  include ${DEPFILE}
endif

#---------------------------------------------------------------------
# "make install" copies program and makefile to the public dir.

MANSECT := 1
MANPAGES := \
  ${wildcard ${addsuffix .${MANSECT},${PROGS}}} \
  DUMMY.1

install:
	@echo "=== install suppressed ==="

# install: install-progs install-man

install-progs: ${PROGS}
	cp -p ${PROGS} ${INST_BIN_DIR}/

install-man:
	@for mp in ${MANPAGES} ; do \
          if [ -r $${mp} ]; then \
            cp -p $${mp} ${INST_MAN_DIR}/man${MANSECT}/ ;\
          else \
            echo "file $${mp} not found - ignored" ;\
          fi ; \
        done

#---------------------------------------------------------------------
# "make uninstall" deletes the exported program and makefile:

uninstall:
	@echo "=== uninstall suppressed ==="

# uninstall: uninstall-progs uninstall-man

uninstall-progs:
	( cd ${INST_BIN_DIR}/. && rm -f ${PROGS} )

uninstall-man:
	if [[ -d ${INST_MAN_DIR}/man${MANSECT} ]]; then cd ${INST_MAN_DIR}/man${MANSECT}/. && rm -f ${MANPAGES}; fi
	if [[ -d ${INST_CAT_DIR}/cat${MANSECT} ]]; then cd ${INST_CAT_DIR}/cat${MANSECT}/. && rm -f ${MANPAGES}; fi

#---------------------------------------------------------------------
# "make tar-export" creates a tarball of sources, manpages, and executables.

THIS_DIR := ${notdir ${shell pwd}}
NOW := ${shell date '+%Y-%m-%d-%H%M%S'}
TARBALL_FILE := ${THIS_DIR}-${NOW}.tgz
  
tar-export: ${PROGS} extract-ho-deps
	@echo "archiving to ${TARBALL_FILE} ..."
	tar -cvzf ${TARBALL_FILE} \
          00-README Makefile \
            extract-ho-deps \
            ${PROGS} \
            ${HFILES} ${CFILES} \
            ${MANPAGES}

#---------------------------------------------------------------------
# "make tar-save" creates a date-stamped tarball of all files except
# edit backups and derived files.

SAVEDIR := SAVE
SAVETARFILE := ${LIBNAME}-${NOW}-save.tgz
  
tar-save: clean
	tar -cvzf ${SAVEDIR}/${SAVETARFILE} \
          `find ./ -type f -print | egrep -v -e '[~]$'`

#---------------------------------------------------------------------
# "make check" runs the tests in the "tests" subdirectory.

TEST_DIR := tests

check: ${TEST_DIR}
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR}/. && ${MAKE} all ) fi

#---------------------------------------------------------------------
# "make clean" deletes all derived files.
# It is a double-colon rule so that clients may add more actions.
# Also recreates the ".deps" files.

clean:: clean-progs clean-tests

clean-progs: 
	-/bin/rm -f *.o *.ho *.a core ${PROGS} *.deps ${DEPFILE}
	touch --date='Jan 01 1970' ${DEPFILE}

clean-tests:
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && make clean ) fi

endif
# End of ${PROG} or ${PROGS} section.
######################################################################

#---------------------------------------------------------------------
# "make debug" prints some debugging info:
debug:
	@echo "PROG = ${PROG}"
	@echo "PROGS = ${PROGS}"
