# Generic Makefile for one or more executable programs that test a library
# Last edited on 2010-05-25 02:39:05 by stolfi

######################################################################
# To be included by the top-level Makefile in the program's directory.
# The client must define 
#
#   ${PROG} or ${PROGS}  test programs (whose main C sources are in current directory).
#   ${TEST_LIB}          the library under test (WITHOUT directory).
#   ${TEST_LIB_DIR}      directory with latest sources (and headers) of ${TEST_LIB}.
#
# The client may also define:
#   
#   ${IGNORE}         list of ".h" or ".c" files that should not be compiled or installed.
#   ${JS_LIBS}        list of Stolfi's libraries to bind with (WITHOUT directories).
#   ${USE_GL}         define as `YES' to bind with the GL headers and libraries.
#   ${USE_X11}        define as `YES' to bind with the X11 headers and libraries.
#   ${OTHER_LIBS}     list of additional libraries (WITH directories).
#   ${OTHER_I_FLAGS}  additional -I flags for C compilation and dependency checks.
#   ${OTHER_C_FLAGS}  additional flags for C compilation.
#   ${OTHER_LD_FLAGS} additional flags for linking.
#   
# All library names must include the extension (".a", ".so", etc.)
# The client must define the "all" target.
# The client may extend the "clean" target with double-colon rules.

ifeq "/${PROGS}" "/"
PROGS := ${PROG}
endif

ifneq "/${PROGS}" "/"
ifneq "/${TEST_LIB}" "/"
ifneq "/${TEST_LIB_DIR}" "/"

SHELL := /bin/sh

# ----------------------------------------------------------------------
# Package components and directories

# Location of headers of the library under test:
TEST_INC_DIR := ${TEST_LIB_DIR}

# Get list of all local ".h" files, and their derived objects:
HFILES := ${filter-out ${IGNORE}, ${wildcard *.h}}
HOFILES := ${subst .h,.ho,${HFILES}}

# Get list of all local ".c" files (including the main sources of ${PROGS}):
CFILES := ${filter-out ${IGNORE}, ${wildcard *.c}}
OFILES := ${subst .c,.o,${CFILES}}

# Get list of ".c" files of good main programs and their objects:
PROG_CFILES := ${addsuffix .c,${PROGS}}
PROG_OFILES := ${subst .c,.o,${PROG_CFILES}}

# Get list of non-main ".c" files and objects:
LIB_CFILES := ${filter-out ${PROG_CFILES} ${IGNORE}, ${CFILES}}
LIB_OFILES := ${subst .c,.o,${LIB_CFILES}}

# ----------------------------------------------------------------------
# Locations of imported packages

# Location of Stolfi's headers and libraries:
JS_PUB_DIR := ${STOLFIHOME}
JS_LIB_DIR := ${JS_PUB_DIR}/lib/${PLATFORM}
JS_INC_DIR := ${JS_PUB_DIR}/include

# Location of X11 headers and libraries (including ICE):
X11_INC_DIR := /usr/include/X11 
X11_LIB_DIR := /usr/lib

# ----------------------------------------------------------------------
# Optional libraries and include directories
  
# Get OpenGL libraries and headers if requested:
ifeq "/${USE_GL}" "/YES" 
  GL_LIBRARIES := \
    ${JS_LIB_DIR}/libglut.so \
    /usr/lib/libGLU.so \
    /usr/lib/libGL.so
  GL_I_FLAGS := \
    -I ${JS_INC_DIR}/GL \
    -I /usr/include/GL
else
  GL_LIBRARIES :=
  GL_I_FLAGS := 
endif

# Get X11 libraries and headers if requested:
ifeq "/${USE_X11}" "/YES" 
  X11_LIBRARIES := \
    ${X11_LIB_DIR}/libX11.so \
    ${X11_LIB_DIR}/libXext.so \
    ${X11_LIB_DIR}/libXmu.so \
    ${X11_LIB_DIR}/libXt.so \
    ${X11_LIB_DIR}/libXi.so \
    ${X11_LIB_DIR}/libSM.so \
    ${X11_LIB_DIR}/libICE.so
  X11_I_FLAGS :=  -I${X11_INC_DIR}
else
  X11_LIBRARIES := 
  X11_I_FLAGS := 
endif

#---------------------------------------------------------------------
# Phony "make" targets:
# Client should define the "all" target

.PHONY:: \
  default-action \
  all actions \
  depend \
  build build-progs do-build-progs \
  clean clean-progs

#---------------------------------------------------------------------
# Default target: call the client's "all" target.
default-action: all

#---------------------------------------------------------------------
# "make actions" performs the actions listed in ${ACTIONS}, if any
actions:
ifeq "/${ACTIONS}" "/"
	@echo '$${ACTIONS} is empty'
else
	${MAKE} ${ACTIONS}
endif
  
#---------------------------------------------------------------------
# Directories to search for #include files:

I_FLAGS := \
  -I. \
  -I${TEST_INC_DIR} \
  -I${JS_INC_DIR} \
  ${GL_I_FLAGS} \
  ${X11_I_FLAGS} \
  ${OTHER_I_FLAGS}

#---------------------------------------------------------------------
# "make depend" recreates the source dependency file ${DEPFILE}.

DEPFILE := Deps.make

depend: ${DEPFILE}

${DEPFILE}: extract-ho-deps ${HFILES} ${CFILES}
	@/bin/rm -f ${DEPFILE}
	@./extract-ho-deps ${I_FLAGS} -I/usr/include \
            ${HFILES} ${CFILES} \
          | egrep -v ': /usr/include' \
          > ${DEPFILE}

# ----------------------------------------------------------------------
# Create a local symlink for {extract-ho-deps}, for {tar-export}'s sake

extract-ho-deps: ${STOLFIHOME}/bin/extract-ho-deps
	ln -s ${STOLFIHOME}/bin/extract-ho-deps

#---------------------------------------------------------------------
# "make build" and "make build-progs" make sure that the ${DEPFILE} is
# up to date, and then recreate the object files from the source files
# as necessary.

build: build-progs
  
build-progs: ${DEPFILE}
	${MAKE} do-build-progs

do-build-progs: ${PROGS}

LIBRARIES := \
  ${addprefix ${TEST_LIB_DIR}/, ${TEST_LIB}} \
  ${addprefix ${JS_LIB_DIR}/, ${JS_LIBS}} \
  ${GL_LIBRARIES} \
  ${X11_LIBRARIES} \
  ${OTHER_LIBS}

CC := /usr/bin/gcc

C_FLAGS := \
  -Wall -Wundef \
  -Wpointer-arith \
  -Wmissing-prototypes \
  -ggdb \
  -fpcc-struct-return \
  -ffloat-store \
  -frounding-math
  
LD_FLAGS := \
  -ggdb

%.o: %.c
	${CC} -c ${C_FLAGS} ${OTHER_C_FLAGS} ${I_FLAGS} $*.c
  
%.ho: %.h
	${CC} -o $*.ho -c ${C_FLAGS} ${OTHER_C_FLAGS} ${I_FLAGS} -x c $*.h \
          || /bin/rm -f $*.ho
  
${PROGS}: ${PROG_OFILES} ${LIB_OFILES} ${LIBRARIES}
	@echo '# - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'
	@echo building $@ ...
	-rm -f $@
	${CC} -o $@ ${LD_FLAGS} ${OTHER_LD_FLAGS} $@.o \
          ${LIB_OFILES} \
          ${LIBRARIES} -lm -lrt
	@echo '# - - - - - - - - - - - - - - - - - - - - - - - - - - - - -'
        
${TEST_LIB_DIR}/${TEST_LIB}: \
	${wildcard ${TEST_LIB_DIR}/*.h ${TEST_LIB_DIR}/*.c} \
	${wildcard ${JS_INC_DIR}/*.ho}
	cd ${TEST_LIB_DIR} && ${MAKE} ${TEST_LIB}

ifneq "/${wildcard ${DEPFILE}}" "/"
  # Include specific dependencies extracted by "make depend"
  include ${DEPFILE}
endif

#---------------------------------------------------------------------
# "make install" is a no-op:

install: 

#---------------------------------------------------------------------
# "make uninstall" is a no-op:

uninstall: 

#---------------------------------------------------------------------
# "make clean" deletes all derived files.
# It is a double-colon rule so that clients may add more actions.
# Also recreates the ".deps" file.

clean::
	-/bin/rm -f *.o *.ho *.a core ${PROGS} *.deps ${DEPFILE}
	${MAKE} depend

endif
endif
endif
# End of ${PROG} ${TEST_LIB} ${TEST_LIB_DIR} section.
######################################################################

