# Generic makefile for a library
# Last edited on 2010-05-25 02:39:24 by stolfi

######################################################################
# To be included by the top-level Makefile in the library's directory.
# The caller must define
#  
#   ${LIBNAME}  name of library (WITHOUT the directory or the ".a")
#
# and may also define
#  
#   ${IGNORE} a list of ".h" or ".c" files that should not be compiled or installed.
#   ${OTHER_I_FLAGS}  additional -I flags for C compilation and dependency checks.
#   ${EXTRA_C_FLAGS} additional flags to compile C source files.

ifneq "/${LIBNAME}" "/"

# ----------------------------------------------------------------------
# Package components and directories

# Library file name:
LIBFILE = ${LIBNAME}.a

# Get list of sources:
HFILES := ${filter-out ${IGNORE}, ${wildcard *.h}}
CFILES := ${filter-out ${IGNORE}, ${wildcard *.c}}

# Derived object files:
HOFILES := ${subst .h,.ho,${HFILES}}
OFILES := ${subst .c,.o,${CFILES}}

# Directory with test programs and runs thereof:
TEST_DIR := tests

# ----------------------------------------------------------------------
# Installation directories

# Where to install the library:
INST_PUB_DIR := ${STOLFIHOME}
INST_INC_DIR := ${INST_PUB_DIR}/include
INST_LIB_DIR := ${INST_PUB_DIR}/lib/${PLATFORM}

# ----------------------------------------------------------------------
# Locations of imported packages

# Location of Stolfi's headers:
JS_PUB_DIR := ${STOLFIHOME}
JS_INC_DIR := ${JS_PUB_DIR}/include

# Location of X11 headers and libraries (including ICE):
X11_INC_DIR := /usr/include/X11

# ----------------------------------------------------------------------
# Optional include directories

# Get OpenGL libraries is requested:
ifeq "/${USE_GL}" "/YES" 
  GL_I_FLAGS := \
    -I ${JS_INC_DIR|/GL \
    -I /usr/include/GL
else
  GL_I_FLAGS := 
endif

# Get X11 libraries and headers if requested:
ifeq "/${USE_X11}" "/YES" 
  X11_I_FLAGS := -I${X11_INC_DIR}
else
  X11_I_FLAGS := 
endif
  
#---------------------------------------------------------------------
# Phony "make" targets:

.PHONY:: \
  depend depend-lib depend-tests \
  all actions \
  uninstall \
  build build-lib do-build-lib build-tests \
  check \
  install \
  tar-export tar-save \
  clean clean-lib clean-tests

#---------------------------------------------------------------------
# Default target: 

all: uninstall build-lib install build-tests

#---------------------------------------------------------------------
# "make actions" performs the actions listed in ${ACTIONS}, if any

actions:
ifeq "/${ACTIONS}" "/"
	@echo '$${ACTIONS} is empty'
else
	${MAKE} ${ACTIONS}
endif

#---------------------------------------------------------------------
# Directories to search for #include files:

I_FLAGS := \
  -I. \
  -I${JS_INC_DIR} \
  ${GL_I_FLAGS} \
  ${X11_I_FLAGS} \
  ${OTHER_I_FLAGS}

#---------------------------------------------------------------------
# "make depend" recreates the source dependency file ${DEPFILE}.

DEPFILE := Deps.make

depend: depend-lib depend-tests

depend-lib: ${DEPFILE}

${DEPFILE}: extract-ho-deps ${HFILES} ${CFILES}
	@/bin/rm -f ${DEPFILE}
	@./extract-ho-deps ${I_FLAGS} -I/usr/include \
            ${HFILES} ${CFILES} \
          | egrep -v ': /usr/include' \
          > ${DEPFILE}

depend-tests:
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && ${MAKE} depend ) fi

# ----------------------------------------------------------------------
# Create a local symlink for {extract-ho-deps}, for {tar-export}'s sake

extract-ho-deps: ${STOLFIHOME}/bin/extract-ho-deps
	ln -s ${STOLFIHOME}/bin/extract-ho-deps

#---------------------------------------------------------------------
# "make build" and "make build-lib" make sure that the ${DEPFILE}
# is up to date, and then recreate the object files from the source files
# as necessary.  "make build" and "make build-tests" also rebuild
# the test programs.

build: build-lib build-tests
  
build-lib: ${DEPFILE}
	${MAKE} do-build-lib
        
do-build-lib: ${LIBFILE}

CC := /usr/bin/gcc

C_FLAGS := \
  -Wall -Wundef \
  -Wpointer-arith \
  -Wmissing-prototypes \
  -ggdb \
  -fpcc-struct-return \
  -ffloat-store \
  -frounding-math

%.o: %.c
	${CC} -c ${C_FLAGS} ${EXTRA_C_FLAGS} ${I_FLAGS} $*.c
  
%.ho: %.h
	${CC} -o $*.ho -c ${C_FLAGS} ${EXTRA_C_FLAGS} ${I_FLAGS} -x c $*.h \
          || /bin/rm -f $*.ho
  
${LIBFILE}: ${HOFILES} ${OFILES}
	-rm -f $*.a
	ar crv $*.a ${OFILES}
	ranlib $*.a

build-tests: ${LIBFILE}
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && ${MAKE} build ) fi

ifneq "/${wildcard ${DEPFILE}}" "/"
  # Include specific dependencies extracted by "make depend"
  include ${DEPFILE}
endif

#---------------------------------------------------------------------
# "make install" copies program and makefile to the public dir.

install: ${HFILES} ${HOFILES} ${LIBFILE}
	cp -dpu ${HFILES} ${INST_INC_DIR}
	cp -dpu ${HOFILES} ${INST_INC_DIR}
	cp -dpu ${LIBFILE} ${INST_LIB_DIR}

#---------------------------------------------------------------------
# "make uninstall" deletes the exported program and makefile:

uninstall:
	( cd ${INST_INC_DIR}/. && rm -f ${HFILES} ${HOFILES} )
	( cd ${INST_LIB_DIR}/. && rm -f ${LIBFILE} )
  
#---------------------------------------------------------------------
# "make tar-export" creates a date-stamped tarball of sources and library.
# Test Makefiles and sources are included, as well as any files
# named '*-in.*'.  Beware that scripts and other test data files
# are NOT included!

NOW := ${shell date '+%Y-%m-%d-%H%M%S'}
TARBALL_FILE := ${LIBNAME}-${NOW}.tgz
  
tar-export: ${LIBFILE} extract-ho-deps
	@echo "archiving to ${TARBALL_FILE} ..."
	tar -cvzf ${TARBALL_FILE} \
          00-README Makefile \
            extract-ho-deps \
            ${HFILES} ${CFILES} \
            ${LIBFILE} \
            `find ${TEST_DIR} \( -name 'Makefile' -o -name '*.[hcf]' -o -name '*-in.*' \) -print`

#---------------------------------------------------------------------
# "make tar-save" creates a date-stamped tarball of all files except
# edit backups and derived files.

SAVEDIR := SAVE
SAVETARFILE := ${LIBNAME}-${NOW}-save.tgz
  
tar-save: clean
	tar -cvzf ${SAVEDIR}/${SAVETARFILE} \
          `find ./ -type f -print | egrep -v -e '[~]$'`

#---------------------------------------------------------------------
# "make check" runs the tests in the ${TEST_DIR} subdirectory.

check:
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && ${MAKE} all ) fi

#---------------------------------------------------------------------
# "make clean" deletes all derived files.

clean: clean-lib clean-tests

clean-lib:
	-/bin/rm -f *.o *.ho *.a core
	if [[ -d ${TEST_DIR} ]]; then ( cd ${TEST_DIR} && ${MAKE} clean ) fi
	touch --date='Jan 01 1970' ${PROG}.deps
	${MAKE} depend
  
clean-tests:
	cd ${TEST_DIR} && make clean
  
endif
# End of ${LIBNAME} section.
######################################################################
