#define PROG_NAME "pgmwiggle"
#define PROG_DESC "mix uniform random noise into a PGM file"
#define PROG_VERS "1.0"

/* Copyright  1996 by the State University of Campinas (UNICAMP).
** See the copyright, authorship, and warranty notice at end of file.
** Last edited on 2008-05-25 03:01:32 by stolfi
*/

/* !!! Handle bad pixel values as in {pgmwfilter}/{pnmwfilter}. !!! */

#define PROG_HELP \
  PROG_NAME " \\\n" \
  "  [ -seed {SEED} ] {AMOUNT} \\\n" \
  "  [ < ] {INFILE}.pgm > {OUTFILE}.pgm\n" \
  "The AMOUNT should be a number in [0.0 _ 1.0].\n" \
  "Value 0.0 means no change, 1.0 means pure noise in [0 .. maxval]."

#define PROG_INFO \
  "NAME\n" \
  "  " PROG_NAME " - " PROG_DESC "\n" \
  "\n" \
  "SYNOPSIS\n" \
  "  " PROG_HELP "\n" \
  "\n" \
  "DESCRIPTION\n" \
  "  The program reads a portable graymap as input, mixes" \
  " it with a specified amount of white noise, and writes it out.\n" \
  "\n" \
  "  The fraction of noise in the output is the {AMOUNT} parameter" \
  " (a real number between 0 and 1). More precisely, the output" \
  " image is {1-AMOUNT} times the input, plus {AMOUNT} times a random" \
  " noise uniformly distributed between 0 and the input's {maxval}.\n" \
  "\n" \
  "  The output image will have the same size and depth as the input.\n" \
  "\n" \
  "  The noise is derived from the random(3C) generator.\n" \
  "\n" \
  "OPTIONS\n" \
  "  -seed {SEED}\n" \
  "    This option causes the randm number generator to" \
  " be initialized with the specified seed. The default" \
  " is a fixed internal seed.\n" \
  "\n" \
  "DOCUMENTATION OPTIONS\n" \
  argparser_help_info_HELP_INFO "\n" \
  "\n" \
  "SEE ALSO\n" \
  "  pgmnoise(1).\n" \
  "\n" \
  "AUTHOR\n" \
  "  The program was created on 1996-11-21 by J. Stolfi, UNICAMP.\n" \
  "\n" \
  "MODIFICATION HISTORY\n" \
  "  2006-11-20 J.Stolfi, IC-UNICAMP: rewrote using Stolfi's PNM library.\n" \
  "\n" \
  "WARRANTY\n" \
  argparser_help_info_NO_WARRANTY "\n" \
  "\n" \
  "RIGHTS\n" \
  "  Copyright  1996 by the State University of Campinas (UNICAMP).\n" \
  "\n" \
  argparser_help_info_STANDARD_RIGHTS

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <bool.h> 
#include <jsrandom.h> 
#include <jspnm.h> 
#include <jsfile.h> 
#include <jspnm_image.h> 
#include <argparser.h> 

typedef struct options_t 
  { char *imgname;
    double amount;
    int seed;
  } options_t;

/* INTERNAL PROTOTYPES */

int main(int argc, char* argv[]);
options_t *parse_options(int argc, char **argv);

/* ROUTINES */
 
int main(int argc, char* argv[])
  {
    options_t *o = parse_options(argc, argv);

    FILE *infile = open_read(o->imgname, TRUE);
 
    /* Read input image header: */
    int chns, cols, rows;
    pnm_sample_t imaxval;
    bool_t iraw, ibits;
    pnm_format_t iformat;
    pnm_read_header(infile, &cols, &rows, &chns, &imaxval, &iraw, &ibits, &iformat);
    
    /* Choose output format and write output image header: */
    pnm_sample_t omaxval = (imaxval < 255 ? 255 : imaxval);
    bool_t oraw, obits;
    pnm_format_t oformat;
    pnm_choose_output_format(omaxval, chns, FALSE, &oformat, &oraw, &obits);
    pnm_write_header(stdout, cols, rows, omaxval, oformat);
    
    pnm_sample_t *ipix = pnm_image_alloc_pixel_row(cols, chns);
    pnm_sample_t *opix = pnm_image_alloc_pixel_row(cols, chns);
    
    int x, y;
    double amount = o->amount, tnuoma = 1 - amount;
    srandom(o->seed);
    for (y = rows-1; y >= 0; y--)
      { pnm_read_pixels(infile, ipix, cols, chns, imaxval, iraw, ibits);
        for (x = 0; x < cols; x++)
          { double iv = (ipix[x] + 0.5)/((double)imaxval + 1.0);
            double ov = tnuoma*iv + amount*drandom();
            opix[x] = pnm_quantize(ov, omaxval, PNM_NO_BADVAL);
          }
        pnm_write_pixels(stdout, opix, cols, chns, omaxval, oraw, obits);
      }
    fclose(infile);
    fclose(stdout);
    return(0);
  } 

options_t *parse_options(int argc, char **argv)
  {
    argparser_t *pp = argparser_new(stderr, argc, argv);
    argparser_set_help(pp, PROG_NAME " version " PROG_VERS ", usage:\n" PROG_HELP);
    argparser_set_info(pp, PROG_INFO);
    argparser_process_help_info_options(pp);
    
    options_t *o = (options_t *)notnull(malloc(sizeof(options_t)), "no mem");
    
    o->imgname = NULL;
    
    
    if (argparser_keyword_present(pp, "-seed"))
      { o->seed = argparser_get_next_int(pp, 1, INT_MAX); }
    else
      { o->seed = 46150417; }


    argparser_skip_parsed(pp);
    o->amount = argparser_get_next_double(pp, 0.0, 1.0);
    
    if (argparser_next(pp) != NULL)
      { o->imgname = argparser_get_next(pp); }
    else
      { o->imgname = "-"; }
    
    argparser_finish(pp);
    return o;
  }

/* COPYRIGHT, AUTHORSHIP, AND WARRANTY NOTICE:
**
**   Copyright  1996 by the State University of Campinas (UNICAMP).
**
** Created on 21-11-1996 by Jorge Stolfi, IC-UNICAMP.       
**
** Permission to use, copy, modify, and redistribute this software and
** its documentation for any purpose and without fee is hereby
** granted, provided that: (1) the copyright notice at the top of this
** file and this copyright, authorship, and warranty notice is retained
** in all derived source files and documentation; (2) no executable
** code derived from this file is published or distributed without the
** corresponding source code; and (3) these same rights are granted to
** any recipient of such code, under the same conditions.
** This software is provided "as is", WITHOUT ANY EXPLICIT OR IMPLICIT
** WARRANTIES, not even the implied warranties of merchantibility and
** fitness for a particular purpose. END OF NOTICE.
*/
