/* See ellipse_aligned.h */
/* Last edited on 2009-02-28 13:27:13 by stolfi */

#define _GNU_SOURCE
#include <stdio.h>
#include <math.h>
#include <values.h>
#include <stdlib.h>
#include <assert.h>

#include <r2.h>
#include <r2x2.h>
#include <interval.h>
#include <affirm.h>

#include <ellipse_aligned.h>

#define Pr fprintf
#define Er stderr
  
void ellipse_aligned_bbox(double a, double b, interval_t bbox[])
  { LO(bbox[0]) = - a;
    HI(bbox[0]) = + a;
    LO(bbox[1]) = - b;
    HI(bbox[1]) = + b;
  }

void ellipse_aligned_int_bbox
  ( double a, double b, 
    double mrg, /* Extra margin. */
    int *xLoP,  /* (OUT) Min X of clip area. */
    int *xHiP,  /* (OUT) Max X of clip area. */
    int *yLoP,  /* (OUT) Min Y of clip area. */
    int *yHiP   /* (OUT) Max Y of clip area. */
  )
  {
    demand(mrg >= 0, "the extra margin {mrg} must be non-negative");
    (*xLoP) = (int)floor(- a - mrg);
    (*xHiP) = - (*xLoP);
    (*yLoP) = (int)floor(- b - mrg);
    (*yHiP) = - (*xHiP);
  }

double ellipse_aligned_position(double a, double b, r2_t *p, r2_t *csp)
  {
    assert(a >= b); assert(b >= 0);
    
    /* Grab the coordinates of {p}: */
    double xp = p->c[0];
    double yp = p->c[1];
    
    if (a == 0)
      { /* Ellipse is a point: */
        assert(b == 0);
        if ((xp == 0) && (yp == 0))
          { if (csp != NULL) { (*csp) = (r2_t){{ NAN, NAN }}; }
            return 1.0;
          }
        else
          { if (csp != NULL)
              { double rp = hypot(xp,yp); 
                (*csp) = (r2_t){{ xp/rp, yp/rp }};
              }
            return +INF;
          }
      }
    else if (b == 0)
      { /* Ellipse is a line segment: */
        if (yp == 0)
          { /* Point is on the major axis: */
            if (fabs(xp) < a)
              { /* Point is on the ellipse's boundary: */
                if (csp != NULL) 
                  { double cp = xp/a; 
                    (*csp) = (r2_t){{ cp, NAN }};
                  }
                return 1.0;
              }
            else
              { /* Point is outside the ellipse: */
                if (csp != NULL) 
                  { double cp = (xp > 0 ? +1.0 : -1.0); 
                    (*csp) = (r2_t){{ cp, 0.0 }};
                  }
                return fabs(xp)/a;
              }
          }
        else if (xp == 0)
          { /* Point is on the Y axis, not at the origin: */
            if (csp != NULL) 
              { double sp = (yp > 0 ? +1.0 : -1.0); 
                (*csp) = (r2_t){{ 0.0, sp }};
              }
            return +INF;
          }
        else
          { /* Point is anywhere else: */
            if (csp != NULL) { (*csp) = (r2_t){{ NAN, NAN }}; }
            return +INF;
          }
      }
    else
      { /* Compute the canonical coords {ap,bp} of {p}: */
        double ap = xp/a;
        double bp = yp/b;
        /* Compute the canonical radius {rp} of {p}: */
        double rp = hypot(ap, bp);
        if (csp != NULL) { (*csp) = (r2_t){{ ap/rp, bp/rp }}; }
        return rp;
      }
  }
  
double ellipse_aligned_nearest_point(double a, double b, r2_t *p, r2_t *q)
  {
    bool_t debug = FALSE;
    /* !!! Delete excess debugging printouts. !!! */

    assert(a >= b); assert(b >= 0);
    if (debug) { Pr(Er, "  a = %13.8f  b = %13.8f\n", a, b); }

    /* Grab the coordinates of {p}: */
    double xp = p->c[0];
    double yp = p->c[1];
    if (debug) { Pr(Er, "  p rel = [ %13.8f %13.8f ]\n", xp, yp); }
    
    if (a == 0)
      { assert(b == 0); /* Since {0 <= b <= a}. */
        /* Ellipse is essentially a point: */
        if (q != NULL) { (*q) = (r2_t){{ 0, 0 }}; }
        return r2_dist(p, q);
      }
   
    if ((a - b) == 0)
      { /* Ellipse is essentially a circle: */
        if ((xp == 0) && (yp == 0))
          { /* {p} is at the center: */
            if (q != NULL) { (*q) = (r2_t){{ a, 0 }}; }
            return -a;
          }
        else
          { /* {p} is eccentric: */
            double rp = r2_norm(p);
            if (q != NULL) { r2_scale(a/rp, p, q); }
            return rp - a;
          }
      }
    
    if (xp == 0)
      { /* Point {p} is on the Y axis; nearest pt is {sgn(yp)*v*b}. */
        double yq = (yp >= 0 ? b : -b);
        if (q != NULL) { (*q) = (r2_t){{ 0, yq }}; }
        return fabs(yp) - b;
      }

    /* Compute the aspect ratio {S} of the ellipse: */  
    assert(a > 0);
    double S = b/a;

    if (debug) { Pr(Er, "  S     = %13.8f\n", S); }
    
    if (S == 0)
      { /* Ellipse is essentially a segment in the {u} direction: */
        double xq, dpq;
        if (xp > +a)
          { xq = +a; dpq = hypot(xp - xq, yp); }
        else if (xp < -a) 
          { xq = -a; dpq = hypot(xp - xq, yp); }
        else
          { xq = xp; dpq = fabs(yp); }
        if (q != NULL) { (*q) = (r2_t){{ xq, 0 }}; }
        return dpq;
      }
      
    /* Compute the slope {T} of {p}: */
    double T = yp/xp;
    
    if (debug) { Pr(Er, "  T     = %13.8f\n", T); }
    
    if (T == 0)
      { /* Point {p} is essentially on the major axis. */
        double e = 1 - S*S;
        double xq = xp/e;
        if (debug) { Pr(Er, "  xq    = %13.8f\n", xq); }
        if (fabs(xq) >= a)
          { /* Point {p} is beyond the focus. */
            /* The nearest point is {ctr  a*u}: */
            xq = (xp >= 0 ? +a : -a);
            if (q != NULL) { (*q) = (r2_t) {{ xq, 0 }}; }
            return fabs(xp) - a;    
          }
        else
          { /* Point {p} is between the foci. */
            double cq = xq/a;
            assert(cq <= 1);
            double yq = b*sqrt(1 - cq*cq);
            if (debug) { Pr(Er, "  yq    = %13.8f\n", yq); }
            if (q != NULL) { (*q) = (r2_t){{ xq, yq }}; }
            return - hypot(xq - xp, yq);
          }
      }
      
    /* Ellipse has non-empty interior, and {p} is not on the axes. */
            
    /* Compute the adimensional parameters {A,B} for 1st quadrant: */
    assert(S > 0);
    assert(S <= 1);
    assert(isfinite(T));
    double A = S*fabs(T); 
    assert(! isnan(A));
    
    double d2 = ((a - b)/a)*(a + b);
    assert(! isnan(d2));
    assert(isfinite(xp));
    assert(xp != 0);
    double B = d2/fabs(xp);
    assert(! isnan(B));
    
    if (debug) { Pr(Er, "  A     = %13.8f\n", A); }
    if (debug) { Pr(Er, "  B     = %13.8f\n", B); }
    
    /* Solve the characteristic polynomial for {t}: */
    double t = ellipse_aligned_compute_t(A, B);
    
    if (debug) { Pr(Er, "  t sol = %13.8f\n", t); }
    
    assert(t >= 0.0);
    assert(t <= 1.0);

    /* Compute the cosine and sine {ct,st} of the angular argument of {q}: */
    double dt = 1 + t*t; 
    double ct = (1 - t*t)/dt; 
    double st = 2*t/dt;

    /* Compute the {u,v} coords of {q}: */
    double xq = (xp >= 0 ? +1 : -1)*a*ct; 
    double yq = (yp >= 0 ? +1 : -1)*b*st; 

    /* Compute {q} and the distance: */
    if (q != NULL) { (*q) = (r2_t){{ xq, yq }}; }
    double dpq = hypot(xp - xq, yp - yq);
    
    /* Return with correct sign: */
    if ((xp > a) || (yp > b)) { /* {p} is definitely outside: */ return dpq; }
    assert(a > 0);
    assert(b > 0);
    
    /* Compute the canonical coords {ap,bp} of {p}: */
    double ap = xp/a;
    double bp = yp/b;
    return (ap*ap + bp*bp < 1.0 ? -dpq : +dpq);
  }
  
double ellipse_aligned_compute_t(double A, double B)
  {
    bool_t debug = FALSE;
    /* !!! Delete excess debugging printouts. !!! */

    /* The polynomial and its derivatives: */
    /* {P(t)    = A*(1+t**2)*(1-t**2) - 2*t*((B+1)*t**2 - (B-1))} */
    /* {P'(t)   = -2*((2*A*t + 3*(B+1))*t**2 - (B-1))} */
    /* {P''(t)  = -12*t*(A*t + (B+1))} */
    /* {P'''(t) = -12*(2*A*t + (B+1))} */
    
    demand(A >= 0, "{A} must be non-negative");
    demand(B >= 0, "{B} must be non-negative");
    
    if (A == 0)
      { /* {P(t)} reduces to {2*t*((B+1)*t**2 - (B-1))}. */
        /* The roots in {[0:1]} are {t=0} and {t=sqrt((B-1)/(B+1))}. */
        if (B-1 <= 0)
          { return 0; }
        else
          { return sqrt((B-1)/(B+1)); }
      }
    
    /* Since {P(0) == A > 0} and {P(1) == -4 < 0}, there is a root in {[0:1]}. */
    /* Solve {P(t)==0} by Newton-Raphson. */
    /* Since {P''(t) <= 0} in {[0:1]}, we can use {t0==1} as starting guess. */
    /* !!! Should use a quadratic approximation to get a good {t0}. !!! */
    /* !!! Or even use a quadratic Newton, since {P'' < 0} and {P''' < 0}. !!! */
    
    double t = 1.0;
    double Bp = B + 1;
    double Bm = B - 1;
    int nIts = 0;  /* Counts iterations. */
    if (debug) { Pr(Er, "  t[%d] = %13.8f\n", nIts, t); }
    while (TRUE)
      { /* Compute {P(t)}: */
        double t2 = t*t;
        double Pt = A*(1+t2)*(1-t2) - 2*t*(Bp*t2 - Bm);
        if (debug) { Pr(Er, "  P[%d] = %13.8f\n", nIts, Pt); }
        if (Pt >= 0) { /* We must have reached or passed the root: */ break; }
        
        /* Compute {P'(t)}: */
        double Dt = -2*((2*A*t + 3*Bp)*t2 - Bm);
        if (debug) { Pr(Er, "  D[%d] = %13.8f\n", nIts, Dt); }
        if (Dt >= 0) { /* We must have reached or passed the root: */ break; }
        
        /* Compute the new {t}: */
        double dt = -Pt/Dt;
        if (debug) { Pr(Er, "  d[%d] = %13.8f\n", nIts, dt); }
        
        assert(dt <= 0);
        double ot = t;
        t += dt;
        nIts++;
        if (debug) { Pr(Er, "  t[%d] = %13.8f\n", nIts, t); }
        if (t < 0) { /* Must be a very small positive root: */ t = 0; break; }
        if (t >= ot) { /* We cannot progress any further: */ break; }
        demand(nIts <= 1000, "did not converge in 1000 iterations");
        if (debug) { Pr(Er, "\n"); }
      }
    if (debug) { Pr(Er, "  t fin = %13.8f\n", t); }
    if (nIts > 25) { Pr(Er, "%s required %d iterations\n", __FUNCTION__, nIts); }
    if (debug) { Pr(Er, "\n"); }
    return t;    
  }
  
bool_t ellipse_aligned_inside(double a, double b, r2_t *p)
  {
    assert(a >= b); assert(b >= 0);
    bool_t debug = FALSE;

    if (debug) { Pr(Er, "  a = %13.8f  b = %13.8f\n", a, b); }

    /* Grab the coordinates of {p}: */
    double xp = p->c[0];
    double yp = p->c[1];
    if (debug) { Pr(Er, "  p rel = [ %13.8f %13.8f ]\n", xp, yp); }
    
    if (b == 0)
      { /* Ellipse is a point or line segment: */
        return FALSE;
      }
    
    if (a == b)
      { /* Ellipse is a circle: */
        return (r2_norm(p) <= a);
      }

    /* Compute the canonical coords {ap,bp} of {p} in the system: */
    double ap = xp/a;
    double bp = yp/b;

    /* Test the implicit equation: */
    return ap*ap + bp*bp <= 1.0;
  }

double ellipse_aligned_border_position(double a, double b, double hwd, r2_t *p)
  {
    bool_t debug = FALSE;
    assert(a >= b); assert(b >= 0);
    
    if (debug) { Pr(Er, "  a = %13.8f  b = %13.8f\n", a, b); }

    /* Grab the coordinates of {p}: */
    double xp = p->c[0];
    double yp = p->c[1];
    if (debug) { Pr(Er, "  p rel = [ %13.8f %13.8f ]\n", xp, yp); }
    
    if (a == b)
      { /* Ellipse is a circle: */
        return fmin(+1.0, fmax(-1.0, (r2_norm(p) - a)/hwd));
      }
    
    /* Check bounding rectangle: */
    if ((fabs(xp) >= a + hwd) || (fabs(yp) >= b + hwd)) { return +1.0; }
    
    if (b == 0)
      { /* Ellipse is a line segment: */
        if (fabs(xp) <= a)
          { /* Straight part of stroke: */
            return fmin(1.0, fabs(yp)/hwd);
          }
        else
          { /* Round end caps: */
            double xq = (xp > 0 ? +a : -a);
            return fmin(1.0, hypot(xp - xq, yp)/hwd);
          }
      }

    /* Let {Q} be the upright square centered at {p} with side {2*hwd}. */
    /* Test the outermost corner of {Q}: */
    double aoc = (fabs(xp) + hwd)/a;
    double boc = (fabs(yp) + hwd)/b;
    if (aoc*aoc + boc*boc < 1.0) 
      { /* {Q} is inside the ellipse, so {p} is well inside: */
        return -1.0;
      }
    /* Test the innermost corner of {Q}, clipped to the quadrant: */
    double aic = fmax(0.0, (fabs(xp) - hwd))/a;
    double bic = fmax(0.0, (fabs(yp) - hwd))/b;
    if (aic*aic + bic*bic > 1.0) 
      { /* {Q} is outside the ellipse, so {p} is well outside: */
        return +1.0;
      }
    /* We are close enough to the boundary. */
    /* Cmpute the exact signe distance {d} to the boundary: */
    double d = ellipse_aligned_nearest_point(a, b, p, NULL);
    return fmax(-1.0, fmin(+1.0, d/hwd));
  }

void ellipse_aligned_print(FILE *wr, double a, double b, char *fmt)
  { fprintf(wr, "{ a: ");
    fprintf(wr, fmt, a);
    fprintf(wr, " b: ");
    fprintf(wr, fmt, b);
    fprintf(wr, " }");
    fflush(wr);
  }
