#ifndef ellipse_aligned_H
#define ellipse_aligned_H

/* ellipse_aligned.h -- tools for axis-aligned, origin-centered ellipses. */
/* Last edited on 2009-02-25 13:34:14 by stolfi */

#include <r2.h>
#include <sign.h>
#include <interval.h>

/* ELLIPSE GEOMETRY

  This moduel has tools for axis-aligned ellipses centered at the origin
  whose longest diameter is in the X direction.
  
  Such an ellipse is defined by the major semidiameter (X radius) {a}
  and the minor semidiameter (Y radius) {b}, always with {a >= b >=
  0}.

  The ellipse is defined by the parametric equation {G(th) =
  (a*cos(th), b*sin(th))}, where {th} is the /angular argument/ of the
  ellipse, varying in {[0:2*PI]}.

  Alternatively, a point {p=(x,y)} is inside, on the boundary of, or
  outside the ellipse depending on whether {G(x,y) = (x/a)^2 + (y/b)^2
  - 1} is negative, zero, or positive.

  The /canonical coordinates/ of a point {p=(x,y)} are the 
  quantities {(x/a,y/b)}.  In the canonical coordinates,
  the ellipse is the unit origin-centered circle. */
  
/* !!! We should remove the restriction {a >= b}. !!! */

void ellipse_aligned_bbox(double a, double b, interval_t bbox[]);
  /* Stores in {bbox[0..1]} a bounding box for the ellipse {G}. */

void ellipse_aligned_int_bbox
  ( double a, double b,
    double mrg, /* Extra margin. */
    int *xLoP,  /* (OUT) Min X of clip area. */
    int *xHiP,  /* (OUT) Max X of clip area. */
    int *yLoP,  /* (OUT) Min Y of clip area. */
    int *yHiP   /* (OUT) Max Y of clip area. */
  );
  /* Stores in {*xLoP,*xHiP,*yLoP,*yHiP} a bounding box for the
    ellipse with radii {a,b} with integer coordinates. The box is at least {mrg}
    units away from the ellipse, on each side. */

bool_t ellipse_aligned_inside(double a, double b, r2_t *p);
  /* Returns TRUE if {p} is inside the ellipse with radii {a,b}, FALSE
    if outside. Beware of roundoff errors. */

double ellipse_aligned_position(double a, double b, r2_t *p, r2_t *csp);
  /* Returns the radial and angular position of {p} relative to the
    ellipse with radii {a,b}. 
    
    The returned result is the position of {p} relative to the
    ellipse's boundary, along the ray from the origin to {p}.
    This value is less than 1 if {p} is inside, and greater than 1 if {p}
    is outside. It is 0 if {p} is at the origin, and 1 if it is on the
    boundary. 
    
    If {csp} is not NULL, the procedure also stores in {*csp} the
    cosine and sine of the angular argument of {p}, measured from the
    X axis. */

double ellipse_aligned_nearest_point(double a, double b, r2_t *p, r2_t *q);
  /* Fids the signed distance from {p} to the boundary of the ellipse
    with radii {a,b}.
    
    If {q} is not NULL, the procedure also stores in {*q} the point on
    the boundary of the ellipse with radii {a,b} that is closest to {p}. 
    
    The return value {+dist(p,q)} if {p} is outside, and {-dist(p,q)}
    if {p} is inside. */

double ellipse_aligned_border_position(double a, double b, double hwd, r2_t *p);
  /* If the distance from {p} to the boundary of the ellipse with
    radii {a,b} is less than {hwd}, returns the signed distance
    (positive outside, negative inside) divided by {hwd}. Otherwise
    returns {+1} if {p} is outside the ellipse, {-1} if it is inside.
    
    This is the relative transversal position of {p} in the strip
    created when the ellipse's boundary is stroked with a round brush
    of radius {hwd}. This procedure is much faster than
    {ellipse_aligned_nearest_point} when {p} is far from the
    stroked region. */

void ellipse_aligned_print(FILE *wr, double a, double b, char *fmt);
  /* Writes to {wr} the ellipse radii {a,b}.  Each parameter
    is printed with format {fmt} (which should be appropriate 
    for a {double} value). */

/* SPECIALIZED TOOLS */

double ellipse_aligned_compute_t(double A, double B);
  /* Solves the polynomial equation {P(t) == 0} that occurs in the
    nearest-point computation, where
    
      { P(t) = A*(1+t**2)*(1-t**2) - 2*t*((B+1)*t**2 - (B-1)) }
      
    Requires {A >= 0, B >= 0}.  The result is in {[0 _ 1]}. */

#endif
 
