/* See r2x2.h. */
/* Last edited on 2009-01-05 16:35:10 by stolfi */

#include <r2x2.h>
#include <r2.h>
#include <math.h>
#include <stdio.h>
#include <rmxn.h>

#define N 2

void r2x2_zero(r2x2_t *M)
  { M->c[0][0] = 0.0;
    M->c[0][1] = 0.0;
    M->c[1][0] = 0.0;
    M->c[1][1] = 0.0;
  }

void r2x2_ident(r2x2_t *M)
  { M->c[0][0] = 1.0;
    M->c[0][1] = 0.0;
    M->c[1][0] = 0.0;
    M->c[1][1] = 1.0;
  }
    
void r2x2_transp (r2x2_t *A, r2x2_t *M)
  {
    double a, b;
    /* Copy diagonal elements: */
    M->c[0][0] = A->c[0][0];
    M->c[1][1] = A->c[1][1];
    /* Copy remaining elements, transposed. */
    a = A->c[0][1]; b = A->c[1][0]; M->c[0][1] = b; M->c[1][0] = a;
  }

void r2x2_map_row (r2_t *x, r2x2_t *A, r2_t *r)
  { r2_t rr;
    rr.c[0] = x->c[0] * A->c[0][0] + x->c[1] * A->c[1][0];
    rr.c[1] = x->c[0] * A->c[0][1] + x->c[1] * A->c[1][1];
    
    r->c[0] = rr.c[0];
    r->c[1] = rr.c[1];
  }

void r2x2_map_col (r2x2_t *A, r2_t *x, r2_t *r)
  { r2_t rr;
    rr.c[0] = A->c[0][0] * x->c[0] + A->c[0][1] * x->c[1];
    rr.c[1] = A->c[1][0] * x->c[0] + A->c[1][1] * x->c[1];
    
    r->c[0] = rr.c[0];
    r->c[1] = rr.c[1];
  }
  
void r2x2_scale (double s, r2x2_t *A, r2x2_t *M)  
  { M->c[0][0] = s * A->c[0][0];
    M->c[0][1] = s * A->c[0][1];
    M->c[1][0] = s * A->c[1][0];
    M->c[1][1] = s * A->c[1][1];
  }

void r2x2_mul (r2x2_t *A, r2x2_t *B, r2x2_t *M)
  { r2x2_t RR;
    RR.c[0][0] = A->c[0][0]*B->c[0][0] + A->c[0][1]*B->c[1][0];
    RR.c[0][1] = A->c[0][0]*B->c[0][1] + A->c[0][1]*B->c[1][1];
    RR.c[1][0] = A->c[1][0]*B->c[0][0] + A->c[1][1]*B->c[1][0];
    RR.c[1][1] = A->c[1][0]*B->c[0][1] + A->c[1][1]*B->c[1][1];
    (*M) = RR;
  }

void r2x2_mul_tr (r2x2_t *A, r2x2_t *B, r2x2_t *M)
  {
    r2x2_t RR;
    RR.c[0][0] = A->c[0][0]*B->c[0][0] + A->c[0][1]*B->c[0][1];
    RR.c[0][1] = A->c[0][0]*B->c[1][0] + A->c[0][1]*B->c[1][1];
    RR.c[1][0] = A->c[1][0]*B->c[0][0] + A->c[1][1]*B->c[0][1];
    RR.c[1][1] = A->c[1][0]*B->c[1][0] + A->c[1][1]*B->c[1][1];
    (*M) = RR;
  }

double r2x2_det (r2x2_t *A)
  { return A->c[0][0] * A->c[1][1] - A->c[1][0] * A->c[0][1]; }

void r2x2_adj (r2x2_t *A, r2x2_t *M)
  { r2x2_t RR;

    RR.c[0][0] =   A->c[1][1];
    RR.c[0][1] = - A->c[0][1];
    RR.c[1][0] = - A->c[1][0];
    RR.c[1][1] =   A->c[0][0];
    
    (*M) = RR;
  }

void r2x2_inv (r2x2_t *A, r2x2_t *M)
  { double d = A->c[0][0] * A->c[1][1] - A->c[1][0] * A->c[0][1];
    r2x2_t RR;

    RR.c[0][0] =   A->c[1][1]/d;
    RR.c[0][1] = - A->c[0][1]/d;
    RR.c[1][0] = - A->c[1][0]/d;
    RR.c[1][1] =   A->c[0][0]/d;
    
    (*M) = RR;
  }

bool_t r2x2_is_unif_scaling(r2x2_t *M, double s)
  {
    int i, j;
    for (i = 0; i < N; i++)
      for (j = 0; j < N; j++)
        { if (M->c[i][j] != (i == j ? s : 0.0)) { return FALSE; } }
    return TRUE;
  }

void r2x2_rot90(r2x2_t *M)
  { M->c[0][0] = 00.0;
    M->c[0][1] = +1.0;
    M->c[1][0] = -1.0;
    M->c[1][1] = 00.0;
  }

void r2x2_sym_eigen(r2x2_t *A, r2_t *e, r2x2_t *M)
  {
    /* Grab the elements of {A}: */
    double A00 = A->c[0][0];
    double A01 = A->c[0][1];
    double A10 = A->c[1][0];
    double A11 = A->c[1][1];
    /* Symmetrize the matrix: */
    double Y = 0.5*A01 + 0.5*A10; /* Off-diagonal element. */
    if (Y == 0)
      { /* Matrix is diagonal -- eigenvalues are {A00} and {A11}: */
        if (A00 >= A11)
          { /* Largest eigenvalue is {A00}, or the two are equal: */
            e->c[0] = A00; e->c[1] = A11;
            /* The corresponding eigenvectors are {(+1,00)} and {(00,+1)}: */
            if (M != NULL) { r2x2_ident(M); }
          }
        else
          { /* Largest eigenvalue is {A11}: */
            e->c[0] = A11; e->c[1] = A00;
            /* The corresponding eigenvectors are {(00,+1)} and {(-1,00)}: */
            if (M != NULL) { r2x2_rot90(M); }
          }
      }
    else
      { /* Get intermediate quantities {T,X}: */
        double T = 0.5*A00 + 0.5*A11; /* Half-trace of matrix. */
        double X = 0.5*A00 - 0.5*A11; /* Half-unbalance of matrix. */
        /* Eigenvalues are {T  sqrt(X^2 + Y^2)}: */
        double Q = hypot(X, Y); /* Square root of discriminant. */
        e->c[0] = T + Q; /* Max eigenvalue. */
        e->c[1] = T - Q; /* Min eigenvalue. */
        /* Compute the eigenvectors if so requested: */
        if (M != NULL)
          { /* assert(Q != 0); */
            /* The eigenvectors are the square roots of {(X/Q + Y/Q*IMAG)}. */
            double x = X/Q, y = Y/Q;
            double c, s;  /* The eigenvector of eigenvalue {T+Q} is {(c,s)}. */
            /* Split into two cases for numerical stability: */
            if (x >= 0)
              { c = sqrt(0.5*(1 + x)); s = 0.5*y/c; }
            else
              { s = sqrt(0.5*(1 - x)); c = 0.5*y/s; }
            /* Store into array with proper signs: */
            /* Fix the sign so that the arg is in {-PI} (exc.) to {+PI} (inc.): */
            if ((c < 0) || ((c == 0) && (s < 0))) { c = -c; s = -s; }
            M->c[0][0] = +c;
            M->c[0][1] = +s;
            M->c[1][0] = -s;
            M->c[1][1] = +c;
          }
      }
  }

void r2x2_print (FILE *f, r2x2_t *A)
  { r2x2_gen_print(f, A, NULL, NULL, NULL, NULL, NULL, NULL, NULL); }

void r2x2_gen_print 
  ( FILE *f, r2x2_t *A,
    char *fmt, 
    char *olp, char *osep, char *orp,
    char *ilp, char *isep, char *irp
  )
  { rmxn_gen_print(f, N, N, &(A->c[0][0]), fmt, olp, osep, orp, ilp, isep, irp); }  
