#ifndef float_image_align_H
#define float_image_align_H

/* Tools for translational alignment between float-valued images. */
/* Last edited on 2009-06-14 15:02:04 by stolfi */ 

#include <bool.h>
#include <r2.h>
#include <i2.h>
#include <float_image.h>

/* 
  DEFINING THE COMPARISON WINDOW
  
  The images to be aligned are compared within a rectangular window
  defined by a center {p[i]} (one per image), two integers {hwx,hwy},
  and two weight tables {wx,xy}.
  
  More precisely, they are compared at a grid of {nwx*nwy} fractional
  sample points {p[i]+(ix,iy)} where {nwx=2*hwx+1}, {nwy=2*hwy+1}, and
  {(ix,iy)} ranges over the integer pairs {{-hwx..+hwx}{-hwy..+hwy}}.
  
  The squared discrepancy between the samples {p[i]+(ix,iy)} for all
  images {i} will be weighted with {wx[hwx+ix]*wy[hwy+iy]}. The arrays
  must have {nwx} and {nwy} elements, respectively. */

typedef double float_image_align_mismatch_t(int ni, int scale, int hwx, double wx[], int hwy, double wy[], r2_t p[]); 
  /* Type of a function that evaluates the mismatch of {ni} images at
    a scale {scale}, by evaluating them at the points described above.  The
    function should return some quadratic measure of the discrepancy
    between these {ni} sampled sub-images. */

void float_image_align_single_scale
  ( int ni,                            /* Number fo images to align. */
    int scale,                         /* Image scale. */  
    float_image_align_mismatch_t *f2,  /* Function that evaluates the mismatch between the images. */
    int hwx,                           /* Half-width of comparison window. */
    double wx[],                       /* Horizontal weights for comparison. */
    int hwy,                           /* Half-height of comparison window. */
    double wy[],                       /* Vertical weights for comparison. */
    r2_t p[],                          /* (IN/OUT) Corresponding points in each image. */
    r2_t adj_rad                       /* Adjustment increments along each axis. */
  );
  /* Adjusts {ni} points {p[0..ni-1]} so that certain {ni} images are
    as similar as possible in the neighborhoords of those points. Uses
    non-linear minimization.
    
    The adjustment finds the alignment points {p[0..ni-1]} that minimize the
    error function 
       { f2(ni,scale,hwx,wx,hwy,wy,p) }
    On input, {p[0..ni-1]} must be a guess for the optimum.  On output,
    {p[0..ni-1]} will be the computed optimum.  The procedure
    only considers displacement sets {p[0..ni-1]} whose sum is the same
    as the sum of all the initial points.
    
    Uses nonlinear minimization. The mismatch function {f2} must
    depend quadratically on {p} in the neighborhood of the minimum. */

void float_image_align_multi_scale
  ( int ni,                            /* Number fo images to align. */
    int scale,                         /* Image scale. */  
    float_image_align_mismatch_t *f2,  /* Function that evaluates the mismatch between the images. */
    r2_t cmp_rad,                      /* Radius of comparison window. */
    r2_t p[],                          /* (IN/OUT) Corresponding points in each image. */
    r2_t adj_rad                       /* Adjustment increments along each axis. */
  );
  /* Similar to {float_image_align_single_scale} but uses multiscale 
    search, starting at a scale {sc0} where {adj_rad} is small enough. */

double float_image_align_rel_disp_sqr(int ni, r2_t p[], r2_t q[], r2_t *r);
  /* Computes the total squared displacement between {p[0..ni-1]} and 
    {q[0..ni-1]} relative to the radius {r}, that is,
    {SUM_{i=0..ni-1,j=0..1} |(p[i].c[j]-q[i].c[j])/r.c[j]|^2 }
    It is useful to add this term to the goal function 
    in order to bias the search towards the neighborhood of the
    initial guess. */

#endif
