#ifndef float_image_geostereo_H
#define float_image_geostereo_H

/* Tools for geometric stereo recosntruction from image pairs. */
/* Last edited on 2009-02-23 20:11:46 by stolfi */ 

#include <stdio.h>

#include <bool.h>
#include <float_image.h>

void fgst_compute_uniscale_displacement_map
  ( float_image_t *f1,  /* Image 1. */
    float_image_t *f2,  /* Image 2. */
    int ncands,         /* Number of candidates to keep. */
    int rx,             /* Window half-width. */
    int ry,             /* Window half-height. */
    int dmin,           /* Minimum signed displacement (pixels). */
    int dmax,           /* Maximum signed displacement (pixels). */
    float_image_t **fd, /* (OUT) Dispmap image. */
    float_image_t **fs  /* (OUT) Scoremap image. */
  );
  /* Stores in {fd[y,x,0..ncands-1]} the best {ncands}
    displacement found centered at pixel {(x,y)}. The displacements
    returned in {fs} are actually integers, in units of 1/3 pixel.
    
    More precisely, if {fd[y,x,k] == d}, then the neighborhood of
    pixel {f1[y,x-d/3]} is similar to that of {f2[y,x+d/3]},
    within a window {[-rx..+rx]  [-ry..ry]} centered at the
    displaced pixel.  Considers only values {d} such that
    {d/3} is in the range {[dmin..dmax]} (pixels).
    
    Also stores in {fs[y,x,0..ncands-1]} the corresponding scores
    (average squared differences). The images {fd,fs} are allocated by
    the procedure; note that they have {ncands} channels. */

void fgst_compute_multiscale_displacement_map
  ( float_image_t *f1,  /* Image 1. */
    float_image_t *f2,  /* Image 2. */
    int nscales,        /* Number of scales to consider (0 = uniscale). */
    int ncands,         /* Number of candidates to keep. */
    int rx,             /* Window half-width. */
    int ry,             /* Window half-height. */
    int dmin,           /* Minimum signed displacement (pixels). */
    int dmax,           /* Maximum signed displacement (pixels). */
    float_image_t **fd, /* (OUT) Dispmap image. */
    float_image_t **fs  /* (OUT) Scoremap image. */
  );
  /* Same as {fgst_compute_uniscale_displacement_map}, but uses a
    multiscale search algorithm with {nscales} levels deep. (If
    {nscales = 0}, uses uniscale search.) */
  
void fgst_refine_and_prune_displacement_map
  ( float_image_t *gd,  /* Displacement map for halfsize images. */
    float_image_t *gs,  /* Score map for halfsize images. */
    float_image_t *f1,  /* Full-size image 1. */
    float_image_t *f2,  /* Full-size image 2. */
    int rx,             /* Window half-width. */
    int ry,             /* Window half-height. */
    int dmin,           /* Minimum signed displacement (pixels). */
    int dmax,           /* Maximum signed displacement (pixels). */
    float_image_t *fd,  /* (OUT) Dispmap for full-size images. */
    float_image_t *fs   /* (OUT) Scoremap for full-size images. */
  );
  /* Arguments {gd,gs} should be the displacements and score maps for
    half-size versions of images {f1,f2}. For each pixel of {f1,f2},
    takes the best {NC} displacements found in {gd} for the
    corresponding point, doubles those displacements (to match the
    scale of {f1,f2}), tries some small adjustments on them (by 1/3
    pixel), and finally stores the results in {fd,fs}.

    Assumes that {fd,fs} have already been allocated. The number {NC}
    is the number of channels of {fd}, which must not be greater than
    that of {gd}. The displacement scores in {fs} will be a
    combination of the rough scores in {gs} and the newly computed
    scores for the adjusted displacements. */

void fgst_local_match
  ( float_image_t *f1,  /* Image 1. */
    float_image_t *f2,  /* Image 2. */
    int x,              /* Central column (origin for displacement). */
    int y,              /* Current row index in image. */
    int dmin,           /* Min displacement, in 1/3 pixels. */
    int dmax,           /* Max displacement, in 1/3 pixels. */
    int rx,             /* Window half-width. */
    int ry,             /* Window half-height. */
    int *dbest,         /* Adjusted displacement, in 1/3 pixels. */
    float *sbest,       /* Score (squared mismatch) for {dbest}. */
    float *w1,          /* Buffer for image 1 window samples. */
    float *w2           /* Buffer for image 2 window samples. */
  );
  /* Returns in {*dbest} the displacement in the range {[dmin..dmax]}
    that has minimum score, and in {*sbest} its score. All
    displacements are in multiples of 1/3 of a pixel. The score is
    computed using {mismatch2}, in window of {f1} and {f2} with
    dimensions {[-rx..+rx]x[-ry..+ry]} centered on row {y} columns
    {x-d/3} and {x+d/3}, respectively. The buffers {w1,w2} must have
    enough space for all window samples.  */

void fgst_insert_disp(int d, float s, float *disp, float *scor, int nd);
  /* Inserts in {disp[0..nd-1]} and {scor[0..nd-1]} the displacement {d}
    and its score {s}, in order of increasing {s}, and discards the 
    last entry (which may be {d,s} itself). */

void fgst_float_image_get_samples
  ( float_image_t *f,  /* Pixel row buffer for image 1. */
    int x,             /* Central column (origin for displacement). */
    int y,             /* Current row index in image. */
    int d,             /* The displacement, in 1/3 pixels. */
    int rx,            /* Window half-width. */
    int ry,            /* Window half-height. */
    float *w           /* (OUT) Window sample buffer. */
  );
  /* Extracts samples from a window with dimensions {[-rx..+rx]x[-ry..+ry]}
    centered on row {y} columns {x-d/3} and {x+d/3}, respectively.
    Also normalizes the samples to have mean 0 and variance 1.
    The buffer {w} must have enough space for all window samples. */

float fgst_interpolate(float sa, float sb, float sc, float sd, int rd);
  /* Given four consecutive samples {sa,sb,sc,sd}, returns the interpolated 
    image valeu at a point {rd/3} between samples {sb} and {sc}.
    Must be called with {rd=1} or {rd=2} only. */

void fgst_normalize_samples(float *w, int npix, int NC);
  /* Independently normalizes each channel of the given samples to have
    mean 0 and unit variance. Ignores {NAN} samples. If all samples
    are equal, sets them all to 0. */

#endif
