#ifndef float_image_paint_H
#define float_image_paint_H

/* Tools for drawing into float images. */
/* Last edited on 2009-08-22 12:38:42 by stolfi */ 

#include <bool.h>
#include <ellipse_crs.h>
#include <float_image.h>

/* IMAGE COORDINATES 

  For the functions in this interface, the domain of a float image {A}
  is an axis-aligned rectangle {[0 _ nx]  [0 _ ny]} of the plane
  {R^2} with pixel in row 0, column 0 adjacent to the origin, where
  {nx = A->sz[1]} and {ny = A->sz[2]}. Each pixel with indices {(x,y)}
  is conceptually a square with side 1, whose corners are {(x,y)} and
  {(x+1,y+1)}, and whose center is {(x+0.5,y+0.5)}.
  
  PROCEDURAL IMAGES
  
  A /procedural image/ is a function {func} that returns a float value
  {func(x,y)} for any {double} coordinates {x,y}.   */
   
typedef float float_image_func_t(double x, double y);
  /* Type of a function that defines a procedural image. */

void float_image_paint_sample
  ( float_image_t *A, 
    int c, 
    int ix, 
    int iy, 
    float_image_func_t *func, 
    float_image_func_t *mask, 
    int m
  );
  /* Overlays pixel {ix,iy} of channel {c} of image {A} with the
    procedural image {func}, whose opacity is defined
    by the procedural image {mask}.
    
    The opacity {mask(x,y)} is interpreted as the local fraction of
    the background that is covered by the overlay. Thus,
    {mask(x,y)<=0} means that the overlay is transparent (preserving
    the underlying image), {mask(x,y)>=1} means it is fully opaque,
    and {mask(x,y)} between 0 and 1 means it is semi-transparent.
    
    As special cases,assumes that the overlay is fully transparent if
    {func(x,y) == NAN}, irrespective of {mask}; otherise, if {mask} is
    NULL, assumes {mask(x,y) == 1} for all {x,y}. Fails if {mask(x,y)}
    is {NAN}.
    
    The result image is filtered by averaging it over a grid of
    {2*m+1} by {2*m+1} subsamples within a window of size {2  2}
    pixels centered on pixel {ix,iy}, with C1 piecewise quadratic
    weights. */

void float_image_paint_samples
  ( float_image_t *A, 
    int c, 
    int xLo, 
    int xHi,
    int yLo,
    int yHi, 
    float_image_func_t *func,
    float_image_func_t *mask,
    int m
  );
  /* Applies {float_image_paint_sample} to all pixels {ix,iy}
    with {ix} in {xLo..xHi} and {iy} in {yLo..yHi}. */

/* SYMBOL PAINTING

  The procedures in this section paint simple geometric 
  shapes on the given image {A}.  

  If the shape is closed, it is filled with the given sample value
  {vfill}, and then its outline is stroked with sample value {vdraw},
  using a round pen with radius {hwd} whose center is dragged around
  the outline. 
  
  An open shape can only be stroked, and admits no
  {vfill} parameter.  Procedures that take a {vback} parameter
  also paint the area outside the shape.
  
  If {vfill} is {NAN}, the shape is not filled (so the original image
  values will remain). Ditto if {vback} is {NAN}. The outline is not
  drawn if {vdraw} is {NAN} or {hwd} is zero.

  The shape is antialiased by taking {2*m+1} subsamples per pixel
  along each axis, as per {float_image_paint_sample}. */

void float_image_paint_dot
  ( float_image_t *A, 
    int c,           /* Channel. */                                
    double xctr,     /* Center's X coordinate. */                  
    double yctr,     /* Center's Y coordinate. */                  
    double rad,      /* Radius of cross (not counting {hwd}). */   
    double hwd,      /* Radius of pen tip. */                      
    bool_t round,    /* If {TRUE}, dor is round, else square. */
    bool_t diagonal, /* If {TRUE}, rotate 45 degrees. */           
    float vfill,     /* Ink value for filling. */                              
    float vdraw,     /* Ink value for stroking. */  
    int m            /* Subsampling parameter. */
  );
  /* Paints into channel {c} of image {A} a dot centered at {xctr,yctr}.
    
    If {round} is true, the dot is a circle with radius {rad} (pixels);
    otherwise it is it is a square with half-side {rad}. 
    In this second case, if {diagonal} is true, the square is rotated by 45 degrees.
    The signs of {rad} and {hwd} are ignored. */

void float_image_paint_smudge
  ( float_image_t *A, 
    int c,         /* Channel. */                                
    double xctr,   /* Center's X coordinate. */                  
    double yctr,   /* Center's Y coordinate. */                  
    double xdev,   /* Standard deviation in X direction. */   
    double ydev,   /* Standard deviation in Y direction. */                      
    bool_t fold,   /* TRUE to fold over at image boundaries. */
    float vctr,    /* Ink value at center. */  
    int m          /* Subsampling parameter. */
  );
  /* Paints onto channel {c} of image {A} a fuzzy dot centered at {xctr,yctr}.
    
    The dot will have color {vfill}; it will be fully opaque at its
    center, fading out to transparency according to an axis-aligned
    Gaussian bell fuction with deviations {xdev,ydev} along each axis.
    If {fold} is true, the smudge will fold over at the image
    boundaries. */

void float_image_paint_cross
  ( float_image_t *A, 
    int c,           /* Channel. */
    double xctr,     /* Center's X coordinate. */
    double yctr,     /* Center's Y coordinate. */
    double rad,      /* Radius of cross (not counting {hwd}). */ 
    double hwd,      /* Radius of pen tip. */
    bool_t diagonal, /* If {TRUE}, rotate 45 degrees. */
    float vdraw,     /* Ink value for stroking. */  
    int m            /* Subsampling parameter. */
  );
  /* Draws into channel {c} of image {A} a cross centered
    at {ctr,yctr}, with arms of length {rad}. 
    
    If {diagonal} is true, the cross is rotated 45
    degrees.  The signs of {rad} and {hwd} are ignored. */

/* ELLIPSE PAINTING */

void float_image_paint_ellipse
  ( float_image_t *A,
    int c,            /* Channel. */
    ellipse_crs_t *E, /* Ellipse parameters. */
    double hwd,       /* Radius of pen tip. */
    float vfill,      /* Ink value for the interior, or {NAN}. */  
    float vdraw,      /* Ink value for stroking the outline, of {NAN}. */  
    int m             /* Subsampling parameter. */
  );
  /* Paints into channel {c} of {A} the image of ellipse {E}. */ 

/* UTILITY FUNCTIONS */

void float_image_paint_get_index_range(double z, double r, int n, int *iLo, int *iHi);
  /* Sets {*iLo} and {*iHi} to the maximum and minimum indices of any
    pixel that may be affected by painting a figure whose real
    coordinates span the range from {z - r} to {z + r}. Takes into
    account that anti-aliasing effectivey enlarges each pixel by 1/2
    pixel all around. The resulting range is clipped to the range
    {0..n-1}. If {*iLo > *iHi} on return, no pixels are affected. */
  

#endif
