/* See {float_image_test.h}. */

/* Last edited on 2009-06-02 23:44:02 by stolfi */ 
/* Created on 2009-06-02 by J. Stolfi, UNICAMP */

#define float_image_test_C_COPYRIGHT \
  "Copyright  2009  by the State University of Campinas (UNICAMP)"

#define _GNU_SOURCE
#include <math.h>

#include <affirm.h>
#include <r2.h>

#include <float_image.h>

#include <float_image_test.h>

void float_image_test_gen_ripples(r2_t *p, int NC, int NX, int NY, float fs[])
  {
    /* Get coordinates of pixel center, relative to image center: */
    double x = p->c[0] - 0.5*NX;
    double y = p->c[1] - 0.5*NY;
    /* Compute circumradius {R} of image (in pixels): */
    double R = 0.5*hypot(NX, NY);
    /* Choose a min wavelength {PMin}: */
    double PMin = 4.0;
    /* Compute circular waves with freq prportional to distance. */
    /* At distance {iR}, the waves must have period {PMin}. */
    double arg = M_PI*(x*x + y*y)/R/PMin;
    int ic;
    for (ic = 0; ic < NC; ic++)
      { /* The wave has a different phase in each channel: */
        fs[ic] = 0.5 + 0.5*cos(arg + 2*M_PI*((double)ic)/((double)NC));
      }
  }

void float_image_test_gen_stripes(r2_t *p, int NC, int NX, int NY, float fs[])
  {
    /* Choose a min wavelength {PMin}: */
    int PMin = 4;
    /* Straight stripes, horizontal (green) and vertical (magenta). */
    int ic;
    for (ic = 0; ic < NC; ic++)
      { int ax = ic % 2; /* Coordinate axis perpendicular to stripes. */
        /* Get image size along this axis: */
        int wd = (ax == 0 ? NX : NY);
        /* Get coordinate of pixel center, relative to image center: */
        double z = p->c[ax]/((double)wd);
        /* Compute the wave's frequency relative to the fundamental: */
        demand(wd % PMin == 0, "input image size must be a multiple of {PMin}");
        int fr = wd/PMin;
        /* Compute the local phase of the wave: */
        double t = 2*M_PI*((double)fr)*z;
        /* Compute the wave's value: */
        fs[ic] = 0.5*(1 + cos(t));
      }
  }

void float_image_test_gen_checker(r2_t *p, int NC, int NX, int NY, float fs[])
  {
    /* A checkerboard pattern, equal to the product of two waves. */

    /* Choose a min wavelength {PMin}: */
    int PMin = 4;
    /* Compute the wave's value in the range {[-1 _ +1]}: */
    double v = 1.0;  /* Value of wave. */
    int ax;
    for (ax = 0; ax < 2; ax++)
      { /* Get image size along this axis: */
        int wd = (ax == 0 ? NX : NY);
        /* Get coordinate of pixel center, relative to image center: */
        double z = p->c[ax]/((double)wd);
        /* Compute the wave's component frequencies: */
        demand(wd % (3*PMin) == 0, "input image size must be a multiple of {3*PMin}");
        int fr1 = wd/PMin;
        int fr3 = wd/(3*PMin);
        /* Compute the local phase of the two components: */
        double t1 = 2*M_PI*((double)fr1)*z;
        double t3 = 2*M_PI*((double)fr3)*z;
        /* Compute the wave value: */
        double vax = 0.5*cos(t1) + 0.5*cos(t3);
        /* Multiply into {v}: */
        v *= vax;
      }
    /* Adjust {v} to the range {[0 _ 1]}: */
    v = 0.5*(1 + v);
    /* Set into pixel: */
    int ic;
    for (ic = 0; ic < NC; ic++) { fs[ic] = v; }
  }

void float_image_test_paint(float_image_t *img, float_image_test_generator_t *proc)
  {
    /* Get image dimensions: */
    int NC = img->sz[0];
    int NX = img->sz[1];
    int NY = img->sz[2];
    
    /* Allocate a pixel's worth of samples: */
    float v[NC];
    
    /* Scan pixels: */
    int ix, iy;
    for (iy = 0; iy < NY; iy++)
      { for (ix = 0; ix < NX; ix++)
          { /* Get coordinates of pixel center, relative to image center: */
            r2_t p = (r2_t){{ ix + 0.5, iy + 0.5 }};
            /* Evaluate procedural image: */
            proc(&p, NC, NX, NY, v);
            /* Store into pixel: */
            float_image_set_pixel(img, ix, iy, v);
          }
      }
  }

