/* See {float_image_transform.h}. */
/* Last edited on 2009-08-22 17:08:56 by stolfi */

#define _GNU_SOURCE
#include <assert.h>
#include <limits.h>
#include <string.h>
#include <math.h>
 
#include <bool.h>
#include <r2.h>
#include <r2x2.h>
#include <r3.h>
#include <r3x3.h>
#include <affirm.h>
#include <float_image.h>
#include <float_image_transform.h>
#include <float_image_interpolate.h>

/* INTERNAL PROTOTYPES */

#define SQRT3 (1.73205080756887729352)

void fitr_debug_double_pixel(char *label, double x, double y, int chns, double v[], char *tail);
  /* Prints the point {(x,y)} and the float-valued samples
    {v[0..chns-1]} to {stderr}, tagged with the string {label} and
    terminated by the string {tail}. */

void fitr_get_transformed_pixel
  ( float_image_t *img, 
    int col, 
    int row, 
    fitr_transform_t *map,
    float undef, 
    bool_t avg,
    int order, 
    float f[],
    bool_t *debugp
  );
  /* Computes pixel on column{col} and row {row} of the transformed
    image. The result is an average (if {avg} is true) or partition
    integral (if {avg} is false) of image values in the neighborhood
    of the the pixel's center. Uses {map} to find the corresponding
    points in the input image. Returns the result in {f[c]}, for {c}
    in {{0..chns-1}} where {chns=img->sz[0]}.
    
    Pixels are interpolated using C0 bilinear interpolation (if
    {order} is 0) or C1 bicubic interpolation (if {order} is 1). Note
    that the latter may overshoot the interval {{0_1]}.
    
    If {map} returns {(NAN,NAN)}, the procedure sets all {f} samples
    to {undef}.
    
    The {*debugp} variable is passed to {map}. (If {debugp} is NULL,
    the procedure provides the address of an internal temporary
    variable.) If {map} sets that variable to TRUE, the procedure
    prints some debugging info. */

void fitr_eval_around_point
  ( float_image_t *img, 
    r2_t *p, 
    r2x2_t *J,
    bool_t avg,
    int order,
    float f[],
    bool_t debug
  );
  /* Evaluates the image {img} at the point {p = (x,y)} (relative to
    the image's lower left corner). The value is an average (if {avg}
    is true) or partition integral (if {avg} is false) of the values
    of {img} around {p}, using a pixel sampling kernel whose domain
    has been transformed by the Jacobian {J}. The result is returned
    in {v[0..chns-1]} where {chns=img->sz[0]}.
    
    Pixels are interpolated using C0 bilinear interpolation (if
    {order} is 0) or C1 bicubic interpolation (if {order} is 1). Note
    that the latter may overshoot the interval {{0_1]}. */

void fitr_persp_pixel_average
  ( float_image_t *img, /* Image to sample. */
    double xlo,         /* Min X in true coords. */
    double xhi,         /* Max X in true coords. */
    double ylo,         /* Min Y in true coords. */
    double yhi,         /* Max Y in true coords. */
    r3x3_t *D2I,        /* Projective map from true coords to image coords. */
    ix_reduction_t red, /* Index reduction method. */ 
    double val[]        /* (OUT) Pixel average. */
  );
  /* Computes the average {val[0..NC-1]} of the the pixels of image
    {img} within the rectangle {[xlo_xhi][ylo_yhi]} in true
    coordinates, given the true-to-image map {D2I}. Pixels are
    interpolated using C0 bilinear interpolation with the index
    reduction {red}. */
    
r2_t fitr_persp_map_xy(double x, double y, r3x3_t *P);
  /* Applies the projective map {P} to the true coordinates {(x,y)},
    and returns the corresponding image coordinates, packaged as an
    {r2_t}. */

void fitr_adjust_sampling(int nsNew, int *nsP, double *stepP, bool_t debug);
  /* Sets the number of samples {*nsP} to {nsNew}, then adjusts
    {*stepP} so that the sampling span {(*nsP)*(*stepP)}: */
    
void fitr_accum_pixel(int chns, double vs[], double wt, double v[], double *wtotP);
  /* Adds {wt*vs[c]} to {v[c]} for {c} in {0..chns-1}. Also adds {wt} to {*wtotP}. */

double fitr_sample_weight(double d, double sigma);
  /* Returns the relative weight of a sample that is displaced by 
    a distance {d} from the center sample, assuming that 
    the samples are to be smoothed by a Gaussian with 
    deviation {sigma}. */

void fitr_check_jacobian(r2_t *p, r2x2_t *J, fitr_transform_t *map, double eps);
  /* Checks the given Jacobian {J} against the numerically estimated
    derivatives of {q} relative to {p}, where {q} is computed from {p}
    by the {map} procedure. The estimates are obtained by central
    divided difference with step not exceeding {eps} in the output or
    input domain. */

void fitr_fill_pixel(int chns, float v[], float und);
  /* Sets the {float} samples {v[0..chns-1]} to the value {und}. */

void fitr_scale_pixel(int chns, double s, double v[], float f[]);
  /* Scales {double} samples {v[0..chns-1]} by {s}, stored in {float} samples {f[0..chns-1]}. */

void fitr_debug_direction(char *label, r2_t *v, char *tail);
  /* Prints the vector {v} to {stderr}. */

void fitr_debug_eigen(char *label, r2_t *e, char *tail);
  /* Prints the eigenvalues {e.c[0..1]} to {stderr}. */

/* IMPLEMENTATIONS */

void fitr_transform
  ( float_image_t *iimg,   /* Input image. */
    fitr_transform_t *map, /* Output-to-input coordinate transformation. */
    float undef,           /* Sample value for undefined output pixels. */
    bool_t avg,            /* TRUE to average pixels, FALSE to add them. */
    int order,             /* Interpolation order. */
    float_image_t *oimg    /* Output image. */
  )
  { demand(iimg->sz[0] == oimg->sz[0], "images must have the same channels");
    int chns = oimg->sz[0];
    int ocols = oimg->sz[1];
    int orows = oimg->sz[2];
    float fo[chns];
    /* Scan rows from top to bottom to make debugging easier: */
    int row, col;
    for (row = orows-1; row >= 0; row--)
      { for (col = 0; col < ocols; col++)
          { bool_t debug = FALSE;
            /* Get transformed pixel: */
            fitr_get_transformed_pixel(iimg, col, row, map, undef, avg, order, fo, &debug);
            if (debug) { fitr_debug_pixel("  fo", col+0.5, row+0.5, chns, fo, "\n"); }
            /* Store it in the image: */
            float_image_set_pixel(oimg, col, row, fo);
            if (debug) { fprintf(stderr, "\n"); }
          }
      }
  }    

#define MAX_TAYLOR_ERROR (+INF)  
  /* For now */

void fitr_get_transformed_pixel
  ( float_image_t *img, 
    int col, 
    int row, 
    fitr_transform_t *map, 
    float undef, 
    bool_t avg,
    int order, 
    float f[],
    bool_t *debugp
  )
  { 
    int chns = img->sz[0];
    bool_t debug = (debugp == NULL ? FALSE : (*debugp));

    /* To make the sampling integrals manageable, we replace the transform
      {map} is by its 1st degree Taylor expansion.  
      
      More precisely, for each output pixel center {op} we approximate
      {map(op+r)}, for small {r}, by {aff(r) = ip + r J}, where {ip}
      is {map(op) and {J} is the Jacobian matrix of {map} evaluated at
      {op}. This approximation assumes that the higher-order terms of
      the Taylor expansion is negligible as long as {r} lies within
      the sampling kernel's support (typically, a few output pixels).
      The integral then reduces to a sum of input pixel values times
      affinely-transformed images of the reconstruction kernel {h}. */

    /* Get Cartesian coordinates {xc,yc} of destination pixel's center: */
    r2_t op = (r2_t) {{ (double)col + 0.5, (double)row + 0.5 }};
    
    /* Compute corresp point {pt=(xt,yt)} in source image, and Jacobian: */
    r2x2_t J; /* Jacobian {{dxt/dxc, dxt/dyc}, {dyt/dxc, dyt/dyc}} */
    r2_t ip;
    map(&op, &ip, &J, &debug);
    bool_t invalid = (isnan(ip.c[0]) || isnan(ip.c[1]));
    
    if (debug) { fprintf(stderr, "  computing output pixel with indices (%d,%d):\n", col,row); }
    if (debug) { fitr_debug_point("    po", &op, NULL, "\n"); }
    if (debug) { fitr_debug_point("    pi", &ip, &J, "\n"); }
    if (debug & (!invalid)) { fitr_check_jacobian(&op, &J, map, 1.0e-5); }

   /* Get input image value at point {pt}: */
    if (invalid)
      { /* There is no image on the negative side of the two-sided plane: */
        fitr_fill_pixel(chns, f, undef);
      }
    else
      { /* demand(err <= MAX_TAYLOR_ERROR, "excessive warp, should use subsampling"); */
        /* Sample points of input image: */
        fitr_eval_around_point(img, &ip, &J, avg, order, f, debug);
      }
    
    /* Return the debug variable to the caller: */
    if (debugp != NULL) { (*debugp) = debug; }
  }

void fitr_apply_radial_map(r2_t *p, r2_t *h, double kappa, r2x2_t *J)
  {
    /* If {p} is already invalid, do nothing: */
    if (isnan(p->c[0]) || isnan(p->c[1])) { return; }
    
    /* If the radial map is not the identity, apply it: */
    if (kappa != 0)
      { /* Get the pixel sensor dimensions {hX,hY} in mm: */
        double hX = h->c[0];
        double hY = h->c[1];
      
        /* Get point {pX,pY} (projected coordinates, in mm): */
        double pX = p->c[0] * hX;
        double pY = p->c[1] * hY;
        
        /* Compute distance squared {s} from optical axis: */
        double s = pX*pX + pY*pY;
        
        /* Compute the scaling factor {f} and its derivative {df_ds}: */
        double m = 2*kappa*s;
        if (m >= 0.995) 
          { /* Point is outside the maximum radius {R}: */
            p->c[0] = p->c[1] = NAN;; 
            return; 
          }
        double h = 1.0/(1.0 - m);
        double f = sqrt(h);
        double df_ds = kappa*h*h/f;
        
        /* Compute the mapped point coordinates: */
        double qX = pX*f; 
        double qY = pY*f;
        
        /* Update the given point {p} (image coords, in pixels): */
        p->c[0] = qX / hX; p->c[1] = qY / hY;

        if (J != NULL)
          { /* Compute the Jacobian {K} of this map: */
            r2x2_t K;
            K.c[0][0]= f + pX*df_ds*2*pX;           /* dqX_dpX */ 
            K.c[0][1]= 0 + pY*df_ds*2*pX * hX / hY; /* dqY_dpX */ 
            K.c[1][0]= 0 + pX*df_ds*2*pY * hY / hX; /* dqX_dpY */ 
            K.c[1][1]= f + pY*df_ds*2*pY;           /* dqY_dpY */ 

            /* Update the Jacobian {J}: */
            r2x2_mul(J, &K, J);
          }
      }
  }

void fitr_apply_projective_map(r2_t *p, r3x3_t *M, r2x2_t *J)
  { 
    /* If {p} is already invalid, do nothing: */
    if (isnan(p->c[0]) || isnan(p->c[1])) { return; }
    
    /* Convert {p} to homogeneous coordinates {hp}: */
    r3_t hp = (r3_t) {{ 1.0, p->c[0], p->c[1] }}; 

    /* Map {hp} to {hq}, get the homogeneous coordinates {qw,qx,qy}: */
    r3_t hq;
    r3x3_map_row(&hp, M, &hq);
    double qw = hq.c[0]; 
    double qx = hq.c[1]; 
    double qy = hq.c[2]; 
    
    /* Check for validity of result: */
    if (qw <= 0.0)
      { /* Result is at infinity, or beyond: */
        p->c[0] = p->c[1] = NAN;; 
        return; 
      }
    else
      { /* Compute the Cartesian coordinates {qX,qY} of {q}: */
        double qX = qx/qw; 
        double qY = qy/qw;
        
        /* Update the given point {p}: */
        p->c[0] = qX; p->c[1] = qY;
        
        if (J != NULL)
          { /* Compute the Jacobian of {hq} (homogeneous) w.r.t {p} (Cartesian): */
            double dqw_dpX = M->c[1][0], dqw_dpY = M->c[2][0];
            double dqx_dpX = M->c[1][1], dqx_dpY = M->c[2][1];
            double dqy_dpX = M->c[1][2], dqy_dpY = M->c[2][2];

            /* Compute the Jacobian of {q} (Cartesian) w.r.t {p} (Cartesian): */
            r2x2_t K;
            K.c[0][0]= (dqx_dpX - qX*dqw_dpX)/qw; /* dqX_dpX */ 
            K.c[0][1]= (dqy_dpX - qY*dqw_dpX)/qw; /* dqY_dpX */ 
            K.c[1][0]= (dqx_dpY - qX*dqw_dpY)/qw; /* dqX_dpY */ 
            K.c[1][1]= (dqy_dpY - qY*dqw_dpY)/qw; /* dqY_dpY */ 

            /* Update the Jacobian {J}: */
            r2x2_mul(J, &K, J);
          }
      }
  }

#define MAX_SAMPLES_PER_AXIS (1000)
  /* Max number of samples to take along each sampling axis when
    computing one output pixel. */

#define MAX_SAMPLES_PER_PIXEL (2000)
  /* Max total number of samples to take to compute one
    output pixel. */

void fitr_eval_around_point
  ( float_image_t *img, 
    r2_t *p,
    r2x2_t *J,
    bool_t avg,
    int order,
    float f[],
    bool_t debug
  )
  {
    /* Get image dimensions: */
    int chns = img->sz[0];
    
    /* Get coordinates {(x,y)} of center point: */
    double x = p->c[0], y = p->c[1];
    
    if (debug) { fprintf(stderr, "    evaluating img(%9.4f,%9.4f)\n", x, y); }

    /* If {J} expands the space along some direction, we may need to
      interpolate at various points near {p}, spaced at most 1 unit
      apart, and combine those samples with the proper weights. To
      ensure the proper spacing, we place the samples at a unit-step
      rectangular grid aligned with the vectors {u*J} and {v*J} where
      {u} and {v} are the eigenvectors of {J*J^t}. These are the
      directions in the input domain along which {J} has the maximum
      and minimum stretch. */

    /* Compute the eigenvalues {e.c[0..1]}} and eigenvectors {R.c[0..1]} of {J*J^t}: */
    r2_t e;   /* Eigenvalues of {J*J^t}. */
    r2x2_t R; /* Rows are the corresponding eigenvectors (eventually). */
    r2x2_mul_tr(J,J,&R);         /* Now {R = J*J^t}. */
    r2x2_sym_eigen(&R, &e, &R);  /* Now {R} has the eigenvectors of {J*J^t}. */

    if (debug) { fitr_debug_eigen("      e", &e, "\n"); }
    if (debug) { fitr_debug_rows("      R", &R, "\n"); }
    
    /* Compute the directions of maximum stretch: */
    r2x2_mul(&R, J, &R);        /* Now {R} has the input resampling axes. */
    if (debug) { fitr_debug_rows("      U", &R, "\n"); }

    /* Compute the number of steps {ns[i]} to take along each vector,
      and their weight factors. If there is no stretching along 
      the sampling axis {i}, then one sample suffices (with unit weight).
      If there is stretching by a factor {f > 1}, we take samples
      at unit distance apart, and average them, to remove the high
      frequencies. */

    double step[2]; /* Step size for smoothing of interpolated values. */
    int ns[2];          /* Will step from {-ns[i]} to {+ns[i]} along sampling axis {i}. */
    double sigma[2];    /* Amount of extra smoothing needed along sampling axis {i}. */
    int i;
    for (i = 0; i < 2; i++)
      { step[i] = 1.0; /* May be increased to avoid excessive sampling. */
        double str = hypot(R.c[i][0], R.c[i][1]); /* Stretch along sampling axis {i}. */
        if (debug) 
          { fprintf(stderr, "      str[%d] = %8.2f\n", i, str);
            double sqe = sqrt(e.c[i]);
            fprintf(stderr, "      sqe[%d] = %8.2f\n", i, sqe);
            affirm(fabs(str-sqe) < 1.0e-6*(str+sqe), "bug in eigenvalue");
          }
        
        if (str <= 1.0)
          { /* Input image is not being squeezed; just interpolate it. */
            sigma[i] = 0.0; ns[i] = 0;
            /* We don't need the sampling axis {i}: */
            R.c[i][0] = R.c[i][1] = 0.0;
          }
        else
          { /* Input image is being squeezed; must interpolate and smooth. */
            /* Normalize sampling axis vector {R[i]} to unit length: */
            R.c[i][0] /= str; R.c[i][1] /= str; 
            /* Compute width {sigma[i]} of Gaussian smoothing kernel: */
            sigma[i] = sqrt(str*str - 1.0);
            /* Compute number of steps needed for {sigma[i]}-smoothing: */
            ns[i] = (int)ceil(3*sigma[i]/step[i]) - 1;
            if (ns[i] < 0) { ns[i] = 0; }
            /* Guard against excessive samples along sampling axis {i}: */
            int nsMax = MAX_SAMPLES_PER_AXIS;
            if (ns[i] > nsMax)
              { fitr_adjust_sampling(nsMax, &(ns[i]), &(step[i]), debug); }
          }
      }
      
    /* Guard against excessive total samples: */
    int ts = (2*ns[0] + 1)*(2*ns[1] + 1);
    int tsMax = MAX_SAMPLES_PER_PIXEL;
    if (ts > tsMax)
      { double sMean = sqrt(((double)tsMax)/((double)ts));  /* {ns} shrink factor. */
        for (i = 0; i < 2; i++)
          { int nsNew = (int)ceil(sMean*ns[i]);
            fitr_adjust_sampling(nsNew, &(ns[i]), &(step[i]), debug);
          }
      }
      
    if (debug) 
      { /* Show sampling parameters: */
        for (i = 0; i < 2; i++)
          { fprintf(stderr, "      sigma[%d] = %8.2f", i, sigma[i]);
            fprintf(stderr, " ns[%d] = %d\n", i, ns[i]);
          }
      }

    /* Temporary result pixel: */
    double v[chns];

    /* Clear the result pixel: */
    int ich;
    for (ich = 0; ich < chns; ich++) { v[ich] = 0.0; }

    /* Accumulate samples: */
    double vs[chns];  /* Pixel at one sample point. */
    double wtot = 0;
    int k0, k1;
    bool_t red = TRUE;
    for (k0 = -ns[0]; k0 <= +ns[0]; k0++)
      { double f0 = k0*step[0]; /* Sample displacement along evec {R[0]}. */
        double w0 = fitr_sample_weight(f0, sigma[0]); /* Sample weight for {f0}. */
        for (k1 = -ns[1]; k1 <= +ns[1]; k1++)
          { double f1 = k1*step[1]; /* Sample displacement along evec {R[1]}. */
            double w1 = fitr_sample_weight(f1, sigma[1]); /* Sample weight for {f1}. */
            /* Compute sample point {(xs,ys)}: */
            double xs = x + f0*R.c[0][0] + f1*R.c[1][0];
            double ys = y + f1*R.c[0][1] + f1*R.c[1][1];
            /* Evaluate source image at fractional point {(xs,ys)}: */
            float_image_interpolate_pixel(img, xs, ys, order, red, vs);
            if (debug) { fitr_debug_double_pixel("        is", xs, ys, chns, vs, ""); }
            double ws = w0*w1; /* Weight of sample. */
            if (debug) { fprintf(stderr, "        weight = %8.4f\n", ws); }
            /* Accumulate: */
            fitr_accum_pixel(chns, vs, ws, v, &wtot);
          }
      }
    if (debug) { fitr_debug_double_pixel("      ac", x, y, chns, v, "\n"); }
    if (debug) { fprintf(stderr, "      wtot = %8.4f\n", wtot); }
    
    /* Compute scale factor to account for non-unit-sum weights: */
    assert(wtot != 0);
    double scale = 1.0/wtot;
    
    /* If {avg} is FALSE, correct for the Jacobian's stretch/shrink: */
    if (! avg) { scale *= fabs(r2x2_det(J)); }

    /* Scale pixel value: */
    fitr_scale_pixel(chns, scale, v, f);
    if (debug) { fitr_debug_pixel("    ev", x, y, chns, f, "\n"); }
  }
  
void fitr_adjust_sampling(int nsNew, int *nsP, double *stepP, bool_t debug)
  { 
    double s = ((double)nsNew)/((double)(*nsP)); /* {ns[i]} shrink factor. */
    double stepNew = (*stepP) / s;
    if (debug) 
      { fprintf(stderr, "        adjusting {ns} from %d to %d\n", (*nsP), nsNew);
        fprintf(stderr, "        adjusting {step} from %.2f to %.2f\n", (*stepP), stepNew );
      }
    (*stepP) = stepNew;
    (*nsP) = nsNew;
  }

double fitr_sample_weight(double d, double sigma)
  { 
    if ((sigma == 0.0) || (d == 0.0))
      { return 1.0; }
    else
      { double z = d/sigma;
        return exp(-z*z/2);
      }
  }

void fitr_accum_pixel(int chns, double vs[], double wt, double v[], double *wtotP)
  {
    if (wt != 0.0)
      { int ich;
        for (ich = 0; ich < chns; ich++)
          { v[ich] += wt*vs[ich]; }
        (*wtotP) += wt;
      }
  }

void fitr_fill_pixel(int chns, float *v, float val)
  {
    int ich;
    for (ich = 0; ich < chns; ich++) 
      { v[ich] = val; }
  }

void fitr_scale_pixel(int chns, double s, double v[], float f[])
  {
    int ich;
    for (ich = 0; ich < chns; ich++) { f[ich] = s*v[ich]; }
  }

void fitr_persp_copy_rectangle
  ( float_image_t *iimg,
    double xlo,         /* Min X in true coords. */
    double xhi,         /* Max X in true coords. */
    double ylo,         /* Min Y in true coords. */
    double yhi,         /* Max Y in true coords. */
    r3x3_t *T2I,        /* Projective map from true coords to image coords. */
    ix_reduction_t red, /* Index reduction method. */ 
    int x0,             /* First output image column. */
    int y0,             /* First output image row. */
    int NX,             /* Number of output image columns. */
    int NY,             /* Number of output image rows. */
    float_image_t *oimg
  )
  { 
    /* !!! Should use the general transform procedure above. */
    if ((NX == 0) || (NY == 0)) { /* Nothing to do: */ return; }
    
    /* Get input image dimensions (samples): */
    int NC, INX, INY;
    float_image_get_size(iimg, &NC, &INX, &INY);
    
    /* Actual pixel sizes (mm): */
    double dx = (xhi - xlo)/NX;
    double dy = (yhi - ylo)/NY;
    /* Extract samples and store in {oimg}: */
    double val[NC];
    int x, y;
    for (y = 0; y < NY; y++)
      { double pix_ylo = ylo + y*dy;
        double pix_yhi = ylo + (y+1)*dy;
        for (x = 0; x < NX; x++)
          { double pix_xlo = xlo + x*dx;
            double pix_xhi = xlo + (x+1)*dx;
            fitr_persp_pixel_average(iimg, pix_xlo, pix_xhi, pix_ylo, pix_yhi, T2I, red, val);
            int c;
            for (c = 0; c < NC; c++)
              { float_image_set_sample(oimg, c, x0 + x, y0 + y, val[c]); }
          }
      }
  }

#define fitr_persp_MAXSMP (100)
  /* Maximum number of subsamples allowed for each output pixel. */

void fitr_persp_pixel_average
  ( float_image_t *img, /* Image to sample. */
    double xlo,         /* Min X in true coords. */
    double xhi,         /* Max X in true coords. */
    double ylo,         /* Min Y in true coords. */
    double yhi,         /* Max Y in true coords. */
    r3x3_t *T2I,        /* Projective map from true coords to image coords. */
    ix_reduction_t red, /* Index reduction method. */ 
    double val[]        /* (OUT) Pixel average. */
  )
  { 
    /* ??? Change {fitr_persp_copy_rectangle} to use general transform, and junk this. ??? */
    bool_t debug = FALSE;
    int NC = img->sz[0];
    int order = 0; /* Bilinear interpolation should be enough. */
    
    if (debug) 
      { fprintf(stderr, "  averaging");
        fprintf(stderr, " [ %+6.1f _ %+6.1f ]", xlo, xhi);
        fprintf(stderr, "  [ %+6.1f _ %+6.1f ]\n", ylo, yhi);
      }
    /* Get corners of sampling area in {img} (homogeneous coords): */
    r2_t bl = fitr_persp_map_xy(xlo, ylo, T2I);
    r2_t br = fitr_persp_map_xy(xhi, ylo, T2I);
    r2_t tl = fitr_persp_map_xy(xlo, yhi, T2I);
    r2_t tr = fitr_persp_map_xy(xhi, yhi, T2I);
    /* Get approx extent in each direction (pixels): */
    double dx = fmax(r2_dist(&bl, &br), r2_dist(&tl, &tr));
    double dy = fmax(r2_dist(&bl, &tl), r2_dist(&br, &tr));
    /* Compute appropriate number of samples: */
    double maxSamplingStep = 0.5;
    int nx = (int)ceil(fmax(1.0,dx)/maxSamplingStep); 
    int ny = (int)ceil(fmax(1.0,dy)/maxSamplingStep);
    demand((nx <= fitr_persp_MAXSMP) && (dy <= fitr_persp_MAXSMP), "input-to-output scale is too big");
    /* Compute subsampling step size (mm): */
    double xStep = (xhi - xlo)/nx;
    double yStep = (yhi - ylo)/ny;
    /* Subsample and get average: */
    double sumW = 0.0;
    double sumWV[NC], sumV2W[NC];
    int c;
    for (c = 0; c < NC; c++) { sumWV[c] = sumV2W[c] = 0; }
    int ix, iy;
    for (iy = 0; iy < ny; iy++)
      { double y = ylo + (iy + 0.5)*yStep;
        for (ix = 0; ix < nx; ix++)
          { double x = xlo + (ix + 0.5)*xStep;
            r2_t p = fitr_persp_map_xy(x, y, T2I);
            double W = 1.0; /* Weight of subsample. */
            for (c = 0; c < NC; c++) 
              { double V = float_image_interpolate_sample(img, c, p.c[0], p.c[1], order, red);
                sumWV[c] += V*W; sumV2W[c] += V*V*W;
              }
            sumW += W; 
          }
      }
    for (c = 0; c < NC; c++) 
      { double avg = sumWV[c]/sumW;
        double var = sumV2W[c]/sumW - avg*avg;
        if (debug) 
          { double dev = sqrt(fmax(var,0));
            fprintf(stderr, "  channel %d: avg = %8.5f  dev = %8.5f\n", c, avg, dev);
          }
        val[c] = avg;
      }
  }

void fitr_persp_rectangle_average
  ( float_image_t *img, /* Image to sample. */
    interval_t tbox[],  /* Rectangle in true coordinates. */
    r3x3_t *T2I,        /* Projective map from true coords to image coords. */
    double mrg,         /* Safety border width (pixels). */
    ix_reduction_t red, /* Index reduction method. */ 
    double avg[]        /* (OUT) Pixel average. */
  )
  { /* Get image channel count: */
    int NC, NX, NY;
    float_image_get_size(img, &NC, &NX, &NY);
    /* Get the inverse (image-to-true) map: */
    r3x3_t I2T; r3x3_inv(T2I, &I2T);
    /* Get a bounding rectangle in image coordinates: */
    interval_t ibox[2];
    fitr_persp_get_rectangle_bbox(tbox, T2I, ibox);
    /* Round image rectangle to integers: */
    int xlo = (int)ceil(ibox[0].end[0]);
    int xhi = (int)floor(ibox[0].end[1]);
    int ylo = (int)ceil(ibox[1].end[0]);
    int yhi = (int)floor(ibox[1].end[1]);
    /* Scan pixels fully contained in bounding rectangle: */
    int NP = 0;       /* Count of pixels inside disk. */
    double davg[NC];  /* Sum of pixel colors inside disk. */
    int c;
    for (c = 0; c < NC; c++) { davg[c] = 0.0; }
    int x, y;
    for (y = ylo; y < yhi; y++)
      { for (x = xlo; x < xhi; x++)
          { /* Reduce indices to image domain: */
            int xr = ix_reduce(x, NX, red); if (xr < 0) { continue; }
            int yr = ix_reduce(y, NY, red); if (yr < 0) { continue; }
            /* Is pixel (fattened by {mrg}) inside the disk image?: */
            if (fitr_persp_pixel_is_inside_rectangle(xr, yr, mrg, &I2T, tbox))
              { /* Accumulate its color in {davg,NP}: */
                float pix[NC];
                float_image_get_pixel(img, xr, yr, pix);
                for (c = 0; c < NC; c++) { davg[c] += pix[c]; }
                NP++;
              }
          }
      }
    /* Compute average color: */
    for (c = 0; c < NC; c++) { avg[c] = davg[c]/((double)NP); }
  }

void fitr_persp_disk_average
  ( float_image_t *img, /* Input image. */
    r2_t *ctr,          /* Disk center in true coordinates. */
    double rad,         /* Disk radius in true coordinates. */
    r3x3_t *T2I,        /* True-to-image projective map matrix. */
    ix_reduction_t red, /* Index reduction method. */ 
    double mrg,         /* Safety border width (pixels). */
    float avg[]         /* (OUT) average disk color. */
  )
  { 
    /* Get image channel count: */
    int NC, NX, NY;
    float_image_get_size(img, &NC, &NX, &NY);
    /* Get the inverse (image-to-true) map: */
    r3x3_t I2T; r3x3_inv(T2I, &I2T);
    /* Get a bounding rectangle in image coordinates: */
    interval_t ibox[2];
    fitr_persp_get_disk_bbox(ctr, rad, T2I, ibox);
    /* Round rectangle to integers: */
    int xlo = (int)ceil(ibox[0].end[0]);
    int xhi = (int)floor(ibox[0].end[1]);
    int ylo = (int)ceil(ibox[1].end[0]);
    int yhi = (int)floor(ibox[1].end[1]);
    /* Scan pixels fully contained in bounding rectangle: */
    int NP = 0;       /* Count of pixels inside disk. */
    double davg[NC];  /* Sum of pixel colors inside disk. */
    int c;
    for (c = 0; c < NC; c++) { davg[c] = 0.0; }
    int x, y;
    for (y = ylo; y < yhi; y++)
      { for (x = xlo; x < xhi; x++)
          { /* Reduce indices to image domain: */
            int xr = ix_reduce(x, NX, red); if (xr < 0) { continue; }
            int yr = ix_reduce(y, NY, red); if (yr < 0) { continue; }
            /* Is pixel (fattened by {mrg}) inside the disk image?: */
            if (fitr_persp_pixel_is_inside_disk(xr, yr, mrg, &I2T, ctr, rad))
              { /* Accumulate its color in {davg,NP}: */
                float pix[NC];
                float_image_get_pixel(img, xr, yr, pix);
                for (c = 0; c < NC; c++) { davg[c] += pix[c]; }
                NP++;
              }
          }
      }
    /* Compute average color: */
    for (c = 0; c < NC; c++) { avg[c] = davg[c]/((double)NP); }
  }

void fitr_persp_get_rectangle_bbox
  ( interval_t tbox[],   /* Rectangle in true coordinates. */
    r3x3_t *T2I,         /* True-to-image projective map matrix. */
    interval_t ibox[]    /* (OUT) bonding box in image coordinates. */
  )
  {
    /* Initialize bounds: */
    int ax;
    for (ax = 0; ax < 2; ax++) { ibox[ax].end[0] = +INF; ibox[ax].end[1] = -INF; }
    /* Hack: map the corners of a slightly wider rectangle and get its bbox */
    double wx = HI(tbox[0]) - LO(tbox[0]);
    double wy = HI(tbox[1]) - LO(tbox[1]);
    int dx, dy;
    for (dx = 0; dx <= 1; dx++)
      { for (dy = 0; dy <= 1; dy++)
          { /* Get a corner of the octagon: */
            double txp = tbox[0].end[dx] + (2*dx - 1)*0.00001*wx;
            double typ = tbox[1].end[dy] + (2*dy - 1)*0.00001*wy;
            /* Map corner to the image coord system: */
            r2_t ip = fitr_persp_map_xy(txp, typ, T2I);
            /* Expand box: */
            for (ax = 0; ax < 2; ax++)
              { if (ip.c[ax] < ibox[ax].end[0]) { ibox[ax].end[0] = ip.c[ax]; }
                if (ip.c[ax] > ibox[ax].end[1]) { ibox[ax].end[1] = ip.c[ax]; }
              }
	  }
      }
  }

void fitr_persp_get_disk_bbox
  ( r2_t *ctr,             /* Disk center in true coordinates. */
    double rad,            /* Disk radius in true coordinates. */
    r3x3_t *T2I,           /* True-to-image projective map matrix. */
    interval_t ibox[]      /* (OUT) bonding box. */
  )
  {
    /* Initialize bounds: */
    int ax;
    for (ax = 0; ax < 2; ax++) { ibox[ax].end[0] = +INF; ibox[ax].end[1] = -INF; }
    /* Hack: map the corners of an enclosing 8-gon and get its bbox */
    double a = 1.00001*rad;             /* Max abs coord of an octagon corner. */
    double b = 1.00001*rad*tan(M_PI/8); /* Min abs coord of an octagon corner. */
    int dx, dy;
    for (dx = -1; dx <= +1; dx += 2)
      { for (dy = -1; dy <= +1; dy += 2)
          { /* Get a corner of the octagon: */
            double txp = ctr->c[0] + dx*a;
            double typ = ctr->c[1] + dy*b;
            /* Try the two transposed copies of the point: */
            int swap;
	    for (swap = 0; swap < 2; swap++)
	      { /* Map it to the image coord system: */
                r2_t ip = fitr_persp_map_xy(txp, typ, T2I);
                /* Expand box: */
                for (ax = 0; ax < 2; ax++)
                  { if (ip.c[ax] < ibox[ax].end[0]) { ibox[ax].end[0] = ip.c[ax]; }
		    if (ip.c[ax] > ibox[ax].end[1]) { ibox[ax].end[1] = ip.c[ax]; }
		  }
                /* Swap true coords: */
                double zz = txp; txp = typ; typ = zz;
	      }
	  }
      }
  }

bool_t fitr_persp_pixel_is_inside_rectangle
  ( int x,
    int y,
    double mrg, 
    r3x3_t *I2T,
    interval_t tbox[]   /* Rectangle in true coordinates. */
  )
  {
    /* Check the four corners of the pixel: */
    int dx, dy;
    for (dx = -1; dx <= +1; dx += 2)
      { for (dy = -1; dy <= +1; dy += 2)
          { /* Get a corner of the pixel, expanded by {mrg}: */
            double ixp = x + 0.5 + dx*(0.5 + mrg);
            double iyp = y + 0.5 + dy*(0.5 + mrg);
	    /* Map it to the true coord system: */
            r2_t cp = fitr_persp_map_xy(ixp, iyp, I2T);
            /* Check if it is inside the rectangle: */
            if ((cp.c[0] < LO(tbox[0])) || (cp.c[0] > HI(tbox[0]))) { return FALSE; }
            if ((cp.c[1] < LO(tbox[1])) || (cp.c[1] > HI(tbox[1]))) { return FALSE; }
	  }
      }
    return TRUE;
  }

bool_t fitr_persp_pixel_is_inside_disk
  ( int x,
    int y,
    double mrg, 
    r3x3_t *I2T,
    r2_t *ctr, 
    double rad
  )
  {
    /* Check the four corners of the pixel: */
    double r2 = rad*rad;
    int dx, dy;
    for (dx = -1; dx <= +1; dx += 2)
      { for (dy = -1; dy <= +1; dy += 2)
          { /* Get a corner of the pixel, expanded by {mrg}: */
            double ixp = x + 0.5 + dx*(0.5 + mrg);
            double iyp = y + 0.5 + dy*(0.5 + mrg);
	    /* Map it to the true coord system: */
            r2_t cp = fitr_persp_map_xy(ixp, iyp, I2T);
            /* Check if it is inside the disk: */
            double d2 = r2_dist_sqr(&cp, ctr);
            if (d2 > r2) { return FALSE; }
	  }
      }
    return TRUE;
  }

r2_t fitr_persp_map_xy(double x, double y, r3x3_t *P)
  { r3_t p = (r3_t){{ 1.0, x, y }};
    r3x3_map_row(&p, P, &p);
    demand(p.c[0] > 0, "point mapped to yonder space");
    return (r2_t){{ p.c[1]/p.c[0], p.c[2]/p.c[0] }};
  }

void fitr_debug_pixel(char *label, double x, double y, int chns, float f[], char *tail)
  { 
    int ich;
    fprintf(stderr, "%s(%9.4f,%9.4f) = (", label, x, y);
    for (ich = 0; ich < chns; ich++) 
      { fprintf(stderr, " %7.4f", f[ich]); }
    fprintf(stderr, " )%s", tail);
  }

void fitr_debug_double_pixel(char *label, double x, double y, int chns, double v[], char *tail)
  { 
    int ich;
    fprintf(stderr, "%s(%9.4f,%9.4f) = (", label, x, y);
    for (ich = 0; ich < chns; ich++) 
      { fprintf(stderr, " %7.4f", v[ich]); }
    fprintf(stderr, " )%s", tail);
  }

void fitr_check_jacobian(r2_t *op, r2x2_t *J, fitr_transform_t *map, double eps)
  {
    fprintf(stderr, "    checking the Jacobian ...\n");
    /* Estimate the magnitudes {mag_J} of the Jacobian and {mag_op} of {op}: */
    double mag_J = fabs(J->c[0][0]) + fabs(J->c[0][1]) + fabs(J->c[1][0]) + fabs(J->c[1][1]); 
    double mag_op = fabs(op->c[0]) + fabs(op->c[1]);
    if (mag_J < 1.0e-100) { return; }
    /* Choose a small enough step ... */
    double step = eps*mag_op/(mag_J < 1 ? 1 : mag_J);
    /* ... but not too small: */
    if (step <  1.0e-6*mag_op) { step = 1.0e-6*mag_op; }
    fprintf(stderr, "    step = %12.8f\n", step);
    /* Establish a limit for the relative error: */
    double tol = 1.0e-4;
    /* Check the partial derivatives numerically: */
    bool_t debug; /* For {map}'s use. */
    int i, j, k;
    for (i = 0; i < 2; i++)
      { /* Evaluate {map} at points {oq[0..1]} , displaced from {op} along axis {i}: */
        r2_t iq[2];
        for (k = 0; k < 2; k++)
          { r2_t oq = (*op); 
            oq.c[i] += (2*k - 1)*step;
            r2_t *iqk = &(iq[k]);
            debug = FALSE;
            map(&oq, iqk, NULL, &debug);
            if (debug) { fitr_debug_point("      oq", &oq, NULL, "\n"); }
            if (debug) { fitr_debug_point("      iq", iqk, NULL, "\n"); }
            if (isnan(iqk->c[0]) || isnan(iqk->c[1])) { return; }
          }
        /* Check the derivatives of {ip} w.r.t. coordinate {i} of {op}: */
        for (j = 0; j < 2; j++)
          { /* Compute the derivative of coordinate {j} w.r.t coordinate {i}: */
            double dnum = (iq[1].c[j] - iq[0].c[j])/(2*step);
            /* Compare with the Jacobian returned by {map}: */
            double djac = J->c[i][j]; 
            double err = fabs(dnum-djac);
            if (err > tol*(fabs(dnum)+fabs(djac)))
              { fprintf(stderr, "    ** Jacobian is inconsistent (err = %+12.6e)\n", err); 
                fprintf(stderr, "    J[%d][%d] = %12.5f", i, j, djac);
                fprintf(stderr, "  numeric d_ip[%d]/d_op[%d] = %12.5f", i, j, dnum);
                fprintf(stderr, "\n");
              }
          }
      }
  } 

void fitr_debug_point(char *label, r2_t *p, r2x2_t *J, char *tail)
  {
    fprintf(stderr, "    %s", label);
    if (isnan(p->c[0]) || isnan(p->c[1]))
      { fprintf(stderr, " is invalid\n"); }
    else
      { fprintf(stderr, " = (%12.5f %12.5f)", p->c[0], p->c[1]);
        if (J != NULL)
          { fprintf
              ( stderr, 
                "  J = [[%6.1f %6.1f] [%6.1f %6.1f]]", 
                J->c[0][0], J->c[0][1], J->c[1][0], J->c[1][1]
              );
          }
      }
    fprintf(stderr, "%s", tail);
  }

void fitr_debug_rows(char *label, r2x2_t *R, char *tail)
  {
    int i;
    for (i = 0; i < 2; i++)
      { fprintf(stderr, "    %s", label);
        fprintf
          ( stderr, 
            "[%d] = [%6.1f %6.1f]", 
            i, R->c[i][0], R->c[i][1]
          );
        fprintf(stderr, "%s", tail);
      }
  }

void fitr_debug_direction(char *label, r2_t *v, char *tail)
  {
    fprintf(stderr, "    %s", label);
    fprintf(stderr, " = (%+8.5f %+8.5f)", v->c[0], v->c[1]);
    fprintf(stderr, "%s", tail);
  }

void fitr_debug_eigen(char *label, r2_t *e, char *tail)
  {
    fprintf(stderr, "    %s", label);
    fprintf(stderr, " = (%12.5f %12.5f)", e->c[0], e->c[1]);
    fprintf(stderr, "%s", tail);
  }
    
