#ifndef float_image_transform_H
#define float_image_transform_H

/* Tools for projective and barrel/pincushion image correction. */
/* Last edited on 2010-08-04 16:07:29 by stolfi */ 

#include <r2.h>
#include <r2x2.h>
#include <r3x3.h>
#include <bool.h>
#include <float_image.h>

typedef void fitr_transform_t (r2_t *p, r2_t *q, r2x2_t *J, bool_t *debugp);
  /* Type of a procedure that defines a geometric transformation
    between the domains of two images.  
    
    The procedure should apply the geometric map to point {*p} and
    store the result into {*q}. If {J} is not NULL, the procedure must
    also store into {*J} the Jacobian of the geometric map, evaluated at {*p}.
    In the Jacobian, element {J[i][j]} is the derivative of {q.c[i]}
    with respect to {p.c[j]}. 
    
    If the map is not defined at {*p}, the procedure should set {*q}
    to {(NAN,NAN)}; in that case, {*J} need not be modified. The
    procedure should do this, in particular, if either coordinate of
    {*p} is {NAN}.
    
    The procedure may set {*debugp=TRUE} to request debugging printout
    for that point. */

void fitr_transform
  ( float_image_t *iimg,   /* Input image. */
    fitr_transform_t *map, /* Output-to-input coordinate transformation. */
    float undef,           /* Sample value for undefined output pixels. */
    bool_t avg,            /* TRUE to average pixels, FALSE to add them. */
    int order,             /* Interpolation order. */
    float_image_t *oimg    /* Output image. */
  );
  /* Applies the transformation {map} to the image {iimg} and
    stores the result in {oimg}. 
    
    The {map} procedure should compute the *inverse* of the desired
    image transformation. More precisely, the value of the transformed
    image {timg} at point {op} is assumed to be the value of {iimg} at
    point {ip}, where {ip} is computed by {map(&op,&ip,&J,&debug)}.
    
    The {order} parameters determines how the value of {iimg} at a
    given point is obtained from nearby samples. See
    {fitr_interpolation_HELP_INFO} below.
    
    Point coordinates are relative to the standard {float_image}
    coordinate system for the relevant image. Namely, the first (X)
    coordinate increases with the column index, the second (Y)
    coordinate increases with the row index, pixels are squares with
    side 1, the image domain is the rectangle {[0 _ NX][0 _ NY]}, and
    the pixel in column 0, row 0 adjacent to the origin. Note that
    pixel centers have half-integer coordinates.
    
    If {avg} is true, each pixel of {oimg} is computed by a suitable
    average of the values of {timg} at several sampling points 
    over the pixel.  That is, the pixel of {oimg} is the 
    average of several values of {iimg} around the point {ip} corresponding
    to the pixel's center.  If {avg} is FALSE, the procedure computes a
    weighted integral, so as to approximately preserve the total
    integral of {iimg} over its domain. This option is appropriate,
    for instance, when {iimg} is a probability distribution, or a
    weight mask.
    
    For the averaging, the procedure assumes that the given {map} can
    be approximated by an affine (1st degree) map, which is determined
    by the output pixel center {op}, the corresponding point {ip} in
    the input domain, and the Jacobian matrix {J} returned by the
    {map} procedure. This approximation is assumed to be valid within
    a couple of pixels of the point {op}.
    
    If {map} returns {(NAN,NAN)} when applied to a pixel center
    {op}, the procedure assums that {op} has no corresponding point in
    the input domain, and sets all pixel's samples to the given
    {undef} value.

    The procedure implicitly extends the input image {ip} to infinity by
    replication of the pixels around its edges. This assumption is
    relevant when {map} returns a point {ip} that is not {(NAN,NAN)}
    but lies outside the domain of {iimg}.
    
    If {map} sets {debug} to TRUE, the procedure prints debugging
    information about the computation of the output pixel with center
    {op}.
  
    !!! Add the {red} parameter. !!! */
    
#define fitr_interpolation_HELP_INFO \
  "The input image value at a given point is obtained from nearby" \
  " samples by C0 bilinear interpolation (if the order is 0) or C1" \
  " bicubic interpolation (if the order is 1). Note that the latter may" \
  " overshoot the interval of the original samples."

/* SOME GEOMETRIC TRANSFORMATIONS */

void fitr_apply_projective_map(r2_t *p, r3x3_t *M, r2x2_t *J);
  /* Applies to {p} a projective transformation defined by the matrix {M}.
    Also post-multiplies the matrix {J} by the Jacobian of the map. */

void fitr_apply_radial_map(r2_t *p, r2_t *h, double kappa, r2x2_t *J);
  /* Applies to point {p} a radial deformation map with parameter
    {kappa}, centered at the origin. The inverse map can be obtained
    by negating {kappa}. Also post-multiplies the matrix {J} by the
    Jacobian of the map.
    
    The map is meant to be equivalent to that of R. Tsai's 1985 paper
    for small {kappa}, but it is not quite the same.
    
    The unit of the parameter {kappa} is {1/mm^2}. The procedure
    assumes that {h.c[0]} and {h.c[1]} are the width and height of
    each pixel, respectively, in mm. The pincushion distortion is
    undefined (returns {(NAN,NAN)}) if {p} lies on or outside the
    radius {R = sqrt(1/(2*kappa))} also in millimeters. So {kappa}
    should not exceed {2/d^2} where {d} is the diagonal of the image
    in millimeters. */

/* PERSPECTIVE TRANSFORMATIONS

  The procedures below require a {33} projective map matrix {T2I}
  from some system of /true coordinates/ to /image coordinates/, that
  is, points in the domain of a given input image {iimg}. */
  
void fitr_persp_copy_rectangle
  ( float_image_t *iimg,
    double xlo,         /* Min X in true coords. */
    double xhi,         /* Max X in true coords. */
    double ylo,         /* Min Y in true coords. */
    double yhi,         /* Max Y in true coords. */
    r3x3_t *T2I,        /* Projective map from true coords to image coords. */
    ix_reduction_t red, /* Index reduction method. */ 
    int x0,             /* First output image column. */
    int y0,             /* First output image row. */
    int NX,             /* Number of output image columns. */
    int NY,             /* Number of output image rows. */
    float_image_t *oimg
  );
  /* Copies a quadrilateral region {Q} from image {iimg} to a rectangular
    sub-image of image {oimg}.

    The receiving sub-image of {oimg} is defined by the lower indices
    {x0,y0}, and its dimensions in pixels {NX,NY}. The quadrilateral
    {Q} is the rectangle {R = [xlo_xhi]  [ylo_yhi]} in the in some
    /true coordinate system/, mapped by the {33} projective matrix
    {T2I}.

    The true rectangle {[xlo_xhi]  {ylo_yhi]} is mapped linearly to
    the rectangle {[x0_x0+NX}  {y0_y0+NY]} in the domain of {dst}.
    
    This procedure works no matter how small the {img} region is.
  
    !!! Add the {order} parameter if/where appropriate. !!! */

void fitr_persp_rectangle_average
  ( float_image_t *img, /* Image to sample. */
    interval_t tbox[],  /* Rectangle in true coordinates. */
    r3x3_t *T2I,        /* Projective map from true coords to image coords. */
    double mrg,         /* Safety border width (pixels). */
    ix_reduction_t red, /* Index reduction method. */ 
    double avg[]        /* (OUT) Pixel average. */
  );
  /* Computes the average {avg[0..NC-1]} of the pixels in image {img}
    within a quadrilateral region {Q}, where {NC=img->sz[0]}. The
    quadrilateral is defined by a recangle {tbox[0]  tbox[1]} in true
    coordinates, and the true-to-image matrix {T2I}. 
    
    The average considers only pixels that are inside the
    quadrilateral {Q} and at least {mrg} pixels away from the boundary
    of the quadrilatera; if there are no such pixels, sets {val} to
    {NAN}s.
    
    Any pixel index that falls outside its valid range {0..N-1} is
    reduced with {ix_reduce(i,N,red)}; if the result is {-1} the
    pixels are omitted from the average. */

void fitr_persp_disk_average
  ( float_image_t *img, /* Input image. */
    r2_t *ctr,          /* Disk center in true coordinates. */
    double rad,         /* Disk radius in true coordinates. */
    r3x3_t *T2I,        /* True-to-image projective map matrix. */
    ix_reduction_t red, /* Index reduction method. */ 
    double mrg,         /* Safety border width (pixels). */
    float avg[]         /* (OUT) average disk color. */
  );
  /* Computes the average {avg[0..NC-1]} of the pixels in image {img}
    within an ellipse {E}, where {NC=img->sz[0]}. The ellipse is defined
    by the center {ctr} and radius {rad} of a disk in true
    coordinates, and the true-to-image matrix {T2I}.

    The average considers only pixels that are inside the ellipse {E}
    and at least {mrg} pixels away from its border in the image. The
    procedure returns all {NAN}s in {avg} if there are no such pixels
    (that is, if the ellipse is too small of too narrow).
    
    Any pixel index that falls outside its valid range {0..N-1} is
    reduced with {ix_reduce(i,N,red)}; if the result is {-1} the
    pixels are omitted from the average. */

void fitr_persp_get_rectangle_bbox
  ( interval_t tbox[],   /* Rectangle in true coordinates. */
    r3x3_t *T2I,         /* True-to-image projective map matrix. */
    interval_t ibox[]    /* (OUT) bonding box in image coordinates. */
  );
  /* Computes an enclosing box for the image of a rectangle. 
    Requires the rectangle {tbox[0]  tbox[1]}, in
    true coordinates, and the true-to-image matrix {T2I}.
    The box is returned as two intervals {ibox[0..1]},
    one for ech axis. */
  
void fitr_persp_get_disk_bbox
  ( r2_t *ctr,         /* Disk center in true coordinates. */
    double rad,        /* Disk radius in true coordinates. */
    r3x3_t *T2I,       /* True-to-image projective map matrix. */
    interval_t ibox[]  /* (OUT) bonding box. */
  );
  /* Computes an enclosing box for the image of a disk. 
    Requires the disk's center {ctr} and radius {rad}, in
    true coordinates, and the true-to-image matrix {T2I}.
    The box is returned as two intervals {ibox[0..1]},
    one for ech axis. */

bool_t fitr_persp_pixel_is_inside_rectangle
  ( int x,              /* Pixel column in image. */
    int y,              /* Pixel row in image. */
    double mrg,         /* Safety margin (pixels). */
    r3x3_t *I2T,        /* Image-to-true projective matrix. */
    interval_t tbox[]   /* Rectangle in true coordinates. */
  );
  /* Returns TRUE iff the image pixel in column {x} and row {y},
    lies well inside the image of a rectangle.  The pixel
    is modeled as a unit square with center at {(x+0.5,y+0.5)}
    fattened by {mrg} all around.

    Requires the rectangle {tbox[0]  tbox[1]} in true
    coordinates, and the {33} image-to-true homogeneous projective
    matrix {I2T}. */

bool_t fitr_persp_pixel_is_inside_disk
  ( int x,        /* Pixel column in image. */
    int y,        /* Pixel row in image. */
    double mrg,   /* Safety margin (pixels). */
    r3x3_t *I2T,  /* Image-to-true projective matrix. */
    r2_t *ctr,    /* Center of disk in true coordinates. */
    double rad    /* Radius of disk in true coordinates. */
  );
  /* Returns TRUE iff the image pixel in column {x} and row {y},
    lies well inside the image of a disk.  The pixel
    is modeled as a unit square with center at {(x+0.5,y+0.5)}
    fattened by {mrg} all around.

    Requires the center {ctr} and radius {rad} of the disk in true
    coordinates, and the {33} image-to-true homogeneous projective
    matrix {I2T}. */

/* DEBUGGING AIDS */

void fitr_debug_pixel(char *label, double x, double y, int chns, float f[], char *tail);
  /* Prints the point {(x,y)} and the float-valued samples
    {f[0..chns-1]} to {stderr}, tagged with the string {label} and
    terminated by the string {tail}. */

void fitr_debug_point(char *label, r2_t *p, r2x2_t *J, char *tail);
  /* Prints the point {(x/w,y/w)} to {stderr}, tagged with the
    string {label} and terminated by the string {tail}. If the
    Jacobian {J} is not NULL, prints it too. */

void fitr_debug_rows(char *label, r2x2_t *R, char *tail);
  /* Prints the rows of {R} as vectors to {stderr}. Each row
    is tagged with the string {label} and its index,
    and terminated by the string {tail}. */

#endif
