/* See jspnm_image.h */
/* Last edited on 2010-06-22 16:27:57 by stolfi */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include <affirm.h>
#include <jsfile.h>
#include <jspnm.h>
#include <jspnm_image.h>

pnm_image_t *pnm_image_new(int cols, int rows, int chns)
  { /* Allocate header: */
    pnm_image_t *im = pnm_image_new_header(cols, rows, chns);
    /* Allocate pixels: */
    if ((cols != 0) && (rows != 0))
      { im->smp = pnm_image_alloc_pixel_array(cols, rows, chns); }
    return(im);
  }
  
void pnm_image_free(pnm_image_t *im)
  { if (im == NULL) return;
    if (im->smp != NULL) 
      { pnm_image_free_pixel_array(im->smp, im->cols, im->rows, im->chns); }
    free(im);
  }

pnm_image_t *pnm_image_new_header(int cols, int rows, int chns)
  { /* Allocate header: */
    pnm_image_t *im = (pnm_image_t *)pnm_malloc(sizeof(pnm_image_t));
    assert(im != NULL);
    /* Initialize fields: */
    im->cols = cols;
    im->rows = rows;
    im->chns = chns;
    /* Leave sample arrays as NULL: */
    im->smp = (pnm_sample_t**)NULL;
    /* Initialize the {maxval} field as 0 (invalid): */
    im->maxval = 0;
    return(im);
  }
  
pnm_sample_t** pnm_image_alloc_pixel_array(int cols, int rows, int chns)
  { int row;
    pnm_sample_t **smp = (pnm_sample_t **)pnm_malloc(rows*sizeof(pnm_sample_t *));
    for (row = 0; row < rows; row++)
      { smp[row] = pnm_image_alloc_pixel_row(cols, chns); }
    return smp;
  }
  
void pnm_image_free_pixel_array(pnm_sample_t **smp, int cols, int rows, int chns)
  { int row;
    if (smp == NULL) { return; }
    for (row = 0; row < rows; row++) 
      { pnm_image_free_pixel_row(smp[row], cols, chns); }
    free(smp);
  }

pnm_sample_t* pnm_image_alloc_pixel_row(int cols, int chns)
  { if (cols == 0)
      { return (pnm_sample_t *)NULL; }
    else
      { return (pnm_sample_t *)pnm_malloc(cols*chns*sizeof(pnm_sample_t)); }
  }
  
void pnm_image_free_pixel_row(pnm_sample_t *row, int cols, int chns)
  { 
    if (row != NULL) { free(row); }
  }

pnm_sample_t pnm_image_get_sample(pnm_image_t *im, int c, int x, int y)
  { if ((c < 0) || (c >= im->chns)) { return 0; }
    if ((x < 0) || (x >= im->cols)) { return 0; }
    if ((y < 0) || (y >= im->rows)) { return 0; }
    return im->smp[y][x*im->chns + c];
  }

void pnm_image_set_sample(pnm_image_t *im, int c, int x, int y, pnm_sample_t pv)
  { if ((c < 0) || (c >= im->chns)) { pnm_error("invalid channel"); }
    if ((x < 0) || (x >= im->cols)) { pnm_error("invalid column"); }
    if ((y < 0) || (y >= im->rows)) { pnm_error("invalid row"); }
    if (pv > im->maxval) { pnm_error("invalid sample value"); }
    im->smp[y][x*im->chns + c] = pv;
  }

pnm_image_t *pnm_image_crop(pnm_image_t *im, int ic, int nc, int ix, int nx, int iy, int ny)
  { demand((ic >= 0) && (ic+nc <= im->chns), "invalid channel range");
    demand((ix >= 0) && (ix+nx <= im->cols), "invalid column range");
    demand((iy >= 0) && (iy+ny <= im->rows), "invalid row range");
    pnm_image_t *om = pnm_image_new(nx, ny, nc);
    om->maxval = im->maxval;
    int chn, col, row;
    for (row = 0; row < ny; row++)
      { for (col = 0; col < nx; col++)
          { pnm_sample_t *pi = &(im->smp[iy + row][(ix + col)*im->chns + ic]);
            pnm_sample_t *po = &(om->smp[row][col*nc]);
            for (chn = 0; chn < nc; chn++) { (*po) = (*pi); po++; pi++; }
          }
      }
    return om;
  }        

pnm_image_t *pnm_image_read(char *name, bool_t verbose)
  { FILE *rd = open_read(name, verbose);
    pnm_image_t *im = pnm_image_fread(rd);
    if (rd != stdin) { fclose(rd); }
    return im;
  }

pnm_image_t *pnm_image_fread(FILE *rd)
  { /* We cannot rely on {pnm_readpnm} because PBMPLUS has different pixel layout. */
    /* Read file header: */
    int cols, rows, chns;
    pnm_sample_t maxval;
    bool_t raw, bits;
    pnm_format_t format;
    pnm_read_header(rd, &cols, &rows, &chns, &maxval, &raw, &bits, &format);
    /* Allocate image and set maxval: */
    pnm_image_t *im = pnm_image_new(cols, rows, chns);
    im->maxval = maxval;
    /* Read pixels: */
    int row;
    for (row = 0; row < rows; row++)
      { pnm_read_pixels(rd, im->smp[row], cols, chns, maxval, raw, bits); }
    return(im);
  }
  
void pnm_image_write(char *name, pnm_image_t *im, bool_t forceplain, bool_t verbose)
  { FILE *wr = open_write(name, verbose);
    pnm_image_fwrite(wr, im, forceplain);
    if (wr != stdout) { fclose(wr); }
  }
    
void pnm_image_fwrite(FILE *wr, pnm_image_t *im, bool_t forceplain)
  { /* We cannot rely on {pnm_writepnm} because PBMPLUS has different pixel layout. */
    if (im->maxval <= 0){ pnm_error("invalid maxval (%u)", im->maxval); }
    pnm_format_t format;
    bool_t raw;
    bool_t bits; 
    pnm_choose_output_format(im->maxval, im->chns, forceplain, &format, &raw, &bits);
    pnm_write_header(wr, im->cols, im->rows, im->maxval, format);
    int row;
    for (row = 0; row < im->rows; row++)
      { pnm_write_pixels(wr, im->smp[row], im->cols, im->chns, im->maxval, raw, bits); }
    fflush(wr);
  }

