/* See ppm_medcutx.h
** Last edited on 2006-11-13 22:58:42 by stolfi
**
** Copyright (C) 1989, 1991 by Jef Poskanzer. See note at end of file.
*/

#include <affirm.h>
#include <jspnm.h>

#include <jsppm_medcutx.h>
#include <jsppm_xhist.h>
#include <jsppm_xtable.h>

/* Options for the median cut algorithm: */

/* How to choose the axis of greatest variation: */

#define LARGE_LUM
/* #define LARGE_NORM */

/* How to choose the representative color in each box: */

#define REP_AVERAGE_PIXELS
/* #define REP_CENTER_BOX */
/* #define REP_AVERAGE_COLORS */

#define vmin(X,Y) ((X)<(Y) ? (X) : (Y))
#define vmax(X,Y) ((X)>(Y) ? (X) : (Y))

/* Internal types */

typedef struct 
  { int lo, hi;       /* Index range in color histogram */
    float spread;     /* Box size */
    int longaxis;     /* Axis of maximum spread (0,1,2) for (R,G,B) */
    float cR, cG, cB; /* Box centroid */
    ppm_pixel_t rep;  /* Pixel representative */
  } ppm_box;
  
typedef ppm_box* ppm_box_vector;

typedef int (*qcomparefn)(const void *, const void *) ;

typedef int (*matchfn)(long R, long G, long B, ppm_xhist_vector cm, int ncolors);
  /* A function that finds the best match to {(R,G,B)} in the colormap {cm}.
    Note that {(R,G,B)} may be somewhat outside of the color cube. */

/* Internal prototypes */

void ppm_set_box(
    ppm_box_vector b,
    ppm_xhist_vector ch,
    int lo, int hi,
    pnm_sample_t maxval
  );
  /* Sets {lo} and {hi}, and computes {center}, {rep}, {longaxis}, 
    and {spread}. */

void ppm_box_center(
    ppm_xhist_vector ch,
    int lo, int hi,
    pnm_sample_t maxval,
    float *cRp, float *cGp, float *cBp,
    ppm_pixel_t *repp
  );
  /*
    Computes a "central" rgb value for this box,
    and rounds it to a "representative" pixel.
  */

void ppm_box_spread(
    ppm_xhist_vector ch,
    int lo,
    int hi,
    ppm_pixel_t rep,
    pnm_sample_t maxval,
    int *laxp,
    float *spreadp
  );
  /* Compute the box spread {*spreadp} and largest 
    dimension {*laxp}, relative to the appointed representative {rep}. */

int ppm_red_compare(const ppm_xhist_vector ch1, const ppm_xhist_vector ch2);

int ppm_green_compare(const ppm_xhist_vector ch1, const ppm_xhist_vector ch2);

int ppm_blue_compare(const ppm_xhist_vector ch1, const ppm_xhist_vector ch2);

int ppm_spread_compare(const ppm_box *b1, const ppm_box *b2);

ppm_box_vector new_ppm_box_vector(int n);

ppm_xhist_vector new_ppm_xhist_vector(int n);

/* Implementation */

ppm_xhist_vector ppm_median_cut_x
  ( ppm_xhist_vector ch,
    int colors, 
    pnm_sample_t maxval,
    int *newcolorsp
  )
  {
    ppm_xhist_vector cm; /* The colormap */
    ppm_box_vector bv;   /* The box tree */
    register int i;
    int boxes;

    bv = new_ppm_box_vector(*newcolorsp);

    /* Set up the initial box. */
    
    ppm_set_box(&(bv[0]), ch, 0, colors, maxval);
    boxes = 1;

    /* Main loop: split boxes until we have enough. */
    while (boxes < (*newcolorsp))
      { 
        register int lo, hi;
        int lax;
        float ctrv;
        qcomparefn cmp;
        
        /* Box 0 should have the largest spread. */
        if (bv[0].spread <= 0.0) break; /* exact */
        lo = bv[0].lo;
        hi = bv[0].hi;
        if (lo >= hi) pnm_error("bad ppm_box spread");
        lax = bv[0].longaxis;
        
        /* Sort box colors in the direction of largest spread: */
        switch(lax)
          { 
            case 0: cmp = (qcomparefn) ppm_red_compare;   ctrv = bv[0].cR; break;
            case 1: cmp = (qcomparefn) ppm_green_compare; ctrv = bv[0].cG; break;
            case 2: cmp = (qcomparefn) ppm_blue_compare;  ctrv = bv[0].cB; break;
            default: pnm_error("huh?"); cmp = NULL; ctrv = 0;
          }
        qsort((void*)&(ch[lo]), hi-lo+1, sizeof(struct ppm_xhist_item), cmp); 
        
        /* Split colors in two groups at centroid: */
        for (i = lo; i <= hi; ++i)
          { pnm_sample_t v = ch[i].color.c[lax];
            if ((float)v >= ctrv) break;
          }
        if ((i > hi) || (i <= lo)) pnm_error("bad centroid in pgm box");
        ppm_set_box(&(bv[0]), ch, lo, i-1, maxval);
        ppm_set_box(&(bv[boxes]), ch, i, hi, maxval);
        ++boxes;

        /* Sort to bring the worst boxes to the top. */
        qsort((void*) bv, boxes, sizeof(ppm_box), (qcomparefn) ppm_spread_compare);
      }

    /* Ok, we've got enough boxes.  Collect their centroids: */
    cm = new_ppm_xhist_vector(boxes);
    for (i = 0; i < boxes; ++i) 
      { cm[i].color = bv[i].rep; }
    (*newcolorsp) = boxes;
    free(bv);
    return (cm);
  }

void ppm_set_box
  ( ppm_box_vector b,
    ppm_xhist_vector ch,
    int lo, int hi,
    pnm_sample_t maxval
  )
  {
    b->lo = lo;
    b->hi = hi;
    ppm_box_center(ch, lo, hi, maxval, &(b->cR), &(b->cG), &(b->cB), &(b->rep));
    ppm_box_spread(ch, lo, hi, b->rep, maxval, &(b->longaxis), &(b->spread));
  }

void ppm_box_center
  ( ppm_xhist_vector ch,
    int lo, int hi,
    pnm_sample_t maxval,
    float *cRp, float *cGp, float *cBp,
    ppm_pixel_t *repp
  )
  {
    long w, v;
    float fv, fw;
    float sumr = 0, sumg = 0, sumb = 0, sumw = 0;
    pnm_sample_t maxr = 0, maxg = 0, maxb = 0;
    pnm_sample_t minr = maxval, ming = maxval, minb = maxval;
    ppm_pixel_t rep;
    int i;
    for (i = lo; i <= hi; ++i)
      { w = ch[i].value;
        fw = (float)w;

        v = ch[i].color.c[0];
        minr = vmin(minr, v);
        maxr = vmax(maxr, v);
        fv = (float)v;
        sumr += fw*fv; 

        v = ch[i].color.c[1];
        ming = vmin(ming, v);
        maxg = vmax(maxg, v);
        fv = (float)v;
        sumg += fw*fv; 
       
        v = ch[i].color.c[2];
        minb = vmin(minb, v);
        maxb = vmax(maxb, v);
        fv = (float)v;
        sumb += fw*fv; 
       
        sumw += fw; 
      }
    if (sumw <= 0.0) pnm_error("empty pgm_box");

    if (minr >= maxr)
      { (*cRp) = minr; v = minr; }
    else
      { fv = sumr / sumw; (*cRp) = fv;
        v = (long)(fv + 0.5);
        if (v > maxr) v = maxr;
        if (v < minr) v = minr;
      }
    rep.c[0] = v;
    
    if (ming >= maxg)
      { (*cGp) = ming; v = ming; }
    else
      { fv = sumg / sumw;
        (*cGp) = fv;
        v = (long)(fv + 0.5);
        if (v > maxg) v = maxg;
        if (v < ming) v = ming;
      }
    rep.c[1] = v;

    if (minb >= maxb)
      { (*cBp) = minb; v = minb; }
    else
      { fv = sumb / sumw;
        (*cBp) = fv;
        v = (long)(fv + 0.5);
        if (v > maxb) v = maxb;
        if (v < minb) v = minb;
      }
    rep.c[2] = v;

    (*repp) = rep;
    /*  
      pm_message("[%d..%d] = [%d-%d]x[%d-%d]x[%d-%d] ctr=(%.1f,%.1f,%.1f) rep=(%d,%d,%d)",
        lo, hi,
        minr, maxr, ming, maxg, minb, maxb,
        (*cRp), (*cGp), (*cBp),
        PPM_GETR(rep), PPM_GETG(rep), PPM_GETB(rep)
      );
    */
  }

void ppm_box_spread
  ( ppm_xhist_vector ch,
    int lo,
    int hi,
    ppm_pixel_t rep,
    pnm_sample_t maxval,
    int *laxp,
    float *spreadp
  )
  {
    float fv, fw;
    float sumRR = 0.0, sumGG = 0.0, sumBB = 0.0;
    pnm_sample_t v;
    long w;
    register int i;
    float frepR = (float)rep.c[0];
    float frepG = (float)rep.c[1];
    float frepB = (float)rep.c[2];
    for (i = lo; i <= hi; ++i)
      {
        w = ch[i].value;
        fw = (float)w;
        
        v = ch[i].color.c[0];
        fv = (float)v - frepR;
        sumRR += fw*fv*fv;
        
        v = ch[i].color.c[1];
        fv = (float)v - frepG;
        sumGG += fw*fv*fv;
        
        v = ch[i].color.c[2];
        fv = (float)v - frepB;
        sumBB += fw*fv*fv;
      }
    if ((sumRR >= sumGG) && (sumRR >= sumBB))
      { (*laxp) = 0; 
        (*spreadp) = sumRR;
      }
    else if ((sumGG >= sumRR) && (sumRR >= sumBB))
      { (*laxp) = 1; 
        (*spreadp) = sumGG;
      }
    else /* if ((sumbb >= sumgg) && (sumbb >= sumrr)) */
      { (*laxp) = 2; 
        (*spreadp) = sumBB;
      }
  }

int ppm_red_compare(ppm_xhist_vector ch1, ppm_xhist_vector ch2)
  {
    return (int) ch1->color.c[0] - (int) ch2->color.c[0];
  }

int ppm_green_compare(ppm_xhist_vector ch1, ppm_xhist_vector ch2)
  {
    return (int) ch1->color.c[1] - (int) ch2->color.c[1];
  }

int ppm_blue_compare(ppm_xhist_vector ch1, ppm_xhist_vector ch2)
  {
    return (int) ch1->color.c[2] - (int) ch2->color.c[2];
  }

int ppm_spread_compare(const ppm_box *b1, const ppm_box *b2)
  {
    return ((int)
      (b1->spread > b2->spread ? -1 : 
      (b1->spread < b2->spread ?  1 :
      0))
    );
  }

ppm_box_vector new_ppm_box_vector(int n)
  { ppm_box_vector bv = (ppm_box_vector)pnm_malloc(n*sizeof(ppm_box));
    return(bv);
  }

ppm_xhist_vector new_ppm_xhist_vector(int n)
  { ppm_xhist_vector ch = (ppm_xhist_vector)pnm_malloc(n*sizeof(struct ppm_xhist_item));
    return(ch);
  }

/* Copyright (C) 1989, 1991 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

