#define PROG_NAME "test_jsmath"
#define PROG_DESC "test of {jsmath.h}"
#define PROG_VERS "1.0"

/* Last edited on 2008-09-29 06:00:55 by stolfi */ 
/* Created on 2007-01-02 by J. Stolfi, UNICAMP */

#define test_jsmath_COPYRIGHT \
  "Copyright  2007  by the State University of Campinas (UNICAMP)"

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include <values.h>

#include <jsmath.h>
#include <jsrandom.h>
#include <affirm.h>
#include <bool.h>

#define bug(FMT_AND_ARGS...) \
  do { \
    fprintf(stderr, "%s:%d: ", __FILE__, __LINE__); \
    fprintf(stderr, FMT_AND_ARGS); \
    fprintf(stderr, "\n"); \
    exit(1); \
  } while(0)

double zrandom(void); /* A random double with random magnitude. */

void test_gcd(int nt);
void test_lcm(int nt);
void test_imod(int nt);
/* !!! should test ifloor, iceil */
void test_ipow(int nt);
void test_uint64_mul(int nt);

// #define N_double_nice 10
// 
// static double double_nice[N_double_nice] = 
//   { 
//     4.45014771701440277e-308, /* 8/DBL_MAX. */                   
//     7.45834073120020674e-155, /* 1/sqrt(DBL_MAX). */
//     5.00000000000000000e-001,
//     9.99999999999999889e-001,
//     1.00000000000000000e+000, 
//     1.00000000000000022e+000,
//     2.00000000000000000e+000,
//     1.34078079299425971e+154, /* sqrt(DBL_MAX). */
//     2.24711641857789464e+307, /* DBL_MAX/8. */                   
//     0
//   };

#define N_uint32_nice 13

static uint32_t uint32_nice[N_uint32_nice] = 
  { 1, 
    2, 
    3, 
    4, 
    12, 
    1023, 
    1024, 
    1025,
    2147483647UL,  /* 2^31-1 */
    2147483648UL, /* 2^31 */
    2147483649UL, /* 2^31+1 */
    4294967295UL, /* 2^32-1 */
    0
  };

#define N_uint64_nice 14

static uint64_t uint64_nice[N_uint64_nice] = 
  { 1, 
    2, 
    3, 
    4, 
    12, 
    1023, 
    1024, 
    1025, 
    4294967295ULL, /* 2^32-1 */
    4294967296ULL, /* 2^32 */
    4294967297ULL, /* 2^32+1 */
    18446744073709551614ULL, /* 2^64-2 */
    18446744073709551615ULL, /* 2^64-1 */
    0
  };

#define N_int64_nice 28

static int64_t int64_nice[N_int64_nice] = 
  { +1,                       -1, 
    +2,                       -2, 
    +3,                       -3, 
    +4,                       -4, 
    +12,                      -12,
    +1023,                    -1023,                  
    +1024,                    -1024,                  
    +1025,                    -1025,                  
    +4294967295LL,            -4294967295LL,             /* 2^32-1 */
    +4294967296LL,            -4294967296LL,             /* 2^32 */
    +4294967297LL,            -4294967297LL,             /* 2^32+1 */
    +9223372036854775806LL,   -9223372036854775806LL,    /* 2^63-2 */
    +9223372036854775807LL,   -9223372036854775807LL,    /* 2^63-1 */
    ((int64_t)-9223372036854775807LL)-1,                 /* -2^63 */
    0
  };

int main (int argn, char **argv)
  { test_gcd(200);
    test_lcm(200);
    test_imod(200);
    test_ipow(200);
    test_uint64_mul(200);

    return 0;
  }

void test_gcd(int nt)
  { fprintf(stderr, "Checking {gcd}...\n");
    int i,j;
    for (i = 0; i < N_uint64_nice + nt; i++)
      { uint64_t a = (i < N_uint64_nice ? uint64_nice[i] : uint64_random());
        for (j = 0; j < N_uint64_nice + nt; j++)
          { uint64_t b = (j < N_uint64_nice ? uint64_nice[j] : uint64_random());
            uint64_t c = gcd(a, b);
            if (c == 0)
              { affirm((a == 0) && (b == 0), "zero gcd"); }
            else
              { affirm(a % c == 0, "gcd does not divide a");
                affirm(b % c == 0, "gcd does not divide b");
                affirm(gcd(a/c, b/c) == 1, "gcd(a/c,b/c) is not 1");
                /* Should check maximality... */
              }
          }
      }
  }

void test_lcm(int nt)
  { fprintf(stderr, "Checking {lcm}...\n");
    int i,j;
    for (i = 0; i < N_uint64_nice + nt; i++)
      { uint64_t a = (i < N_uint64_nice ? uint64_nice[i] : uint64_random());
        for (j = 0; j < N_uint64_nice + nt; j++)
          { uint64_t b = (j < N_uint64_nice ? uint64_nice[j] : uint64_random());
            uint64_t g = gcd(a, b);
            if ((a == 0) || (b == 0))
              { affirm(lcm(a,b) == 0, "lcm of zero is not zero"); }
            else if (a/g <= UINT64_MAX/b)
              { int64_t c = lcm(a, b);
                affirm(c != 0, "zero lcm");
                affirm(c % a == 0, "lcm does not divide a");
                affirm(c % b == 0, "lcm does not divide b");
                /* Should check minimality... */
              }
          }
      }
  }

void test_imod(int nt)
  { fprintf(stderr, "Checking {imod}...\n");
    int i,j;
    for (i = 0; i < N_int64_nice + nt; i++)
      { int64_t a = (i < N_int64_nice ? int64_nice[i] : int64_random());
        for (j = 0; j < N_int64_nice + nt; j++)
          { int64_t b = (j < N_int64_nice ? int64_nice[j] : int64_random());
            if (b > 0)
              { int64_t c = imod(a, b);
                affirm(c >= 0, "imod is negative");
                affirm(c < b, "imod is too big");
                /* Check {(a - c)%b}, but beware of overflow: */
                if (a < 0)
                  { affirm((a + (b - c)) % b == 0, "imod is not remainder"); }
                else
                  { affirm((a - c) % b == 0, "imod is not remainder"); }
                  
                /* Should check minimality... */
              }
          }
      }
  }

void test_ipow(int nt)
  { fprintf(stderr, "Checking {ipow}...\n");
    int iy,ix;
    for (iy = 0; iy < N_uint32_nice + nt; iy++)
      { uint32_t y = (iy < N_uint32_nice ? uint32_nice[iy] : (uint32_t)int64_random());
        for (ix = 0; ix < N_int64_nice + nt; ix++)
          { int64_t x = (ix < N_int64_nice ? int64_nice[ix] : int64_random());
            if ((y > 1) && ((x > +1) || (x < -1)))
              { /* Reduce {y} if needed so that the power will not overflow: */
                uint32_t ymax;
                if ((x > 0) || (y % 2 == 0))
                  { int64_t pmax = +9223372036854775807LL; /* 2^63-1 */
                    ymax = 0; do { pmax /= x; ymax++; } while ((pmax/x) != 0);

                  }
                else
                  { int64_t pmax = ((int64_t)-9223372036854775807LL)-1; /* -2^63 */
                    ymax = 0; do { pmax /= x; ymax++; } while ((pmax/x) != 0);
                  }
                if (y > ymax) { y = ymax; }  
              }
            /* Compute the power {z}: */
            int64_t z = ipow(x, y);
            /* Check whether {x} divides {z} {y} times exactly, leaving 1: */
            if (y == 0)
              { affirm(z == 1, "x^0 is not 1"); }
            else if (x == 0)
              { affirm(z == 0, "0^y is not 0"); }
            else if (x == +1)
              { affirm(z == +1, "1^y is not 1"); }
            else if (x == -1)
              { affirm(z == (y % 2 == 0 ? +1 : -1), "(-1)^y is not (-1)^(y%2)"); }
            else if (y == 1)
              { affirm(z == x, "x^1 is not x"); }
            else
              { int64_t p = z;
                while ((p > 1) || (p < -1))
                  { int64_t q = p/x;
                    affirm(p - q*x == 0, "x^y is not a power of x"); 
                    p = q;
                  }
                if (p != 1) { fprintf(stderr, "x = %lld y = %u z = %lld p = %lld\n", x, y, z, p); }
                affirm(p == 1, "power is not divisible y times by x");
              }
          }
      }
  }
  
void test_uint64_mul(int nt)
  { fprintf(stderr, "Checking {uint64_mul}...\n");
    int i,j;
    for (i = 0; i < N_uint64_nice + nt; i++)
      { uint64_t x = (i < N_uint64_nice ? uint64_nice[i] : uint64_random());
        for (j = 0; j < N_uint64_nice + nt; j++)
          { uint64_t y = (j < N_uint64_nice ? uint64_nice[j] : uint64_random());
            uint64_t Z[2];
            uint64_mul(x, y, Z);
            /* Slow multiplication: */
            int i;
            uint64_t P[2] = { 0, 0 };
            uint64_t M = 1;
            for (i = 0; i < 64; i++)
              { if ((M & x) != 0)
                  { /* Add {y} shifted by {i} onto {P}: */
                    uint64_t Q = P[0] + (y << i);
                    if (Q < P[0]) { /* carry happened: */ P[1] += 1; }
                    P[0] = Q;
                    if (i > 0) { P[1] += (y >> (64 - i)); }
                  }
                M = (M << 1);
              }
                    
            if ((P[0] != Z[0]) || (P[1] != Z[1]))
              { 
                fprintf(stderr, "x = %llu  y = %llu", x, y);
                fprintf(stderr, "  Z = %llu 2^64 + %llu", Z[1], Z[0]);
                fprintf(stderr, "  P = %llu 2^64 + %llu\n", P[1], P[0]);
                affirm(FALSE, "bug");
              }
          }
      }
  }

#define LOG_DBL_MAX (709.78271289338397)

double zrandom(void)
  { 
    return drandom()*exp(LOG_DBL_MAX*(2*drandom() - 1));
  }
