/* See {sve_minn.h} */
/* Last edited on 2009-02-28 13:29:30 by stolfi */

#define _GNU_SOURCE
#include <math.h>
#include <assert.h>

#include <gauss_elim.h>
#include <bool.h>
#include <vec.h>
#include <jsmath.h>
#include <rmxn.h>
#include <rmxn_extra.h>

#include <sve_minn.h>

#define Pr fprintf
#define Er stderr

/* INTERNAL PROTOTYPES */

void sve_clip_step
  ( int n,
    double x0[],
    double x[],
    double y[],
    double radius,
    double dMax,
    double rMax,
    bool_t debug
  );
  /* Trims the computed minimum {y[0..n-1]} so that 
    it is at most {dMax} away from the starting point {x0},
    and at most {min(8*radius,rMax)} away from the current center {x}. 
    Assumes that {x} itself is at most {dMax} away from {x0}. */

void sve_take_step
  ( int n,
    double x[],
    double *FxP,
    double y[],
    double Fy,
    int nv,
    double v[],
    int nf,
    double Fv[],
    double radius,
    double *dP,
    bool_t debug
  );
  /* Updates the current center point {x[0..n-1]} and its function
    value {*FxP}, and defines the nominal length {*dP} of that step.

    The candidates for the next point are the current center {x}
    itself, the estimated minimum {y} computed by quadratic
    minimization, and the probe points of this iteration (the vertices
    {V(i)} and edge midpoints {V(i,j)} of the current simplex).
    
    Assumes that {y} has been clipped as appropriate; that {*FxP} and
    {Fy} are the function values at {x} and {y}; that {v} contains the
    coordinates of the simplex vertices, by rows; and that {Fv}
    contains the function values at all probe points. Also assumes
    that {radius} is the radius of the current simplex. */

void sve_print_probes(FILE *wr, int nv, int n, double v[], int nf, double Fv[]);
  /* Prints the probe values {Fv[0..nf-1]} and the probe points.
    Assumes that {v[0..nv*n-1]} are the coordinates of the vertices,
    stored by rows. */

void sve_minn_step(int n, double Fv[], double cm[])
  { int nv = n+1; /* Number of vertices in simplex. */
    int rows = nv+1; /* {n+1} stationary eqs, and one unit-sum eq. */
    int cols = nv+2; /* {n+1} barycentric coords, one Lagrange multip, and the indep term. */
    double M[rows*cols];
    int i, j;
    /* Fill in the main submatrix: */
    int jb = cols - 1; /* Column of independent term. */
    for (i = 0; i < nv; i++)
      { for (j = 0; j < nv; j++)
          { int ij = (j <= i ? i*(i+1)/2 + j : j*(j+1)/2 + i);
            double Fij = Fv[ij];
            M[i*cols + j] = Fij;
            if (i == j) { M[i*cols + jb] = Fij/4; }
          }
      }
    /* Fill in the row {nv} and column {nv} with the unit-sum constraint: */
    int ije = n+1; /* Index of constraint row & column. */
    for (i = 0; i <= n; i++) { M[i*cols + ije] = 1; M[ije*cols + i] = 1; }
    M[ije*cols + ije] = 0;
    M[ije*cols + jb] = 1;
    /* Solve the system: */
    gsel_triangularize(rows, cols, M, TRUE);
    gsel_diagonalize(rows, cols, M, TRUE);
    gsel_normalize(rows, cols, M, TRUE);
    double x[n+2];
    int r = gsel_extract_solution(rows, cols, M, 1, x, TRUE);
    if (r < rows) 
      { Pr(Er, "%s: warning - indeterminate system, r = %d\n", __FUNCTION__, r); }
    /* Extract the solution: */
    double sum = 0.0;
    for (i = 0; i <= n; i++) { cm[i] = x[i]; sum += cm[i]; }
    if (fabs(sum - 1.0) > 0.5e-7) 
      { Pr(Er, "%s: warning - normalization failed, sum = %24.16e\n", __FUNCTION__, sum); }
    /* Just to be sure: */
    if ((sum != 0) && (sum != 1)) { for (i = 0; i <= n; i++) { cm[i] /= sum; } }
  }

void sve_sample_function(int n, sve_goal_t *F, double v[], double Fv[])
  { double x[n];
    int nv = n + 1;
    int i, j;
    for (i = 0; i < nv; i++)
      { for (j = 0; j <= i; j++)
          { /* Set {x[0..n-1]} to the midpoint of simplex corners {i,j}: */
            double *vi = &(v[i*n]);
            double *vj = &(v[j*n]);
            int k; for (k = 0; k < n; k++) { x[k] = (vi[k] + vj[k])/2; }
            /* Get the function's value {F(x)} at {x}, store {F(x)} into {Fv}: */
            int ij = i*(i+1)/2 + j;
            Fv[ij] = F(n, x);
          }
      }
  }

void sve_minn_iterate
  ( int n, 
    sve_goal_t *F, 
    sve_pred_t *OK,
    double x[], 
    double rIni,
    double rMin, 
    double rMax,
    double dMax,
    int maxEvals,
    bool_t debug
  )
  { /* Save first guess: */
    double x0[n];   /* Cartesian coords of initial guess. */
    int k; 
    for (k = 0; k < n; k++) { x0[k] = x[k]; }
    
    /* Allocate storage for the simplex: */
    int nv = n+1; /* Number of vertices in simplex. */
    double_vec_t vv = double_vec_new(nv*n); 
    double *v = vv.e; /* Cartesian coords of simplex vertices. */
    
    /* Allocate storage for the sample values: */
    int nf = (n+1)*(n+2)/2; /* Number of probe points. */
    double_vec_t Fvv = double_vec_new(nf);
    double *Fv = Fvv.e; /* Sampled function values. */

    /* Iteration variables: */
    double cm[nv]; /* Barycentric coords of next solution. */
    double y[n];    /* Cartesian coords of next solution. */
    double radius = rIni; /* Current probe simplex radius: */
    double dPrev = -1; /* Distance moved in previous iteration (-1 if none). */
    int iter = 0;   /* Counts main iterations. */
    int nEvals = 0; /* Counts calls to {F}. */
    
    /* Compute the initial function value {Fx}: */
    double Fx = F(n, x);
    nEvals++;
        
    while(TRUE) 
      { if (debug) 
          { Pr(Er, "  iteration %4d radius = %12.8f\n", iter, radius);
            Pr(Er, "    cur x = \n");
            rn_gen_print(Er, n, x, "%20.16f", "    [ ", "\n      ", "   ]\n");
            Pr(Er, "    function = %22.16e\n", Fx);
            Pr(Er, "\n");
          }
        /* Check budget: */
        if (nEvals > maxEvals) 
          { if (debug) { Pr(Er, "  budget was exhausted\n"); }
            break;
          }
        /* Check for termination: */
        if ((OK != NULL) && (OK(n, x, Fx))) 
          { if (debug) { Pr(Er, "  client is satisfied\n"); }
            break;
          }
        /* Choose a simplex around {x}: */
        rmxn_regular_simplex(n, v);
        rmxn_spin_rows(nv, n, v, v);
        /* Compute the scaling factor {scale} for the regular simplex: */
        double scale = radius/rmxn_regular_simplex_radius(n);
        rmxn_scale(nv, n, scale, v, v);
        rmxn_shift_rows(nv, n, v, x, v);
        /* Evaluate the goal function at sample points: */
        sve_sample_function(n, F, v, Fv);
        if (debug)
          { /* Print the current simplex: */
            Pr(Er, "    radius = %20.16f\n", radius);
            Pr(Er, "    simplex vertices:\n");
            rmxn_gen_print(Er, nv, n, v, "%20.16f", "", "\n", "", "    [ ", " ", " ]");
            Pr(Er, "\n");
            /* Print the probe points and values: */
            Pr(Er, "    probe values and points:\n");
            sve_print_probes(Er, nv, n, v, nf, Fv); 
            Pr(Er, "\n");
          }
        nEvals += nf;
        /* Optimize as a quadratic: */
        sve_minn_step(n, Fv, cm);
        /* Convert solution to Cartesian coordinates: */
        rmxn_map_row(nv, n, cm, v, y);
        if (debug) 
          { Pr(Er, "    raw y = \n");
            rn_gen_print(Er, n, y, "%20.16f", "    [ ", "\n      ", " ]\n");
            Pr(Er, "\n");
          }
        /* Enforce maxmum step limits: */
        sve_clip_step(n, x0, x, y, radius, dMax, rMax, debug);
        /* Evaluate at new point: */
        double Fy = F(n, y);
        if (debug) 
          { Pr(Er, "    clp y = \n");
            rn_gen_print(Er, n, y, "%20.16f", "    [ ", "\n      ", " ]\n");
            Pr(Er, "    function = %22.16e\n", Fy);
            Pr(Er, "\n");
          }
        /* Update the next point {x} and set the nominal step length {d}: */
        double d;
        sve_take_step(n, x, &Fx, y, Fy, nv, v, nf, Fv, radius, &d, debug);
        if (debug) 
          { Pr(Er, "    new x = \n");
            rn_gen_print(Er, n, x, "%20.16f", "    [ ", "\n      ", " ]\n");
            Pr(Er, "    function = %22.16e\n", Fx);
            Pr(Er, "    step len = %22.16e\n", d);
            Pr(Er, "\n");
          }
        /* Check for convergence: */
        double dEst = (dPrev <= d ? +INF : d*d/(dPrev - d));
        if (dEst <= rMin) 
          { if (debug) 
              { Pr(Er, "  seems to have converged:\n");
                Pr(Er, "    dPrev = %22.16e\n", dPrev);
                Pr(Er, "    d =     %22.16e\n", d);
                Pr(Er, "    dEst =  %22.16e\n", dEst);
              }
            break;
          }
        /* The motion is at least the cener-to-edge radius of the simplex: */
        /* Remember how much we moved in this iteration: */
        dPrev = d;
        /* Adjust radius for next iteration: */
        double rNew = d/2;
        if (rNew < 0.5*radius) { rNew = 0.5*radius; }
        if (rNew > 2.0*radius) { rNew = 2.0*radius; }
        radius = rNew;
        if (radius < rMin) { radius = rMin; }
        if (radius > rMax) { radius = rMax; }
        iter++;
      }
    free(v);
    free(Fv);
  }

void sve_clip_step
  ( int n,
    double x0[],
    double x[],
    double y[],
    double radius,
    double dMax,
    double rMax,
    bool_t debug
  )
  {
    int k;
    /* Enforce the limit {dMax} for the total motion: */
    double dyx0 = rn_dist(n, x0, y);
    if (dyx0 > dMax) 
      { /* Moved too much, curb it: */
        double s = dMax/dyx0;
        if (debug) 
          { Pr(Er, "  moved too far from initial guess\n");
            Pr(Er, "  shrinking by s = %22.16e", s);
          }
        for (k = 0; k < n; k++) { y[k] = x0[k] + s*(y[k] - x0[k]); }
      }
    /* Check motion of this iteration: */
    double stepMax = fmin(8*radius, rMax); /* Max move in this iteration. */
    double dyx = rn_dist(n, x, y);
    if (dyx > stepMax)
      { /* Moved too much, curb it: */
        double s = stepMax/dyx;
        if (debug) 
          { Pr(Er, "  moved too much\n");
            Pr(Er, "  reducing the step by s = %22.16e", s);
          }
        for (k = 0; k < n; k++) { y[k] = x[k] + s*(y[k] - x[k]); }
      }
  }

void sve_take_step
  ( int n,
    double x[],
    double *FxP,
    double y[],
    double Fy,
    int nv,
    double v[],
    int nf,
    double Fv[],
    double radius,
    double *dP,
    bool_t debug
  )
  {
    /* Grab the current function value: */
    double Fx = (*FxP);
    
    /* Find the minimum value among {Fv[0..nf-1]}: */
    double FMin = +INF; /* {FMin} is the minimum. */
    int iMin = -1, jMin = -1; /* {V(iMin,jMin)} is the minimum sample point. */
    int i, j;
    for (i = 0; i < nv; i++)
      { for (j = 0; j <= i; j++)
          { int ij = i*(i+1)/2 + j;
            if (Fv[ij] < FMin) { iMin = i; jMin = j; FMin = Fv[ij]; }
          }
      }
    assert((iMin < 0) == (jMin < 0));

    /* Choose the next center {x} and set the nominal step {d}: */
    double scale = radius/rmxn_regular_simplex_radius(n);
    double d;
    int k;
    if ((Fx < Fy) || (FMin < Fy))
      { /* Quadratic step failed: */
        if (debug) { Pr(Er, "    quadratic step failed - not minimum\n"); }
        if (Fx <= FMin)
          { /* The minimum is still {x}: */
            if (debug) { Pr(Er, "    minimum is still the simplex center\n"); }
            /* The step length is the center-to-edge distance: */
            d = 0.5*scale*rmxn_regular_simplex_edge(n);
          }
        else
          { /* The minimum is {V(iMin,jMin)}: */
            if (debug) { Pr(Er, "    minimum is V(%d,%d)\n", iMin, jMin); }
            /* The step length is the distance from center to {V(iMin,jMin)}: */
            if (iMin == jMin)
              { d = radius; }
            else
              { d = 0.5*scale*rmxn_regular_simplex_edge(n); }
            double *vi = &(v[iMin*n]);
            double *vj = &(v[jMin*n]);
            for (k = 0; k < n; k++) { x[k] = (vi[k] + vj[k])/2; }
            Fx = FMin;
          }
      }
    else
      { /* Quadratic step seems to have succeeded: */ 
        if (debug) { Pr(Er, "    quadratic step yielded a new minimum\n"); }
        /* The step length is the distance from the old center: */
        d = rn_dist(n, x, y);
        for (k = 0; k < n; k++) { x[k] = y[k]; }
        Fx = Fy;
      }
    /* Return {Fx,d}: */
    (*dP) = d;
    (*FxP) = Fx;
  }

void sve_print_probes(FILE *wr, int nv, int n, double v[], int nf, double Fv[])
  {
    int i, j;
    for (i = 0; i < nv; i++)
      { for (j = 0; j <= i; j++)
          { int ij = i*(i+1)/2 + j;
            fprintf(wr, "    %24.16e", Fv[ij]);
            fprintf(wr, "  ");
            double *vi = &(v[i*n]);
            double *vj = &(v[j*n]);
            int k;
            for (k = 0; k < n; k++) 
              { fprintf(wr, " %20.16f", (vi[k] + vj[k])/2); }
            fprintf(wr, "\n");
          }
      }
  }
