#ifndef sve_minn_H
#define sve_minn_H

/* Quadratic minimzation by the simplex vertex-edge method. */
/* Last edited on 2009-02-22 21:21:58 by stolfi */

/* SIMPLICES

  An /{n}-simplex/, is a list {V} of {n+1} points of {R^n}, for some
  {n >= 0}. Those points are the /corners/ of the simplex, and any two
  distinct corners define an /edge/. The simplex is /degenerate/ if
  some corner can be written as an affine combination of the other
  corner; and is /proper/ otherwise.
  
  In this interface, {V(i)} denotes corner number {i} of the simplex {V} 
  
  SIMPLEX REPRESENTATION
  
  In this interface, an {n}-simplex {V} in {R^n} is represented
  as an array {v} with has {(n+1)*n} elements, conceptually
  {n+1} rows and {n} columns; where row {i} contains the {n}
  coordinates of corner {i} of the simplex. 
  
  SIMPLEX NODES
  
  Let {V(i,j)} denote the midpoint of {V(i)} to {V(j)}. Note
  that {V(i,j)==V(j,i)} for all {i,j} in {0..n}, and also that
  {V(i,j)==V(i)==V(j)} when {i == j}.
  
  The main quadratic minimization routine ({sve_minn-step})
  requires the values of the goal function at all the corners
  and edge midpoints of some {n}-simplex {V}; that is, at 
  all points {V(i,j)} such that {0 <= j <= i <= n}.
  
  The number of such /nodes/ is {K(n) = (n+1)*(n+2)/2}. */

#include <bool.h>

void sve_minn_step(int n, double Fv[], double cm[]);
  /* Given the values {fv[0..K(n)-1]} of a quadratic function {F} at the 
    nodes of some {n}-simplex {V}, returns in {cm[0..n]} 
    the barycentric coordinates of the stationary point of {F} in the 
    affine subspace spanned by {V}.

    The procedure assumes that the array {fv} has {K(n)} elements, and
    that {Fv[i*(i+1)/2+j] == F(V(i,j))} for all {i,j} such that 
    {0 <= j <= i <= n}. */

typedef double sve_goal_t(int n, double x[]);
  /* The type of a procedure that can be provided as argument to
    {sve_sample_function} and {sve_optimize} below. It should compute
    some function of the point {x[0..n-1]}. */
    
void sve_sample_function(int n, sve_goal_t *F, double v[], double Fv[]);
  /* Evaluates the {n}-variate goal function {F} at the nodes of an
    {n}-simplex {V} in {R^n}, ad stores the values into
    {fv[0..K(n)-1]}, in the order expected by {sve_minn_step}.
    Assumes that {v} has {K(n)*n} elements and contains the 
    coordinates of the vertices of the simplex, stored by rows.
    
    More precisely, sets {fv[i*(i+1)/2+j]} to {F(V(i,j))} for all
    {i,j} such that {0 <= j <= i <= n}. */

typedef int sve_pred_t(int n, double x[], double Fx);
  /* The type of a procedure that can be provided as argument to
    {sve_minn_iterate} below. It should check the current solution
    {x[0..n-1]} and the corresponding goal function value {Fx}, and
    return 1 to stop the iteration, 0 to continue it. */
    
void sve_minn_iterate
  ( int n, 
    sve_goal_t *F, 
    sve_pred_t *OK,
    double x[],
    double rIni,
    double rMin, 
    double rMax,
    double dMax,
    int maxEvals,
    bool_t debug
  );
  /*  Tries to find a stationary point {x[0..n-1]} of the {n}-argument
    function {F}, by repeated calls to {sve_minn_step}.
    
    Upon entry, the vector {x[0..n-1]} should contain the initial
    guess. Upon exit, it will contain the final guess. The distance
    between the latter and the former is limited to {dMax}. At each
    iteration, the procedure chooses a random probe simplex centered
    on the current guess {x[0..n-1]}. The radius {r} of the simplex is
    dynamically adjusted, starting with {rIni} but staying within the
    range {[rMin _ rMax]}.
    
    The procedure looks for a stationary point of {F}, which may be a
    local minimum, a local maximum, or a saddle point. Thus, the value
    of {F} at the current guess {x} may increase or decrease during
    the search; especially if the function is neither concave nor
    convex, or has a significant non-quadratic behavior in the region
    searched. In that case, the final guess {x} may be neither the
    minimum nor the maximum of all sample points. Therefore, the
    client may want to add code to the goal function {F} that saves
    the absolute minimum or maximum found during the search, as
    appropriate.
    
    The iterations will stop when (A) the current guess {x} satisfies
    the predicate {OK(n,x,F(n,x))}; or (B) the distance between
    successive guesses, extrapolated to infinitely many iterations, is
    less than {rMin}; or (C) the function {F} has been evaluated
    {maxEvals} times or more. Note that each iteration performs at
    least {K(n) + 1} calls to {F}, so the budget may be
    overrun by {K(n)} calls.
    
    If {debug} is TRUE, the procedure prints various diagnostic messages. */


#endif
