/* See pst_imgsys.h 
**
** Created on 2005-10-01 by Jorge Stolfi, unicamp, <stolfi@dcc.unicamp.br>
** Based on the work of Rafael Saracchini, U.F.Fluminense.
** Last edited on 2010-05-04 03:48:06 by stolfi
** See the copyright and authorship notice at the end of this file. 
*/

#define _GNU_SOURCE
#include <math.h>
#include <stdlib.h>
#include <assert.h>
#include <stdio.h>

#include <pst_imgsys.h>
#include <pst_height_map.h>

#include <float_image.h>
#include <float_image_mscale.h>
#include <affirm.h>

double pst_imgsys_sol_change(double *Zold, double *Z, long int N);
  /* Returns the maximum of {abs(Zold[k]-Z[k])}, for {k = 0..N-1}. */

imgsys_t *pst_imgsys_new(int NX, int NY, long int N, long int *ix, int *col, int *row)
  {
    imgsys_t *S = (imgsys_t*)malloc(sizeof(imgsys_t));
    S->N = N;
    S->eq = (imgsys_equation_t*)malloc(sizeof(imgsys_equation_t)*N);
    S->ix = ix;
    S->col = col;
    S->row = row;
    return S;
  }

void pst_imgsys_free(imgsys_t *S)
  {
    free(S->eq);
    free(S->col);
    free(S->row);
    free(S->ix);
    free(S);
  }

void pst_imgsys_solve
  ( imgsys_t *S, 
    double Z[],
    long int maxIter, 
    double convTol,
    int para, 
    int szero,
    bool_t verbose,
    int indent,
    pst_imgsys_solution_report_proc_t *reportSol
  )
  {
    long int N = S->N;
    long int k;

    /* Previous solution: */
    double *Zold = (double*)malloc(sizeof(double)*N);
    
    long int iter;
    double change = +INF;  /* Max {Z} change in last iteration. */
    for (iter = 0; iter < maxIter; iter++)
      {
        /* Report the current solution if so requested: */
        if (reportSol != NULL) { reportSol(iter, change, FALSE, N, Z); }

        /* Another pass over all unknowns {Z[k]}: */
        for (k = 0; k < N; k++)
          { /* Save current solution in {Zold}: */
            Zold[k] = Z[k];
            /* Get equation {k}: */
            imgsys_equation_t *eqk = &(S->eq[k]);
            /* Compute {Z[k]} using equation {k}: */
            double sum = eqk->rhs; /* Right-hand side of equation. */
            double cf_k = 0.0; /* Coefficient of {Z[k]} in equation. */
            int t;
            for(t = 0; t < eqk->nt; t++)
              {
                /* Get hold of another unknown {Z[j] entering in equation {k}: */
                int j = eqk->ix[t];
                demand((j >= 0) && (j < N), "invalid variable index in system");
                double cf_j = eqk->cf[t];
                if (j == k) 
                  { /* This term uses {Z[k]}, store the coefficient: */
                    cf_k = cf_j;
                  }
                else if (cf_j != 0)
                  { /* The unknown {Z[j]} is distinct from {Z[k]}. */
                    /* Get the appropriate value (new or old) of {Z[j]}: */
                    double Zj = (para && (j < k) ? Zold[j] : Z[j]);
                    /* Subtract from the right-hand side: */
                    sum = sum - Zj * cf_j;
                  }
              }
            
            /* Require that {eqk} depends on {Z[k]}: */
            demand(cf_k != 0.0, "system's matrix has a zero in the diagonal"); 
            
            /* Solve the equation: */
            Z[k] = sum / cf_k;
          }
          
        if (szero)
          { /* Normalize for zero sum: */
            double sum = 0;
            for (k = 0; k < N; k++) { sum += Z[k]; }
            double avg = sum/N;
            for (k = 0; k < N; k++) { Z[k] -= avg; }
          }
        iter++;
        
        /* Check for apparent convergence: */
        change = pst_imgsys_sol_change(Zold, Z, N);
        if (verbose)
          { fprintf(stderr, "%*s  iteration %3ld change = %16.8f\n", indent, "", iter, change); }
        if (change <= convTol) { /* Converged: */ break; }
      }
    if (change <= convTol)
      { /* Failed to converge: */
        fprintf(stderr, "%*s** gave up after %6ld iterations, last change = %16.8f\n", indent, "", iter, change);
      }
    else
      { /* Converged: */
        fprintf(stderr, "%*sconverged after %6ld iterations,last change = %16.8f\n", indent, "", iter, change);
      }

    if (reportSol != NULL) { reportSol(iter, change, TRUE, N, Z); }

    free(Zold);
  }

double pst_imgsys_sol_change(double* Zold, double* Z, long int N)
  {
    int k;
    double max_change = 0;
    long int max_k = 0;
    for(k = 0; k< N; k++)
      { double dk = Z[k] - Zold[k];
        if(dk < 0) { dk = - dk; }
        if(dk > max_change) { max_k = k; max_change = dk; }
      }
    return max_change;
  }
    
void pst_imgsys_write(FILE *wr, imgsys_t *S)
  {
    long int k;
    for(k = 0; k < S->N; k++)
      {
        /* Get equation {k}: */
        imgsys_equation_t *eqk = &(S->eq[k]);
        
        /* Compute indices of pixel corresponding to the unknown {Z[k]}: */
        int xk = S->col[k]; int yk = S->row[k];
         
        /* Print indices of main pixel: */
        fprintf(wr, "eq[%d][%d]:", xk, yk);

        /* Print the equation coefficients: */
        int t;
        for(t = 0; t < eqk->nt; t++)
          { /* Get hold of another unknown {Z[j]} that occurs in equation {k}: */
            long int j = eqk->ix[t];
            /* Compute indices of the corresponding pixel. */
            int xj = S->col[j]; int yj = S->row[j];
            double cf_j = eqk->cf[t];
            if (t > 0) { fprintf(wr, " + "); }
            fprintf(wr, "%f*Z[%d][%d]", cf_j, xj, yj);
            demand((j >= 0) && (j < S->N), "invalid variable index in system");
          }
        /* Print the independent term: */
        fprintf(wr, " = %f", S->eq[k].rhs);
        fprintf(wr, "\n");
      }
    fflush(wr);
  }

void pst_imgsys_write_report(imgsys_t *S, char *filePrefix, int level, char *tag, int indent)
  {
    char *fileName = float_image_mscale_file_name(filePrefix, level, -1, tag, "sys");
    fprintf(stderr, "%*swriting %s ...", indent, "", fileName);
    FILE* wr = fopen(fileName, "wt");
    assert(wr != NULL); 
    pst_imgsys_write(wr, S);
    if (wr == stdout) { fflush(wr); } else { fclose(wr); }
    fprintf(stderr, "\n");
    free(fileName);
  }

/*
**
** Copyright (C) Jorge Stolfi, Unicamp.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty. Neither the author nor its employers are liable to
** any damages which may result from its use.
*/
