/* imgsys - linear system solving for image-related problems.
**
** Created on 2005-12-04 by Jorge Stolfi, unicamp, <stolfi@dcc.unicamp.br>
** Based on the work of Rafael Saracchini, U.F.Fluminense.
** Last edited on 2010-05-03 16:27:09 by stolfi
** See the copyright and authorship notice at the end of this file. */

#ifndef pst_imgsys_H
#define pst_imgsys_H

#include <float_image.h>
#include <pst_height_map.h>

#define MAXCOEFS 5

typedef struct imgsys_equation_t
  { int nt;                  /* {nt} is the number of terms in the equation. */
    long int ix[MAXCOEFS];   /* {ix[k]} is the index of some unknown in the equation. */
    double cf[MAXCOEFS];     /* Coefficient of that unknown. */
    double rhs;              /* Right-hand side of equation. */
  } imgsys_equation_t;
  /* An {imgsys_equation_t} record represents a linear equation 
    with at most {MAXCOEFS} nonzero terms, namely
      {SUM {cf[j]*h[ix[j]] : j = 0..nt-1} == rhs},
    where {h[0..N-1]} are all the unknowns. */

typedef struct imgsys_t
  { long int N;             /* Number of unknowns and equations. */
    imgsys_equation_t* eq;  /* The equations are {eq[0..N-1]}. */
    int *col;      /* Maps index of an unknown/equation to {x} index. */
    int *row;      /* Maps index of an unknown/equation to {y} index. */
    long int *ix;  /* Maps {x + y*NX_Z} to index of unknown/equation. */
  } imgsys_t;
  /* An {imgsys_t} represents {N} linear equations {eq[0..N-1]} on {N} unknowns {Z[0..N-1}. */

imgsys_t *pst_imgsys_new(int NX, int NY, long int N, long int *ix, int *col, int *row);
  /* Creates a new linear system with {N} equations on {N} unknowns
    for an image with {NX} columns and {NY} rows, using the given
    index mapping tables. Allocates the equations but does not
    initialize any coefficient or independent term. */

void pst_imgsys_free(imgsys_t *S);
  /* Deallocates all storage used by {S}, including the index tables. */

typedef void pst_imgsys_solution_report_proc_t(long int iter, int change, bool_t final, long int N, double Z[]);
  /* Type of a procedure that is used to report the progress of the solution. */

void pst_imgsys_solve
  ( imgsys_t *S, 
    double Z[],
    long int maxIter, 
    double convTol,
    int para, 
    int szero,
    bool_t verbose,
    int indent,
    pst_imgsys_solution_report_proc_t *reportSol
  );
  /* Solves system {S}, and stores the solution into the vector {Z}.  
     
     Uses an iterative method, and therefore assumes that the unknown
     {Z[i]} appears in some term of equation {S->eq[i]}, where {i =
     0..N-1} and {N = S->N}, and that its coefficient is ``large enough''. Upon
     entry, the {OZ} image must contain the starting guess. Executes
     at most {max_iter} iterations, but stops whenever two consecutive
     iterations do not change any variable by more than the tolerance
     {tol}.

     If {para} is TRUE, uses the ``parallel'' variant of the method
     (Jacobi), otherwise uses the sequential variant (Gauss-Seidel).
     If {szero == 1}, adjusts the solution so that it adds to zero,
     after each iteration.
     
     If {verbose} is true, prints information about the iterations to {stderr}.
     All messages will be indented by {indent} spaces.
     
     if {reportSol} is not null, will call
     {reportSol(iter,change,FALSE,N,Z)} before every iteration,
     {reportSol(iter,change,TRUE,N,Z)} after the last one; where
     {iter} is the number of iterations done so far, and {change} is
     the maximum absolute change in any {Z} value from the previous
     iteration. When {iter} is zero, {change} is irrelevant.
     If {maxIter} is zero, makes only one call with {iter=0,final=TRUE}.
     When {maxIter>0}, the first call has {iter=0,final=FALSE}
     and the last call has {iter>0,final=TRUE}*/
          
/* DEBUGGING */
    
typedef void pst_imgsys_report_proc_t(int level, imgsys_t *S); 
  /* Type of a client-given procedure that may be called
    by recursive integrators to report the system used at each scale.
    Uses {col} and {row} to map indices of unknowns to pixel indices. */   

/* I/O */
    
void pst_imgsys_write(FILE *wr, imgsys_t *S);
  /* Writes the system {S} to stream {wr}.  The indices of unknowns and 
    equations are mapped to column and row indices
    with the tables {col[0..S.N-1]} and {row[0..S.N-1]}.  */

void pst_imgsys_write_report(imgsys_t *S, char *filePrefix, int level, char *tag, int indent);
  /* Writes the system {S} to a file called
    "{filePrefix}-{level}-{tag}.sys". If {tag} is null or empty the
    "-{tag}" is omitted. Uses {pst_imgsys_write}. Diagnostic messages
    are indented by {indent} spaces. */

#endif

/*
**
** Copyright (C) Jorge Stolfi, Unicamp.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty. Neither the author nor its employers are liable to
** any damages which may result from its use.
*/
