/* See pst_slope_map.h */
/* Last edited on 2010-05-04 03:41:50 by stolfi */

#define _GNU_SOURCE
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>

#include <float_image.h>
#include <float_image_mscale.h>
#include <jsfile.h>
#include <r2.h>

#include <pst_basic.h>
#include <pst_imgsys.h>
#include <pst_slope_map.h>
#include <pst_weight_map.h>
#include <pst_height_map.h>

/* INTERNAL PROTOTYPES */
    
#define get_sample float_image_get_sample
#define set_sample float_image_set_sample
 
/* IMPLEMENTATIONS */

r2_t pst_slope_map_get_pixel(float_image_t *G, int x, int y)
  { demand(G->sz[0] == 2, "wrong slope map depth");
    r2_t grd;
    float *p = float_image_get_sample_address(G, 0, x, y);
    int axis;
    for (axis = 0; axis < 2; axis++)
      { grd.c[axis] = (*p); p += G->st[0]; }
    return grd;
  }

void pst_slope_map_set_pixel(float_image_t *G, int x, int y, r2_t *grd)
  { demand(G->sz[0] == 2, "wrong slope map depth");
    float *p = float_image_get_sample_address(G, 0, x, y);
    int axis;
    for (axis = 0; axis < 2; axis++)
      { (*p) = grd->c[axis]; p += G->st[0]; }
  }

#define pst_slope_map_size_cutoff 1
  /* Do not recurse on images smaller than this. */

float_image_t *pst_slope_map_to_depth_map_recursive
  ( float_image_t *IG, 
    float_image_t *IW, 
    int level,
    int avgWidth,
    bool_t harmAvgW,
    bool_t newStyle,
    long int maxIter,
    double convTol,
    bool_t verbose,
    long int reportIter,
    pst_slope_map_report_proc_t *reportData,
    pst_imgsys_report_proc_t *reportSys,
    pst_height_map_report_proc_t *reportHeights
  )
  {
    /* Get image size and check compatibility: */
    demand(IG->sz[0] == 2, "wrong {IG} channels");
    int NX_G = IG->sz[1];
    int NY_G = IG->sz[2];
    
    if (IW != NULL)
      { demand(IW->sz[0] == 1, "wrong {IW} channels");
        demand(IW->sz[1] == NX_G, "wrong {IW} cols");
        demand(IW->sz[2] == NY_G, "wrong {IW} rows");
      }

    float_image_t* HW = NULL;
    
    int indent = 2*level+2; /* Indentation for messages. */

    if (verbose)
      { fprintf(stderr, "%*sEntering level %d with G size %d�%d ...\n", indent, "", level, NX_G, NY_G); }

     if (reportData != NULL) { reportData(level, IG, IW); }

    /* Allocate the height map: */
    int NX_Z = NX_G+1;
    int NY_Z = NY_G+1;
    float_image_t* OZ;

    /* Decide whether to use multi-scale: */
    bool_t small = ((NX_G < pst_slope_map_size_cutoff) && (NY_G < pst_slope_map_size_cutoff));
    bool_t atomic = ((NX_G == 1) || (NY_G == 1));
    bool_t trivial = ((NX_G == 1) && (NY_G == 1));
    bool_t direct = (newStyle ? trivial : (small || atomic));
    
    /* Get the initial guess {OZ} for the heights: */
    if (direct)
      { /* Initialize {OZ} with zeros: */
        OZ = float_image_new(1, NX_Z, NY_Z);
        if (newStyle)
          { /* Solution for the 1x1 system: */
            /* Get slopes: */
            float dZdX = get_sample(IG, 0, 0, 0); 
            float dZdY = get_sample(IG, 1, 0, 0); 
            /* Assume an affine function that is 0 at center, with the given gradient: */
            double z10 = 0.5*(dZdX - dZdY);
            double z11 = 0.5*(dZdX + dZdY);
            set_sample(OZ, 0, 0, 0, -z11);
            set_sample(OZ, 0, 0, 1, -z10);
            set_sample(OZ, 0, 1, 0, +z10);
            set_sample(OZ, 0, 1, 1, +z11);
          }
        else
          { /* Set all to zero: */
            int x, y;
            for(y = 0; y < NY_Z; y++) 
              for(x = 0; x < NX_Z; x++) 
                {  set_sample(OZ, 0, x, y, 0.0); }
          }
      }
    else
      { /* Initialize {OZ} with the solution to a coarser problem. */
      
        /* Shrink slope maps and weight map by half: */
        if (verbose) { fprintf(stderr, "%*sShrinking slope maps ...\n", indent, ""); }
        float_image_t *SIG = pst_slope_map_shrink(IG, IW, avgWidth);
        
        if (verbose) { fprintf(stderr, "%*sShrinking weight map ...\n", indent, ""); }
        float_image_t *SIW = (IW == NULL ? NULL : pst_weight_map_shrink(IW, harmAvgW, avgWidth));
        HW = ( IW != NULL ? pst_weight_map_expand_height_weights(SIW) : NULL);
	if(HW != NULL){
	  char* nome = NULL;
	  asprintf(&nome,"t-deb-%d-HW.fni",level);
	  FILE* arq = open_write(nome,TRUE);
	  float_image_write(arq,HW);
	  fclose(arq);
	}
	
	//if (reportData != NULL) { reportData(level-1, IG, IW,HW); }
        
        /* Compute the half-scale height map: */
        float_image_t *SOZ = pst_slope_map_to_depth_map_recursive
          ( SIG, SIW, level+1, 
            avgWidth, harmAvgW, newStyle, maxIter, convTol, 
            verbose, reportIter,
            reportData, reportSys, reportHeights
          );
        
        /* Expand the computed height map to double size: */
	
	/*if( SIW != NULL){
	  if (verbose) { 
	    fprintf(stderr, "%*sExpanding weight map to %d�%d ...\n", indent, "", NX_Z, NY_Z); 
	    HW = pst_weight_map_expand_height_weights(SIW);
	  }
	}*/
	
        if (verbose) { fprintf(stderr, "%*sExpanding height map to %d�%d ...\n", indent, "", NX_Z, NY_Z); }
        int expOrder = 2; /* Bilinear for now. !!! Generalize? !!! */
        OZ = pst_height_map_expand(SOZ,HW, NX_Z, NY_Z, expOrder);
          
        /* Free the working storage: */
        float_image_free(SIG);
        if (SIW != NULL) { float_image_free(SIW); float_image_free(HW); }
        float_image_free(SOZ);
      }

    bool_t solve_sys = (! (newStyle && trivial));
//      if (reportData != NULL) { reportData(level, IG, IW,HW); }
     
    if (solve_sys)
      { /* Build the linear system: */
        long int NP_Z = NX_Z*NY_Z;
        if (verbose) { fprintf(stderr, "%*sBuilding linear system for %ld pixels ...\n", indent, "", NP_Z); }
        bool_t full = FALSE; /* Should be parameter. FALSE means exclude indeterminate pixels. */
        imgsys_t* S = pst_slope_map_build_integration_system(IG, IW, full);
        if (verbose) { fprintf(stderr, "%*sSystem has %ld equations and %ld unknowns\n", indent, "", S->N, S->N); }
        if (reportSys != NULL) { reportSys(level, S); }

        /* Solve the system for the corner heights: */
        if (verbose) { fprintf(stderr, "%*sSolving the system ...\n", indent, ""); }
        int para = 0; /* Should be parameter. 1 means parallel execution, 0 sequential. */
        int szero = 1; /* Should be parameter. 1 means adjust sum to zero, 0 let it float. */
        pst_slope_map_solve_system(S, OZ, maxIter, convTol, para, szero, verbose, level, reportIter, reportHeights);
        pst_imgsys_free(S);
      }
    else
      { /* The initial guess is the solution: */
        if (reportSys != NULL) { reportSys(level, NULL); }
        if (reportHeights != NULL) { reportHeights(level, 0, 0.0, TRUE, OZ); }
      }

    if (verbose)
      { fprintf
          ( stderr, "%*sLeaving level %d (OZ = %lld�%lld) ...\n", 
            indent, "", level, OZ->sz[1], OZ->sz[2]
          );
      }
    return OZ;
  }

void pst_slope_map_solve_system
  ( imgsys_t *S, 
    float_image_t *OZ, 
    long int maxIter, 
    double convTol, 
    int para, 
    int szero, 
    bool_t verbose, 
    int level, 
    long int reportIter, 
    pst_height_map_report_proc_t *reportHeights
  )
  {
    int indent = 2*level+2;
    
    auto void reportSol(long int iter, int change, bool_t final, long int N, double Z[]);
      /* A procedure that is called by the Gauss-Seidel solver at each iteration.
         When appropriate, it copies {Z} into the image {OZ} and calls {reportHeights}. 
         This happens when {final} is TRUE or when {reportIter != 0} and {iter}
         is a multiple of {reportIter}. */
    
    void reportSol(long int iter, int change, bool_t final, long int N, double Z[])
      { bool_t doit = final || ((reportHeights != NULL) && (reportIter != 0) && (iter % reportIter == 0));
        if (doit) 
          { pst_slope_map_copy_sol_vec_to_height_map(S, Z, OZ);
            if (reportHeights != NULL) { reportHeights(level, iter, change, final, OZ); }
          }
      }

    double *Z = (double*)malloc(sizeof(double)*S->N);
    pst_slope_map_copy_height_map_to_sol_vec(S, OZ, Z);
    pst_imgsys_solve(S, Z, maxIter, convTol, para, szero, verbose, indent, reportSol);
    /* {reportSol} must have copied the final solution into {OZ}. */
    free(Z);
  }

void pst_slope_map_copy_height_map_to_sol_vec(imgsys_t *S, float_image_t *IZ, double VZ[])
  {
    long int N = S->N;
    long int k;
    for(k = 0; k < N; k++)
      { int x = S->col[k];
        int y = S->row[k];
        VZ[k] = float_image_get_sample(IZ, 0, x, y);
      }
  }

void pst_slope_map_copy_sol_vec_to_height_map(imgsys_t *S, double VZ[], float_image_t *IZ)
  {
    int NX = IZ->sz[1];
    int NY = IZ->sz[2];
    int x, y;
    for(y = 0; y < NY; y++)
      { for(x = 0; x < NX; x++)
          { long int k = S->ix[x + NX*y];
            float_image_set_sample(IZ, 0, x, y, (k < 0 ? 0.0 : VZ[k]));
          }
      }
  }

float_image_t *pst_slope_map_shrink_old(float_image_t *IG, float_image_t *IW)
  { 
    int NC = IG->sz[0];
    int NXI = IG->sz[1]; int NXJ = (NXI+1)/2;
    int NYI = IG->sz[2]; int NYJ = (NYI+1)/2;
    
    if (IW != NULL)
      { assert(IW->sz[0] == 1);
        assert(IW->sz[1] == NXI);
        assert(IW->sz[2] == NYI);
      }

    float_image_t *JG = float_image_new(NC, NXJ, NYJ);
    int xJ, yJ;
    for(yJ = 0; yJ < NYJ; yJ++)
      { for(xJ = 0; xJ < NXJ; xJ++)
          { int xI = 2*xJ, yI = 2*yJ;
            double w00 = 1.0, w01 = 1.0, w10 = 1.0, w11 = 1.0; /* Weights of each sample: */
            if (IW == NULL)
              { w00 = get_sample(IW, 0, xI, yI);
                w10 = ((xI+1 < NXI) ? get_sample(IW, 0, xI+1, yI) : w00);
                w01 = ((yI+1 < NYI) ? get_sample(IW, 0, xI, yI+1) : w00);
                w11 = ((xI+1 < NXI) && (yI+1 < NYI) ? get_sample(IW, 0, xI+1, yI+1) : w00);
              }
            double wtot = w00 + w01 + w10 + w11;
            if (wtot == 0.0) { w00 = w01 = w10 = w11 = 0.25; wtot = 1.0; }
            int c;
            for (c = 0; c < NC; c++)
              { /* Samples in original image (with border padding): */
                double d00 = get_sample(IG, c, xI, yI);
                double d10 = ((xI+1 < NXI) ? get_sample(IG, c, xI+1, yI) : d00);
                double d01 = ((yI+1 < NYI) ? get_sample(IG, c, xI, yI+1) : d00);
                double d11 = ((xI+1 < NXI) && (yI+1 < NYI) ? get_sample(IG, c, xI+1, yI+1) : d00);
                float v = (w00*d00 + w01*d01 + w10*d10 + w11*d11)/wtot;
                /* All three coordinate {X,Y,Z} get scaled, so the derivatives don't: */
                /* Set sample in shrunken image: */
                set_sample(JG, c, xJ, yJ, v);
              }
          }
      }
    return JG;
  }

float_image_t *pst_slope_map_shrink(float_image_t *IG, float_image_t *IW, int avgWidth)
  { 
    int NX_JG = (IG->sz[1]+1)/2;
    int NY_JG = (IG->sz[2]+1)/2;
    int dxy = (avgWidth-1)/2;
    return float_image_mscale_shrink(IG, IW, NX_JG, NY_JG, dxy, dxy, avgWidth);
  }

imgsys_t* pst_slope_map_build_integration_system(float_image_t *G, float_image_t *W, bool_t full) 
  {
    /* Get/check the sizes of the slope maps: */
    int NC = G->sz[0]; assert(NC == 2);
    int NX_G = G->sz[1];
    int NY_G = G->sz[2];
    
    if (W != NULL)
      { assert(W->sz[0] == 1);
        assert(W->sz[1] == NX_G);
        assert(W->sz[2] == NY_G);
      }
    
    /* Check the size of the system: */
    int NX_Z = NX_G+1;
    int NY_Z = NY_G+1;
    int NXY_Z = NX_Z*NY_Z; /* Number of unknowns (heights). */
    
    /* Conceptually, we define two quadratic mismatch terms
      {qx[x,y](h)} and {qy[x,y](h)} for each edge in the pixel grid.
      
      The term {qx[x,y](h)} is the square of the difference between
      the mean slope along the edge from {(x,y)} to {(x+1,y)}, as
      estimated from the slope map {G}, and the mean slope computed
      by numerical difference of the (unknown) heights at the two
      endpoints of that edge. This terms exists for all {x} in {0..NX_G-1}
      and all {y} in {0..NY_G}.
      
        { qx[x,y](h) = (dx[x,y] - (- h[k00] + h[k10]))^2 }

      where {h[k00]} and {h[k10]} are the {h} variables
      corresponding to heights {z[x,y]} and {z[x+1,y]}, respectively;
      and {dx} is the interpolated slope
      
        { dx[x,y] = (wpm*dxpm + wpp*dxpp)/wx[x,y] }

      where {wpm == IW[0,x,y-1]}, {wpp == IW[0,x,y]}, {wx[x,y] == wpm+wpp},
      {dxpm == G[0,x,y-1]}, and {dxpp == G[0,x,y]}.
      
      The term {qy[x,y](h)} is similarly defined for edges parallel to
      the Y axis. It exists for all {x} in {0..NX_G} and all {y} in
      {0..NY_G-1}.
      
        { qy[x,y](h) = (dy[x,y] - (- h[k00] + h[k01]))^2 }
      
        { dy[x,y] = (wmp*dxmp + wpp*dxpp)/wy[x,y] }

      where {wmp == IW[0,x,y-1]}, {wy[x,y] == wmp+wpp},
      {dymp == G[1,x-1,y]}, and {dypp == G[1,x,y]}.

      We want to minimize the sum {Q(h)} of all terms {qx[x,y](h)} and 
      {qy[x,y](h)}, each weighted by the corresponding edge weight
      {wx[x,y]} or {wy[x,y]}. 
      
      Note that if the slope map {G} is zero, we can anihilate all
      those terms (and therefore Q) by setting all heights equal to 1.
      Therefore, if the system is to be solved iteratively, one must
      take care to exclude this solution.
       
      Minimizing {Q} is equivalent to finding the {h} vector that
      anihilates the gradient {G} of {Q}. The gradient of {Q} is an
      affine function of {h}, namely {G(h) = A h - b}. Each term
      {wx[x,y]*qx[x,y](h)} contributes to two components of the
      gradient, namely those with indices {k00,k10}. Similarly, each
      term {wy[x,y]*qy[x,y]} contributes to two components, with
      indices {k00,k01}. Thus, to build the system, we add those
      contributions to the appropriate elements of {A} and {b}.
    */
    
    /* Make sure that we can build the system: */
    assert(MAXCOEFS >= 5);
    
    /* Count the elements that are non-null and assemble the {h}-index table {ix[0..NXY_Z-1]}: */
    long int *ix = (long int *)notnull(malloc(NXY_Z*sizeof(long int)), "no mem");
    long int N;
    if (full || (W == NULL))
      { N = NXY_Z;
        long int xy;
        for (xy = 0; xy < NXY_Z; xy++) { ix[xy] = xy; }
      }
    else
      { N = 0;
        int x, y;
        for(y = 0; y <= NY_G; y++)
          { for(x = 0; x <= NX_G; x++)
              { /* Compute the position in the {ix} array: */
                long int xy = x + y*NX_Z;
                
                /* Get the slopes and relative weights of the pixels incident to {(x,y)}: */
                bool_t determined = 
                  ((x > 0)    && (y > 0)    && get_sample(W, 0, x-1, y-1) > 0) ||
                  ((x > 0)    && (y < NY_G) && get_sample(W, 0, x-1, y) > 0)   ||
                  ((x < NX_G) && (y > 0)    && get_sample(W, 0, x, y-1) > 0)   ||
                  ((x < NX_G) && (y < NY_G) && get_sample(W, 0, x, y) > 0);
                  
                if (determined)
                  { /* Some neighboring slope has positive weight. */
                    /* Add the element {Z[x,y]} to the system: */
                    ix[xy] = N;
                    N++;
                  }
                else
                  { /* All neighboring slopes have weight 0. */
                    /* Exclude the element {Z[x,y]} from the system: */
                    ix[xy] = -1;
                  }
              }
          }
      }
    assert(N <= NXY_Z);

    /* Build the inverse tables {col[0..N-1],row[0..N-1]}: */
    int *col = (int *)notnull(malloc(N*sizeof(int)), "no mem");
    int *row = (int *)notnull(malloc(N*sizeof(int)), "no mem");
    { long int xy;
      for (xy = 0; xy < NXY_Z; xy++) 
        { long int k = ix[xy];
          if (k >= 0) { col[k] = xy%NX_Z; row[k] = xy/NX_Z; }
        }
    }

    /* Allocate the system's equations: */
    imgsys_t *S = pst_imgsys_new(NX_Z, NY_Z, N, ix, col, row);

    auto long int ind_h(int x, int y); 
      /* This function computes the index into the solution vector {h}
        corresponding to vertex {[x,y]} of the images. 
        Returns -1 if {Z[x,y]} is not in the system. */
    
    long int ind_h(int x, int y) 
      { if ((x < 0) || (x >= NX_Z)) { return -1; }
        if ((y < 0) || (y >= NY_Z)) { return -1; }
        return S->ix[x + y*NX_Z];
      }
    
    /* Now build the equations for the variables in the system: */
    long int k;
    for(k = 0; k < N; k++)
      { /* Get the indices of the pixel corresponding to the unknown {h[k]}: */
        int x = col[k];
        int y = row[k];
        
        /* Get the slopes and relative weights of the pixels incident to {(x,y)}: */
        double dxmm = 0.0, dxmp = 0.0, dxpm = 0.0, dxpp = 0.0;
        double dymm = 0.0, dymp = 0.0, dypm = 0.0, dypp = 0.0;
        double wmm = 0.0, wmp = 0.0, wpm = 0.0, wpp = 0.0;

        if ((x > 0) && (y > 0))
          { dxmm = get_sample(G, 0, x-1, y-1);
            dymm = get_sample(G, 1, x-1, y-1);
            wmm = (W == NULL ? 1.0 : get_sample(W, 0, x-1, y-1));
            assert(wmm >= 0.0);
          }

        if ((x > 0) && (y < NY_G))
          { dxmp = get_sample(G, 0, x-1, y);
            dymp = get_sample(G, 1, x-1, y);
            wmp = (W == NULL ? 1.0 : get_sample(W, 0, x-1, y));
            assert(wmp >= 0.0);
          }

        if ((x < NX_G) && (y > 0))
          { dxpm = get_sample(G, 0, x, y-1);
            dypm = get_sample(G, 1, x, y-1);
            wpm = (W == NULL ? 1.0 : get_sample(W, 0, x, y-1));
            assert(wpm >= 0.0);
          }

        if ((x < NX_G) && (y < NY_G))
          { dxpp = get_sample(G, 0, x, y);
            dypp = get_sample(G, 1, x, y);
            wpp = (W == NULL ? 1.0 : get_sample(W, 0, x, y));
            assert(wpp >= 0.0);
          }

        assert(! isnan(wmm));
        assert(! isnan(wmp));
        assert(! isnan(wpm));
        assert(! isnan(wpp));

        /* Get the weights of the edges incident to {(x,y)}: */
        double wxm = wmm + wmp;
        double wxp = wpm + wpp;
        double wym = wmm + wpm;
        double wyp = wmp + wpp; 

        /* Compute the edge slopes times the edge weights: */
        double wdxm = 0.0, wdxp = 0.0, wdym = 0.0, wdyp = 0.0;

        double wtot = wxm + wxp + wym + wyp;
        if (wtot == 0.0)
          { /* This can only happen if we asked for the full system: */
            assert(full);
            /* Height {z[x,y]} is loose. Make it be the average of its neighbors: */
            wxm = (x <= 0  ? 0.0 : 1.0);
            wxp = (x >= NX_G ? 0.0 : 1.0);
            wym = (y <= 0  ? 0.0 : 1.0);
            wyp = (y >= NY_G ? 0.0 : 1.0);
            wtot = wxm + wxp + wym + wyp;
            /* Leave {wdxm,wdxp,wdym,wdyp} all zero. */
          }
        else
          { /* Compute {wdxm,wdxp,wdym,wdyp} from given slopes: */
            wdxm = wmm*dxmm + wmp*dxmp;
            wdxp = wpm*dxpm + wpp*dxpp;
            wdym = wmm*dymm + wpm*dypm;
            wdyp = wmp*dymp + wpp*dypp;
          }
        assert(wtot > 0.0);
        assert(! isnan(wdxm));
        assert(! isnan(wdxp));
        assert(! isnan(wdym));
        assert(! isnan(wdyp));

        /* Get the index {k} into {h} of {IZ[x,y]} and its cardinal neighbors: */
        long int kxm = ind_h(x-1, y); 
        long int kxp = ind_h(x+1, y); 
        long int kym = ind_h(x, y-1); 
        long int kyp = ind_h(x, y+1); 

        /* Assemble the equation for the height {Z[x,y]}: */
        imgsys_equation_t *eqk = &(S->eq[k]);
        int nt = 0; /* Number of terms in equation. */
        eqk->rhs = 0.0;
        eqk->ix[nt] = k;  eqk->cf[nt] = 1.00; nt++;
        if ((wxm != 0.0) && (kxm >= 0))
          { eqk->ix[nt] = kxm; eqk->cf[nt] = -wxm/wtot; eqk->rhs += +wdxm/wtot; nt++; }
        if ((wxp != 0.0) && (kxp >= 0))
          { eqk->ix[nt] = kxp; eqk->cf[nt] = -wxp/wtot; eqk->rhs += -wdxp/wtot; nt++; }
        if ((wym != 0.0) && (kym >= 0))
          { eqk->ix[nt] = kym; eqk->cf[nt] = -wym/wtot; eqk->rhs += +wdym/wtot; nt++; }
        if ((wyp != 0.0) && (kyp >= 0))
          { eqk->ix[nt] = kyp; eqk->cf[nt] = -wyp/wtot; eqk->rhs += -wdyp/wtot; nt++; }
        eqk->nt = nt;
      }
      
    return S;
  }
