#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pnm.h>
#include <assert.h>
#include <math.h>

struct Imagem{
  int x, y, max;
  pixel** matriz;
};

/****************************** Funções internas ***********/
void pnm_read_header(FILE* f, int* nx, int* ny,int* max, char* type);
void limpalinha(char* s,int tam);

/******************************Funções de Interface da Imagem ***********/
int get_width(Imagem* imagem)
{
  return imagem->x;
}

int get_height(Imagem* imagem)
{
  return imagem->y;
}

unsigned short int get_maxval(Imagem* imagem)
{
  return imagem->max;
}

pixel get_pixel(Imagem* imagem,int x, int y)
{
  if (x < 0) { x = 0; } else if (x >= imagem->x) { x = imagem->x - 1; } 
  if (y < 0) { y = 0; } else if (y >= imagem->y) { y = imagem->y - 1; } 
  return imagem->matriz[x][y];
}

void set_pixel(Imagem* imagem,int x, int y,pixel p)
{
  if ((x < 0) || (x >= imagem->x)) { return; }
  if ((y < 0) || (y >= imagem->y)) { return; }
  imagem->matriz[x][y] = p;
}

unsigned short int get_sample(Imagem* imagem, int x, int y, int canal)
{
  if (x < 0) { x = 0; } else if (x >= imagem->x) { x = imagem->x - 1; } 
  if (y < 0) { y = 0; } else if (y >= imagem->y) { y = imagem->y - 1; } 
  assert((canal >= 0) && (canal < 3)); 
  return imagem->matriz[x][y].canal[canal];
}

void set_sample(Imagem* imagem, int x, int y, int canal, unsigned short int grau)
{
  if ((x < 0) || (x >= imagem->x)) { return; }
  if ((y < 0) || (y >= imagem->y)) { return; }
  assert((canal >= 0) && (canal < 3)); 
  imagem->matriz[x][y].canal[canal] = grau;
}
/************************************************************************/

void limpalinha(char* s,int tam)
{
  int i;
  for(i = 0;i<tam;i++){
    s[i] = '\0';
  }
  return;
}


Imagem*  image_alloc(int x, int y, int maxval)
{
  assert(maxval > 0);
  assert(maxval <= PNM_MAX_MAXVAL);
  Imagem* novo = (Imagem*)malloc(sizeof(Imagem));
  int ind;
  novo->x = x;
  novo->y = y;
  novo->max = maxval;
  novo->matriz = (pixel**)malloc(sizeof(pixel*)*(x+1));
  assert(novo->matriz != NULL);
  for(ind = 0; ind < x; ind++) {
    novo->matriz[ind] = (pixel*)malloc(sizeof(pixel)*(y+1));
    assert(novo->matriz[ind] != NULL);
  }
  return novo;
}



void pnm_read_header(FILE* f, int* nx, int* ny, int* max, char* type)
{
  char line[256];
  int teste;
  char* testgets;
  testgets = fgets(line,255,f);
  sscanf(line,"%s",type);
  teste =1;

  while(teste) {
    //loop para pular os coment?ios...
    testgets = fgets(line,255,f);
    if(testgets == NULL) teste = 0;
    //fprintf(stderr, "Linha %s\n",line);
    if(line[0] != '#') {
      sscanf(line,"%d %d",nx,ny);
      teste = 0;
    }
  }
  fscanf(f,"%d",max);
  assert(*max > 0);
  assert(*max <= PNM_MAX_MAXVAL);
  //remove o ultimo /n que nao nos interessa !
  fgetc(f);
}




void pnm_image_read(char *nome, Imagem **novo)
{
  FILE *fp = fopen(nome,"rb");
  if (fp != NULL)
    {
      char type[10]; 
      int  x,y,nx,ny,Imax; 
	
      pnm_read_header(fp, &nx, &ny, &Imax, type);
      if ((strcmp(type,"P6")==0) || (strcmp(type,"P3")==0) ) {
	*novo =  image_alloc(nx, ny, Imax);
	if(strcmp(type,"P6") == 0) {
	  assert(Imax <= PNM_MAX_MAXVAL);
	  for (y=0; y < ny; y++){
	    for (x=0; x < nx; x++){
	      (*novo)->matriz[x][y].canal[0] = (int)fgetc(fp);
	      (*novo)->matriz[x][y].canal[1] = (int)fgetc(fp);
	      (*novo)->matriz[x][y].canal[2] = (int)fgetc(fp);
	    }
	  }
	} else {
	  for (y=0; y < ny; y++){
	    for (x=0; x < nx; x++){
	      int r,g,b;
	      fscanf(fp,"%d %d %d",&r,&g,&b);
	      (*novo)->matriz[x][y].canal[0] = r;
	      (*novo)->matriz[x][y].canal[1] = g;
	      (*novo)->matriz[x][y].canal[2] = b;
	    }
	  }
	}
      } else {
	fprintf(stderr, "ERRO - cabecalho invalido da imagem - %s\n",type);
      }
      fclose(fp);
    } else {
      fprintf(stderr, "ERRO - nao pode abrir arquivo - %s\n",nome);
    }
}

Imagem** pnm_image_list_read(char** nome, int n)
{
	Imagem** imagem;
	int luz;
	imagem = (Imagem**)malloc(sizeof(Imagem*)*n);
	for(luz = 0; luz<n; luz++){
		fprintf(stderr,  "Iluminação: %d\n",luz);
		fprintf(stderr,  "Nome <%s>\n",nome[luz]);
		pnm_image_read(nome[luz],&(imagem[luz]));
      		if(imagem[luz] == NULL){
			fprintf(stderr,  "ERRO ao abrir imagem !!!\n");
			return NULL;
		}
      	}
	return imagem;
}



void pnm_image_write(char *nome, Imagem *imagem)
{
  FILE *fp=NULL;
  int  x,y; 

  fp = fopen(nome,"wb");
  if (fp != NULL) {
    fprintf(fp,"P6\n");
    fprintf(fp,"%d %d\n",imagem->x,imagem->y);
    fprintf(fp,"%d\n", imagem->max);
    for (y=0; y < imagem->y; y++){
      for (x=0; x < imagem->x; x++){
	fputc((unsigned char)imagem->matriz[x][y].canal[0],fp);
	fputc((unsigned char)imagem->matriz[x][y].canal[1],fp);
	fputc((unsigned char)imagem->matriz[x][y].canal[2],fp);
      }
    }
    fclose(fp);
  } else {
    fprintf(stderr, "ERRO - nao pode abrir arquivo - %s\n",nome);
  }
}

void image_free(Imagem** imagem)
{
  int i;
  for (i = 0; i < (*imagem)->x; i++) {
    free((*imagem)->matriz[i]);
  }
  free(*imagem);
  *imagem = NULL;

}

double intensity_from_sample(short int grau, short int maxval, double gamma)
{
  double v = ((double)grau)/((double) maxval);
  if ((v > 0.0) && (v < 1.0) && (gamma != 1.0)) { v = pow(v, gamma); }
  return v;
}

short int sample_from_intensity(double v, short int maxval, double gamma)
{
  if ((v > 0.0) && (v < 1.0) && (gamma != 1.0)) { v = pow(v, 1.0/gamma); }
  if (v < 0.0) { v = 0.0; }
  if (v > 1.0) { v = 1.0; }
  int grau = (int)floor(v*maxval + 0.5);
  return grau;
}

double get_intensity(int x, int y, Imagem* imagem, int canal, double gamma)
{
  int nx = get_width(imagem);
  int ny = get_height(imagem);
  if (x < 0) { x = 0; } else if (x >= nx) { x = nx - 1; } 
  if (y < 0) { y = 0; } else if (y >= ny) { y = ny - 1; }
  short int grau = get_sample(imagem, x, y, canal);
  return intensity_from_sample(grau, get_maxval(imagem), gamma);
}

double Catmull_Rom_spline(double t,double p0,double p1,double p2,double p3)
{
  double a,b,c,d;
  double resp;
	
  a = -p0 + (3.0*p1) - (3.0*p2) + (p3);
  b = (2.0*p0) - (5.0*p1) + (4.0*p2) -p3;
  c = -p0 +p2;
  d = 2.0*p1;
  resp = (a*t*t*t)  + (b*t*t) + (c*t) +d;
  //fprintf(stderr, "%f\n",resp);
  return resp/2.0;
}

double image_interpolate_bicubic(double x, double y, Imagem* imagem, Imagem *mascara, int canal, double gamma)
{
  /* Map coords to pixel indices: */
  x = x - 0.5;
  y = y - 0.5;

  int ix = (int)floor(x);
  int iy = (int)floor(y);
  double tx = x - ix;
  double ty = y - iy;

  double v[4];
  short int w[4];
  int jx;
  for(jx = ix-1; jx < (ix+3); jx++){
    int ind = jx - ix +1;
    /* Obtém os quatro valores vizinhos na coluna {x}: */
    double v0 = get_intensity(jx, iy - 1, imagem, canal, gamma);
    double v1 = get_intensity(jx, iy + 0, imagem, canal, gamma);
    double v2 = get_intensity(jx, iy + 1, imagem, canal, gamma);
    double v3 = get_intensity(jx, iy + 2, imagem, canal, gamma);
    /* Obtém os pesos correspondentes: */
    short int w0 = (mascara == NULL ? 1 : get_sample(mascara, jx, iy - 1, canal));
    short int w1 = (mascara == NULL ? 1 : get_sample(mascara, jx, iy + 0, canal));
    short int w2 = (mascara == NULL ? 1 : get_sample(mascara, jx, iy + 1, canal));
    short int w3 = (mascara == NULL ? 1 : get_sample(mascara, jx, iy + 2, canal));
    /* Interpola na vertical: */
    /* O ponto está entre {v1} e {v2}. */
    w[ind] = 1;
    if ((w0 != 0) && (w1 != 0) && (w2 != 0) && (w3 != 0)) {
      /* Interpolação cubica: */
      v[ind] = Catmull_Rom_spline(ty,v0,v1,v2,v3);
    } else if ((w1 != 0) && (w2 != 0)) {
      /* Interpolação linear: */
      v[ind] = (1 - ty)*v1 + ty*v2;
    } else if (w1 != 0) {
      /* Pega pixel: */
      v[ind] = v1; 
    } else if (w2 != 0) {
      /* Pega pixel: */
      v[ind] = v2;
    } else {
      /* Nâo dá para interpolar: */
      v[ind] = 0.5;
      w[ind] = 0;
    }
  }
  /* Interpola na horizontal: */
  /* O ponto está entre {v[1]} e {v[2]}. */
  double vt = 1.0;
  short int wt = 1;
  if ((w[0] != 0) && (w[1] != 0) && (w[2] != 0) && (w[3] != 0)) {
    /* Interpolação cubica: */
    vt = Catmull_Rom_spline(tx, v[0],v[1],v[2],v[3]);
  } else if ((w[1] != 0) && (w[2] != 0)) {
    /* Interpolação linear: */
    vt = (1 - tx)*v[1] + tx*v[2];
  } else if (w[1] != 0) {
    /* Pega pixel: */
    vt = v[1]; 
  } else if (w[2] != 0) {
    /* Pega pixel: */
    vt = v[2];
  } else {
    /* Nâo dá para interpolar: */
    vt = 0.5;
    wt = 0;
  }
  return vt;
}

double image_interpolate_bilinear(double x, double y, Imagem* imagem, int canal, double gamma)
{
  int nx = get_width(imagem);
  int ny = get_height(imagem);
  /* Map coords to pixel indices: */
  x = x - 0.5;
  y = y - 0.5;
  /* Bilinear interpolation: */
  int px = floor(x);
  double fx = x - px;
  double wx[2] = {1-fx,fx};
  int py = floor(y);
  double fy = y - py;
  double wy[2] = {1-fy,fy};
  double svw = 0;
  double sw = 1e-20;
  int dx,dy;
  for(dx = 0; dx <=1; dx++){
    for(dy = 0; dy <= 1; dy++){
      int kx = px + dx;
      int ky = py + dy;
      if((kx >= 0) && (kx < nx) && (ky >= 0) && (ky < ny)){
	unsigned short int v = get_sample(imagem,kx,ky,canal);
	double w = wx[dx]*wy[dy];
	sw = sw + w;
	svw = svw + v*w;
      }
    }
  }
  return svw/sw;
}

