#include <lighting_models.h>
#include <assert.h>
#include <tabela.h>
#include <rn.h>

/*Generic model type*/

ls_model_t* create_ls_lighting_model(lighting_type_t type){
  
  ls_model_t* lm = (ls_model_t*) malloc(sizeof(ls_model_t));
  //lm->ls_data = NULL;
  lm->type = type;
  lm->num_weights = 0;
  lm->weights = NULL;
  lm->wpos  = NULL;
  if(type == POINTLIKE_MODEL){
    lm->phi = pointlike_phi;
    lm->get_num_components = pointlike_get_number_components;
    lm->retrieve_components = pointlike_retrieve_components;
    lm->update_weights = pointlike_update_weights;
    lm->validate_results= NoValidation;
    lm->evaluate = pointlike_shading;
    lm->write_param = pointlike_write_params;
    lm->copy_data =  pointlike_copy_lighting_data;
    lm->compare = pointlike_compare_lightings;
    lm->release_data = pointlike_release_lighting_data;
    lm->compute_pos_weights = pointlike_compute_pos_weights;
    lm->read_param = pointlike_read_params;
  }
  else if(type == HARMONIC_MODEL){
    lm->phi = harmonic_phi;
    lm->get_num_components = harmonic_get_number_components;
    lm->retrieve_components = harmonic_retrieve_components;
    lm->update_weights = harmonic_update_weights;
    lm->validate_results= NoValidation;
    lm->evaluate = harmonic_shading;
    lm->write_param = harmonic_write_params;
    lm->copy_data =  harmonic_copy_lighting_data;
    lm->compare = harmonic_compare_lightings;
    lm->release_data = harmonic_release_lighting_data;
    lm->compute_pos_weights = harmonic_compute_pos_weights;
    lm->read_param = harmonic_read_params;
  }else if(type == RADIALBASIS_MODEL){
    lm->phi = radialbasis_phi;
    lm->get_num_components = radialbasis_get_number_components;
    lm->retrieve_components = radialbasis_retrieve_components;
    lm->update_weights = radialbasis_update_weights;
    lm->validate_results= NoValidation;
    lm->evaluate = radialbasis_shading;
    lm->write_param = radialbasis_write_params;
    lm->copy_data =  radialbasis_copy_lighting_data;
    lm->compare = radialbasis_compare_lightings;
    lm->release_data = radialbasis_release_lighting_data;
    lm->compute_pos_weights = radialbasis_compute_pos_weights;
    lm->read_param = radialbasis_read_params;
  }else if( type == GLOSSYLIKE_MODEL){
    lm->phi = glossylike_phi;
    lm->get_num_components = glossylike_get_number_components;
    lm->retrieve_components = glossylike_retrieve_components;
    lm->update_weights = glossylike_update_weights;
    lm->validate_results= NoValidation;
    lm->evaluate = glossylike_shading;
    lm->write_param = glossylike_write_params;
    lm->copy_data =  glossylike_copy_lighting_data;
    lm->compare = glossylike_compare_lightings;
    lm->release_data = glossylike_release_lighting_data;
    lm->compute_pos_weights = glossylike_compute_pos_weights;
    lm->read_param = glossylike_read_params;
  }else if(type == HARMONICSG_MODEL){
    lm->phi = harmonicSG_phi;
    lm->get_num_components = harmonicSG_get_number_components;
    lm->retrieve_components = harmonicSG_retrieve_components;
    lm->update_weights = harmonicSG_update_weights;
    lm->validate_results= NoValidation;
    lm->evaluate = harmonicSG_shading;
    lm->write_param = harmonicSG_write_params;
    lm->copy_data =  harmonicSG_copy_lighting_data;
    lm->compare = harmonicSG_compare_lightings;
    lm->release_data = harmonicSG_release_lighting_data;
    lm->compute_pos_weights = harmonicSG_compute_pos_weights;
    lm->read_param = harmonicSG_read_params;
  }else{
    free(lm);
    return NULL;
  }
  return lm;
}

/*Lambert Model*/

double pointlike_phi(int r, double* x,void* l_data){
  
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  
  pointlike_lighting_t* ll = (pointlike_lighting_t*) l_data;
  if(r < 3){ /* Elements for direct lighting by point source: */
    double s = r3_dot(&(ll->f), &normal);
    return (s <= 0 ? 0 : normal.c[r]);
  }else if (r == 3){ /* Element for indirect lighing: */
    return (ll->BPL ? 0.5*(1 - normal.c[2]) : 1);
  }else{ /* No such element: */
    assert(FALSE);
  }
  
}

int pointlike_get_number_components(void* l_data){
  return 4;
}

pointlike_lighting_t* pointlike_init_components(double albedo,bool_t BPL,r3_t view_dir){
  
  pointlike_lighting_t* ll = (pointlike_lighting_t*) malloc(sizeof(pointlike_lighting_t));
  ll->view_dir = view_dir;
  ll->f =  (r3_t){{ 0,0,1}};
  ll->Ed = 0;
  ll->Eb = 0;
  ll->Ei = 0;
  ll->BPL = BPL;
  ll->albedo = albedo;
  return ll;
  
}

void pointlike_retrieve_components(double* C, int n,void* l_data){
  
  pointlike_lighting_t* ll = (pointlike_lighting_t*) l_data;
  ll->f.c[0] = C[0];
  ll->f.c[1] = C[1];
  ll->f.c[2] = C[2];
  ll->Ed = r3_dir(&(ll->f),&(ll->f))/ll->albedo;
  if(ll->BPL){
    ll->Eb = C[3]/ll->albedo;
    ll->Ei = 0;
  }else{
    ll->Eb = 0;
    ll->Ei = C[3]/ll->albedo;
  }
  
}

void pointlike_update_weights(void* l_data,double** X, double* F,double* weights, int n,double* sigma){
  int i;
  
  pointlike_lighting_t* ll = (pointlike_lighting_t*) l_data;
  for(i = 0; i < n; i++){
    r3_t normal = (r3_t) {{X[i][0],X[i][1],X[i][2]}};
    double s = r3_dot(&(ll->f), &normal);
    if(s > 0){
      weights[i] = 1.0;
    }else{
      weights[i] = 0.0;
    }
  }
  
}


double pointlike_shading(double* x,void* l_data){
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  
  pointlike_lighting_t* ll = (pointlike_lighting_t*) l_data;
  double vald = fmax(0,ll->Ed*r3_dot(&(ll->f),&normal));
  double valb = ll->Eb*(1-(normal.c[2]))/2.0;
  double vali = ll->Ei;
  double radiation = ll->albedo*(vald+valb+vali);
  return radiation;
  
}

void pointlike_write_params(FILE* arq,void* l_data){
  pointlike_lighting_t* ll = (pointlike_lighting_t*) l_data;
  fprintf(arq, "  %8.6lf", ll->albedo);
  r3_gen_print(arq, &(ll->f), "%9.6lf", "  ", " ", "");
  fprintf(arq, "  %8.6lf %8.6lf %8.6lf", ll->Ed, ll->Eb, ll->Ei);
  fprintf(arq, "\n");
}


void* pointlike_read_params(FILE* arq){
  pointlike_lighting_t* ll = (pointlike_lighting_t*) malloc(sizeof(pointlike_lighting_t));
  fscanf(arq, "%lf",&(ll->albedo));
  //r3_gen_print(arq, &(ll->f), "%9.6lf", "  ", " ", "");
  fscanf(arq,"%lf %lf %lf",&(ll->f.c[0]),&(ll->f.c[1]),&(ll->f.c[2]));
  fscanf(arq, "%lf %lf %lf", &(ll->Ed), &(ll->Eb), &(ll->Ei));
  if(ll->Eb == 0.0){
    ll->BPL = FALSE;
  }else{
    ll->BPL = TRUE;
  }
  return ll;
}


void* pointlike_copy_lighting_data(void* l_data){
  pointlike_lighting_t* ll = (pointlike_lighting_t*) l_data;
  pointlike_lighting_t* npl = (pointlike_lighting_t*) malloc(sizeof(pointlike_lighting_t));
  npl->f = ll->f;
  npl->albedo = ll->albedo;
  npl->Ei = ll->Ei;
  npl->Ed = ll->Ed;
  npl->Eb = ll->Eb;
  npl->BPL = ll->BPL;
  return npl;
}

double pointlike_compare_lightings(void* l_data1,void* l_data2){
  pointlike_lighting_t* ll1 = (pointlike_lighting_t*) l_data1;
  pointlike_lighting_t* ll2 = (pointlike_lighting_t*) l_data2;
  r3_t f = ll1->f;
  r3_t f_new = ll2->f;
  double diff = r3_dist(&f, &f_new);
  return diff;
}

void pointlike_release_lighting_data(void* l_data){
  free(l_data);
}


double pointlike_compute_pos_weights(double* x,void* l_data){
  double zMin = 0.0;
  pointlike_lighting_t* ll = (pointlike_lighting_t*)l_data;
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  double dot = r3_dot(&(ll->view_dir),&normal);
  double wraw = fmax(0,( dot- zMin)/(1.0 - zMin));
  return   pow(wraw,2.0);
}

/*Polynomial model*/


void harmonic_write_params(FILE* arq,void* l_data){
  
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  fprintf(arq," %d\n",pl->num_comp);
  int i;
  for(i = 0; i < pl->num_comp;i++){
    fprintf(arq, "%8.6lf %8.6lf %8.6lf %8.6lf\n",pl->coeficients[i],pl->triplets[i][0],pl->triplets[i][1],pl->triplets[i][2] );
  }
  
}

void* harmonic_read_params(FILE* arq){
  
  harmonic_lighting_t* pl = (harmonic_lighting_t*) malloc(sizeof(harmonic_lighting_t));
  fscanf(arq,"%d",&(pl->num_comp));
  int i;
  int num_comp = pl->num_comp;
  pl->coeficients = (double*)malloc(sizeof(double)*num_comp);
  pl->triplets = (double**)malloc(sizeof(double*)*num_comp);
  for(i = 0; i < pl->num_comp;i++){
    pl->triplets[i] = (double*)malloc(sizeof(double)*3);
    fscanf(arq, "%lf %lf %lf %lf",&(pl->coeficients[i]),&(pl->triplets[i][0]),&(pl->triplets[i][1]),&(pl->triplets[i][2]) );
  }
  
  return pl;
  
}



double harmonic_phi(int r, double* x, void* l_data){
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  int ri = pl->triplets[r][0];
  int rj = pl->triplets[r][1];
  int rk = pl->triplets[r][2];
  double phiR = (pow(normal.c[0],ri))*(pow(normal.c[1],rj))*(pow(normal.c[2],rk));
  //phiR = phiR*pl->weights[r];
  return phiR;
}

int harmonic_get_number_components(void* l_data){
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  return pl->num_comp;
}


harmonic_lighting_t* harmonic_init_components(int g,r3_t view_dir){
  
  harmonic_lighting_t* pl = (harmonic_lighting_t*)malloc(sizeof(harmonic_lighting_t));
  int num_comp = (g+1)*(g+1);
  pl->num_comp = num_comp;
  pl->view_dir = view_dir;
  pl->coeficients = (double*)malloc(sizeof(double)*num_comp);
  pl->triplets = (double**)malloc(sizeof(double*)*num_comp);
  
  int i,j,k;
  for(i = 0; i < num_comp; i++){
    pl->coeficients[i] = 1.0;
    pl->triplets[i] = (double*)malloc(sizeof(double)*3);
  }
  
  int count = 0;
  for(i = g; i >= 0; i--){
    for( j = g-i; j >= 0; j--){
      k =  g - i - j;
      pl->triplets[count][0] = i;
      pl->triplets[count][1] = j;
      pl->triplets[count][2] = k;
      count++;
    }
  }
  g = g -1;
  for(i = g; i >= 0; i--){
    for( j = g-i; j >= 0; j--){
      k =  g - i - j;
      pl->triplets[count][0] = i;
      pl->triplets[count][1] = j;
      pl->triplets[count][2] = k;
      count++;
    }
  }
  pl->num_comp = count;
  
  
  return pl;
  
}

void harmonic_retrieve_components(double* C, int n,void* l_data){
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  int i;
  assert(pl->num_comp >= n);
  for(i = 0; i < n; i++){
    pl->coeficients[i] = C[i];
  }
  for(i = i ; i < pl->num_comp; i++){
    pl->coeficients[i] = 0;
  }
  
}



double harmonic_shading(double* x,void* l_data){
  
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  int r;
  double val = 0;
  for(r = 0; r < pl->num_comp; r++){
    val+= (pl->coeficients[r]*harmonic_phi(r,x,l_data));
  }
  
  return val;
}


void harmonic_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma){
  double sumd2 = 0.0;
  double sumw = 1.0e-200;
  double sigma2 = *sigma;
  sigma2 = sigma2*sigma2;
  int i;
  for(i = 0; i<  n; i++){
  //  r3_t normal = (r3_t){{X[i][0],X[i][1],X[i][2]}};
    double computedGO = harmonic_shading(X[i],l_data);
    double originalGO = F[i];
    double d2 = (computedGO-originalGO)*(computedGO-originalGO);
    weights[i] = (d2 > 5.0*sigma2 ? 0.0 : exp(-d2/(2.0*sigma2)) );
    sumd2+= d2;
    sumw+=weights[i];
  }
  
  *sigma = sqrt(sumd2/(double)n);
}

void* harmonic_copy_lighting_data(void* l_data){
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  harmonic_lighting_t* npl = (harmonic_lighting_t*) malloc(sizeof(harmonic_lighting_t));
  npl->num_comp = pl->num_comp;
  npl->triplets = (double**)malloc(sizeof(double*)*(npl->num_comp));
  npl->coeficients  = (double*)malloc(sizeof(double)*(npl->num_comp));
  int i;
  for(i  = 0; i < npl->num_comp;i++){
    npl->coeficients[i] = pl->coeficients[i];
    npl->triplets[i] = (double*)malloc(sizeof(double)*3);
    int j;
    for(j = 0; j < 3;j++){
      npl->triplets[i][j] = pl->triplets[i][j]; 
    }
  }
  return npl;
}

double harmonic_compare_lightings(void* l_data1,void* l_data2){
  return 0.0;
}

void harmonic_release_lighting_data(void* l_data){
  harmonic_lighting_t* pl = (harmonic_lighting_t*) l_data;
  int i;
  for(i  = 0; i < pl->num_comp;i++){
    free(pl->triplets[i]);
  }
  free(pl->triplets);
  free(pl->coeficients);
  free(l_data);
}

double harmonic_compute_pos_weights(double* x,void* l_data){
  double zMin = 0.0;
  harmonic_lighting_t* ll = (harmonic_lighting_t*)l_data;
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  double dot = r3_dot(&(ll->view_dir),&normal);
  double wraw = fmax(0,( dot- zMin)/(1.0 - zMin));
  return   pow(wraw,2.0);
}


/*Radial Basis Model*/



double radialbasis_phi(int r, double* x, void* l_data){
  r3_t normal = (r3_t) {{x[0],x[1],x[2]}};
  radialbasis_lighting_t* rl = (radialbasis_lighting_t*) l_data;
  r3_t bcenter = rl->bcenter[r];
  double bradius = rl->bradius[r];
  double dot = r3_dot(&normal,&bcenter);
  double d2 = (1.0 - dot)/(bradius*bradius);
  if(d2 >= 1.0) return 0.0;
  return (1 - d2)*(1 - d2);
  
}

void radialbasis_write_params(FILE* arq,void* l_data){
  
  radialbasis_lighting_t* rl = (radialbasis_lighting_t*) l_data;
  fprintf(arq," %d\n",rl->num_comp);
  int i;
  for(i = 0; i < rl->num_comp;i++){
    fprintf(arq,"%8.6lf %8.6lf",rl->coeficients[i],rl->bradius[i]); 
    r3_gen_print(arq, &(rl->bcenter[i]), "%9.6lf", "  ", " ", "");
    fprintf(arq,"\n");
  }
  
}

void* radialbasis_read_params(FILE* arq){
  
  radialbasis_lighting_t* rl = (radialbasis_lighting_t*)malloc(sizeof(radialbasis_lighting_t));
  fscanf(arq,"%d",&(rl->num_comp));
  int i;
  int num_comp = rl->num_comp;
  rl->bradius = (double*)malloc(sizeof(double)*num_comp);
  rl->coeficients = (double*)malloc(sizeof(double)*num_comp);
  rl->bcenter = (r3_t*)malloc(sizeof(r3_t)*num_comp);
  
  for(i = 0; i < rl->num_comp;i++){
    fscanf(arq,"%lf %lf",&(rl->coeficients[i]),&(rl->bradius[i])); 
    //r3_gen_print(arq, &(rl->bcenter[i]), "%9.6lf", "  ", " ", "");
    fscanf(arq,"%lf %lf %lf",&(rl->bcenter[i].c[0]),&(rl->bcenter[i].c[1]),&(rl->bcenter[i].c[2])); 
//     fprintf(arq,"\n");
  }
  return rl;
  
}

int radialbasis_get_number_components(void* l_data){
  radialbasis_lighting_t* rl = (radialbasis_lighting_t*) l_data;
  return rl->num_comp;
}


radialbasis_lighting_t* radialbasis_init_components(int resolution,double span,r2_t center,double raio, r2_t estica,r3_t pole){
  	r2_t* pontos;
	r3_t* normals;
	int num_comp;
	
	gera_pontos_no_gabarito_eliptico(resolution,center,raio,estica,&pontos, &normals,&num_comp,pole);
//	gera_normais_no_gabarito(resolution,center, num_comp,pole);
	radialbasis_lighting_t* rl = (radialbasis_lighting_t*)malloc(sizeof(radialbasis_lighting_t));
	rl->num_comp = num_comp;
	rl->bradius = (double*)malloc(sizeof(double)*num_comp);
	rl->coeficients = (double*)malloc(sizeof(double)*num_comp);
	rl->bcenter = normals;
	rl->view_dir = pole;
	
	int i;
	for(i = 0; i < num_comp; i++){
		r3_t v = normals[i];
		rl->bradius[i] = span/(resolution*v.c[2]);
		rl->coeficients[i] = 1.0;
		// fprintf(stderr, "  %3d ", i);
		//r3_print(stderr, &((*bcenter)[i]));
		//fprintf(stderr, "  %9.6f\n", br[i]);
	}
	
	free(pontos);
	
	
		
	return rl;
}

void radialbasis_retrieve_components(double* C, int n,void* l_data){
  
  radialbasis_lighting_t* rl = (radialbasis_lighting_t*) l_data;
  int i;
  assert(rl->num_comp >= n);
  for(i = 0; i < n; i++){
    rl->coeficients[i] = C[i];
  }
  for(i = i ; i < rl->num_comp; i++){
    rl->coeficients[i] = 0;
  }
  
}

void radialbasis_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma){
  double sumd2 = 0.0;
  double sumw = 1.0e-200;
  double sigma2 = *sigma;
  sigma2 = sigma2*sigma2;
  int i;
  for(i = 0; i<  n; i++){
//    r3_t normal = (r3_t){{X[i][0],X[i][1],X[i][2]}};
    double computedGO = radialbasis_shading(X[i],l_data);
    double originalGO = F[i];
    double d2 = (computedGO-originalGO)*(computedGO-originalGO);
    weights[i] = (d2 > 5.0*sigma2 ? 0.0 : exp(-d2/(2.0*sigma2)) );
    sumd2+= d2;
    sumw+=weights[i];
  }
  
  *sigma = sqrt(sumd2/(double)n);
}

double radialbasis_shading(double* x,void* l_data){

  radialbasis_lighting_t* rl = (radialbasis_lighting_t*) l_data;
  int r;
  double val = 0;
  for(r = 0; r < rl->num_comp; r++){
    val+= (rl->coeficients[r]*radialbasis_phi(r,x,l_data));
  }
  
  return val;
}

void* radialbasis_copy_lighting_data(void* l_data){
  radialbasis_lighting_t* pl = (radialbasis_lighting_t*) l_data;
  radialbasis_lighting_t* npl = (radialbasis_lighting_t*) malloc(sizeof(radialbasis_lighting_t));
  npl->num_comp = pl->num_comp;
  npl->bcenter= (r3_t*)malloc(sizeof(r3_t)*(npl->num_comp));
  npl->coeficients  = (double*)malloc(sizeof(double)*(npl->num_comp));
  npl->bradius  = (double*)malloc(sizeof(double)*(npl->num_comp));
  int i;
  for(i  = 0; i < npl->num_comp;i++){
    npl->bradius[i] = pl->bradius[i];
    npl->bcenter[i] = pl->bcenter[i];
    npl->coeficients[i] = pl->coeficients[i];
  }
  return npl;
}

double radialbasis_compare_lightings(void* l_data1,void* l_data2){
  return 0.0;
}

void radialbasis_release_lighting_data(void* l_data){
  radialbasis_lighting_t* pl = (radialbasis_lighting_t*) l_data;
  free(pl->bradius);
  free(pl->bcenter);
  free(pl->coeficients);
  free(l_data);
}

double radialbasis_compute_pos_weights(double* x,void* l_data){
  double zMin = 0.0;
  radialbasis_lighting_t* ll = (radialbasis_lighting_t*)l_data;
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  double dot = r3_dot(&(ll->view_dir),&normal);
  double wraw = fmax(0,( dot- zMin)/(1.0 - zMin));
  return   pow(wraw,2.0);
}


/*Validation functions*/
int NoValidation(double* c, int basis_size, bool_t* validity_array){
  return basis_size;
}



glossylike_lighting_t* glossylike_init_components(double albedo,double g0,double g1,bool_t BPL, r3_t view_dir){
  glossylike_lighting_t* gl = (glossylike_lighting_t*)malloc(sizeof(glossylike_lighting_t));
  gl->view_dir = view_dir;
  gl->f =  (r3_t){{ 0,0,1}};
  gl->Ed = 0;
  gl->Eb = 0;
  gl->Ei = 0;
  gl->BPL = BPL;
  gl->albedo = albedo;
  gl->g0 = g0;
  gl->gI0 =  1.0;
  gl->g1 = g1;
  gl->gI1 =  1.0;
  return gl;
}

double glossylike_phi(int r, double* x, void* l_data){
  glossylike_lighting_t* gl = (glossylike_lighting_t*)l_data;
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  
  if(r <= 2){
    double s = r3_dot(&(gl->f), &normal);
    return (s <= 0 ? 0 : normal.c[r]);
  }else if(r == 3){
    return (gl->BPL ? 0.5*(1 - normal.c[2]) : 1);
  }else if((r >= 4) && (r<=5) ){
    double g;
    if(r == 4) g = gl->g0;
    if(r == 5) g = gl->g1;
    r3_t h;
    r3_t  unit_vec = (r3_t){{0,0,1}};
    r3_add (&(gl->f),&unit_vec,&h);
    r3_scale(0.5,&h,&h);
    double nh = r3_dot(&normal,&h);
    nh = pow(nh,exp(g));
    double nd = r3_dot(&normal,&(gl->f));
    return (nd > 0 ? nh*nd : 0);
  
  }else{
    assert(FALSE);
  }
    
}

int glossylike_get_number_components(void* l_data){
  return 5;
}

void glossylike_retrieve_components(double* C, int n,void* l_data){
  glossylike_lighting_t* gl = (glossylike_lighting_t*) l_data;
  gl->f.c[0] = C[0];
  gl->f.c[1] = C[1];
  gl->f.c[2] = C[2];
  gl->Ed = r3_dir(&(gl->f),&(gl->f))/gl->albedo;
  if(gl->BPL){
    gl->Eb = C[3]/gl->albedo;
    gl->Ei = 0;
  }else{
    gl->Eb = 0;
    gl->Ei = C[3]/gl->albedo;
  }
  
  /*Note - there is not update of the glossiness factor yet !*/
  gl->gI0 = C[4];
  gl->gI1 = C[5];
}

void glossylike_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma){
  int i;
  glossylike_lighting_t* ll = (glossylike_lighting_t*) l_data;
  for(i = 0; i < n; i++){
    r3_t normal = (r3_t) {{X[i][0],X[i][1],X[i][2]}};
    double s = r3_dot(&(ll->f), &normal);
    if(s > 0){
      weights[i] = 1.0;
    }else{
      weights[i] = 0.0;
    }
  }
}

double glossylike_shading(double* x,void* l_data){
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  
  glossylike_lighting_t* gl = (glossylike_lighting_t*) l_data;

  double vald = fmax(0,gl->Ed*r3_dot(&(gl->f),&normal));
  double valb = gl->Eb*(1-(normal.c[2]))/2.0;
  double vali = gl->Ei;
  
  double valg = 0;
  double nd = r3_dot(&normal,&(gl->f));
  if( nd > 0){
    r3_t  h;
    r3_t  unit_vec = (r3_t){{0,0,1}};
    r3_add (&(gl->f),&unit_vec,&h);
    r3_scale(0.5,&h,&h);
    double nh = r3_dot(&normal,&h);
    nh = pow(nh,exp(gl->g0));
    valg = gl->gI0*nh;
    
    r3_add (&(gl->f),&unit_vec,&h);
    r3_scale(0.5,&h,&h);
    nh = r3_dot(&normal,&h);
    nh = pow(nh,exp(gl->g1));
    valg+= gl->gI1*nh;
    
  }    
  double radiation = gl->albedo*(vald+valb+vali+valg);
  return radiation;
}

void glossylike_write_params(FILE* arq,void* l_data){
  glossylike_lighting_t* gl = (glossylike_lighting_t*)l_data;
  fprintf(arq, "  %8.6lf", gl->albedo);
  r3_gen_print(arq, &(gl->f), "%9.6lf", "  ", " ", "");
  fprintf(arq, "  %8.6lf %8.6lf %8.6lf", gl->Ed, gl->Eb, gl->Ei);
  fprintf(arq," %8.6lf %8.6lf %8.6lf %8.6lf",gl->g0,gl->gI0,gl->g1,gl->gI1);
  fprintf(arq, "\n");
}

void* glossylike_read_params(FILE* arq){
  glossylike_lighting_t* gl = (glossylike_lighting_t*) malloc(sizeof(glossylike_lighting_t));
  fscanf(arq, "%lf",&(gl->albedo));
  //r3_gen_print(arq, &(ll->f), "%9.6lf", "  ", " ", "");
  fscanf(arq,"%lf %lf %lf",&(gl->f.c[0]),&(gl->f.c[1]),&(gl->f.c[2]));
  fscanf(arq, "%lf %lf %lf", &(gl->Ed), &(gl->Eb), &(gl->Ei));
  
  fscanf(arq,"%lf %lf %lf %lf",&(gl->g0),&(gl->gI0),&(gl->g1),&(gl->gI1));
  
  if(gl->Eb == 0.0){
    gl->BPL = FALSE;
  }else{
    gl->BPL = TRUE;
  }
  return gl;
}

void* glossylike_copy_lighting_data(void* l_data){
  glossylike_lighting_t* novo = (glossylike_lighting_t*)malloc(sizeof(glossylike_lighting_t));
  glossylike_lighting_t* data = (glossylike_lighting_t*)l_data;
  *novo = *data;
  return novo;
}

double glossylike_compare_lightings(void* l_data1,void* l_data2){
  glossylike_lighting_t* ll1 = (glossylike_lighting_t*) l_data1;
  glossylike_lighting_t* ll2 = (glossylike_lighting_t*) l_data2;
  r3_t f = ll1->f;
  r3_t f_new = ll2->f;
  
  double vf[] = {f.c[0],f.c[1],f.c[2],ll1->gI0,ll1->gI1};
  double vf_new[] = {f_new.c[0],f_new.c[1],f_new.c[2],ll2->gI0,ll2->gI1};;
  
  double diff = rn_dist(5,vf, vf_new);
  return diff;
}

void glossylike_release_lighting_data(void* l_data){
  free(l_data);
}

double glossylike_compute_pos_weights(double* x,void* l_data){
  double zMin = 0.0;
  glossylike_lighting_t* ll = (glossylike_lighting_t*)l_data;
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  double dot = r3_dot(&(ll->view_dir),&normal);
  double wraw = fmax(0,( dot- zMin)/(1.0 - zMin));
  return   pow(wraw,2.0);
}

/*Stereographic projection of normals based on polynoms*/


double harmonicSG_phi(int r, double* x, void* l_data){
  /*Stereographic coordinates*/
  double X,Y;
  X = x[0]/(x[2] + 1.0);
  Y = x[1]/(x[2] + 1.0);
  
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  int ri = pl->tuples[r][0];
  int rj = pl->tuples[r][1];
  double phiR = (pow(X,ri))*(pow(Y,rj));
  //phiR = phiR*pl->weights[r];
  return phiR;
}

int harmonicSG_get_number_components(void* l_data){
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  return pl->num_comp;
}


harmonicSG_lighting_t* harmonicSG_init_components(int g,r3_t view_dir){
  
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*)malloc(sizeof(harmonicSG_lighting_t));
  int num_comp = (g+1)*(g+2)/2.0;
  pl->num_comp = num_comp;
  pl->view_dir = view_dir;
  pl->coeficients = (double*)malloc(sizeof(double)*num_comp);
  pl->tuples = (double**)malloc(sizeof(double*)*num_comp);
  
  int k,i,j;
  for(i = 0; i < num_comp; i++){
    pl->coeficients[i] = 1.0;
    pl->tuples[i] = (double*)malloc(sizeof(double)*2);
  }
  
  int count = 0;
  
  //fprintf(stderr,"TESTE");
  
  for(k = 0; k <= g; k++){
    for(i = k; i >= 0; i--){
      j = k - i ;
      pl->tuples[count][0] = i;
      pl->tuples[count][1] = j;
      //fprintf(stderr,"Seq [%02d]: %d %d \n",count,i,j);
      count++;
    }
  }
  
  //fprintf(stderr,"FIM");
//   g = g -1;
//   for(i = g; i >= 0; i--){
//     j = g - i;
//     pl->tuples[count][0] = i;
//     pl->tuples[count][1] = j;
//     count++;
//   }
  pl->num_comp = count;
  
  
  return pl;
  
}

void harmonicSG_retrieve_components(double* C, int n,void* l_data){
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  int i;
  assert(pl->num_comp >= n);
  for(i = 0; i < n; i++){
    pl->coeficients[i] = C[i];
  }
  for(i = i ; i < pl->num_comp; i++){
    pl->coeficients[i] = 0;
  }
  
}



double harmonicSG_shading(double* x,void* l_data){
  
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  int r;
  double val = 0;
  for(r = 0; r < pl->num_comp; r++){
    val+= (pl->coeficients[r]*harmonicSG_phi(r,x,l_data));
  }
  
  return val;
}


void harmonicSG_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma){
  double sumd2 = 0.0;
  double sumw = 1.0e-200;
  double sigma2 = *sigma;
  sigma2 = sigma2*sigma2;
  int i;
  for(i = 0; i<  n; i++){
  //  r3_t normal = (r3_t){{X[i][0],X[i][1],X[i][2]}};
    double computedGO = harmonicSG_shading(X[i],l_data);
    double originalGO = F[i];
    double d2 = (computedGO-originalGO)*(computedGO-originalGO);
    weights[i] = (d2 > 5.0*sigma2 ? 0.0 : exp(-d2/(2.0*sigma2)) );
    sumd2+= d2;
    sumw+=weights[i];
  }
  
  *sigma = sqrt(sumd2/(double)n);
}

void* harmonicSG_copy_lighting_data(void* l_data){
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  harmonicSG_lighting_t* npl = (harmonicSG_lighting_t*) malloc(sizeof(harmonicSG_lighting_t));
  npl->num_comp = pl->num_comp;
  npl->tuples = (double**)malloc(sizeof(double*)*(npl->num_comp));
  npl->coeficients  = (double*)malloc(sizeof(double)*(npl->num_comp));
  int i;
  for(i  = 0; i < npl->num_comp;i++){
    npl->coeficients[i] = pl->coeficients[i];
    npl->tuples[i] = (double*)malloc(sizeof(double)*2);
    int j;
    for(j = 0; j < 2;j++){
      npl->tuples[i][j] = pl->tuples[i][j]; 
    }
  }
  return npl;
}

double harmonicSG_compare_lightings(void* l_data1,void* l_data2){
  return 0.0;
}

void harmonicSG_release_lighting_data(void* l_data){
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  int i;
  for(i  = 0; i < pl->num_comp;i++){
    free(pl->tuples[i]);
  }
  free(pl->tuples);
  free(pl->coeficients);
  free(l_data);
}

double harmonicSG_compute_pos_weights(double* x,void* l_data){
  double zMin = 0.0;
  harmonicSG_lighting_t* ll = (harmonicSG_lighting_t*)l_data;
  r3_t normal = (r3_t){{x[0],x[1],x[2]}};
  double dot = r3_dot(&(ll->view_dir),&normal);
  double wraw = fmax(0,( dot- zMin)/(1.0 - zMin));
  return   pow(wraw,2.0);
}

void harmonicSG_write_params(FILE* arq,void* l_data){
  
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) l_data;
  fprintf(arq," %d\n",pl->num_comp);
  int i;
  for(i = 0; i < pl->num_comp;i++){
    fprintf(arq, "%8.6lf %8.6lf %8.6lf\n",pl->coeficients[i],pl->tuples[i][0],pl->tuples[i][1] );
  }
  
}

void* harmonicSG_read_params(FILE* arq){
  
  harmonicSG_lighting_t* pl = (harmonicSG_lighting_t*) malloc(sizeof(harmonicSG_lighting_t));
  fscanf(arq,"%d",&(pl->num_comp));
  int i;
  int num_comp = pl->num_comp;
  pl->coeficients = (double*)malloc(sizeof(double)*num_comp);
  pl->tuples = (double**)malloc(sizeof(double*)*num_comp);
  for(i = 0; i < pl->num_comp;i++){
    pl->tuples[i] = (double*)malloc(sizeof(double)*2);
    fscanf(arq, "%lf %lf %lf",&(pl->coeficients[i]),&(pl->tuples[i][0]),&(pl->tuples[i][1]));
  }
  
  return pl;
  
}


