#ifndef __MODELS_H__
#define __MODELS_H__
#include <r2.h>
#include <r3.h>
#include <ls_system.h>



typedef enum{ POINTLIKE_MODEL, HARMONIC_MODEL, RADIALBASIS_MODEL,GLOSSYLIKE_MODEL, HARMONICSG_MODEL } lighting_type_t;
/*Generic Lighting model functions*/
ls_model_t* create_ls_lighting_model(lighting_type_t type);

/*Pointlike Model Functions*/
struct Pointlike_Lighting_t{
  r3_t f; /*Light source direction*/
  double Ed;/*Light source intensity*/
  double Eb;/*Back Plane light intensity*/
  double Ei;/*Isotropic term intensity*/
  bool_t BPL;/*Enables/disables BackPlane Ligthing model*/
  double albedo; /*Gauge albedo*/
  r3_t view_dir;
};

typedef struct Pointlike_Lighting_t pointlike_lighting_t;

pointlike_lighting_t* pointlike_init_components(double albedo,bool_t BPL,r3_t view_dir);
/*The lighting model initialization cannot be not generic, since each one has its init params
For the Pointlike case, they are the gauge albedo{albedo} and if the model supports BackPlane Lighting or not*/

double pointlike_phi(int r, double* x,void* l_data);
int pointlike_get_number_components(void* l_data);
void pointlike_retrieve_components(double* C, int n,void* l_data);
void pointlike_update_weights(void* l_data,double** X, double* F,double* weights, int n,double* sigma);
double pointlike_shading(double* x,void* l_data);
void pointlike_write_params(FILE* arq,void* l_data);
void* pointlike_copy_lighting_data(void* l_data);
double pointlike_compare_lightings(void* l_data1,void* l_data2);
void pointlike_release_lighting_data(void* l_data);
double pointlike_compute_pos_weights(double* x,void* l_data);
void* pointlike_read_params(FILE* arq);

/*Harmonic Model Functions*/

struct Harmonic_Lighting_t{
  int num_comp; /*Number of triplests in the harmonic function*/
  double** triplets; /*Set of triplets of the harmonic function*/
  double* coeficients /*Value of each coeficient of the harmonic function*/;
  r3_t view_dir;
};

typedef struct Harmonic_Lighting_t harmonic_lighting_t;

harmonic_lighting_t* harmonic_init_components(int g,r3_t view_dir);
/*Determines the degree of the triplets of the harmonic model*/

double harmonic_phi(int r, double* x, void* l_data);
int harmonic_get_number_components(void* l_data);
void harmonic_retrieve_components(double* C, int n,void* l_data);
void harmonic_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma);
double harmonic_shading(double* x,void* l_data);
void harmonic_write_params(FILE* arq,void* l_data);
void* harmonic_read_params(FILE* arq);
void* harmonic_copy_lighting_data(void* l_data);
double harmonic_compare_lightings(void* l_data1,void* l_data2);
void harmonic_release_lighting_data(void* l_data);
double harmonic_compute_pos_weights(double* x,void* l_data);

/*Radial Basis model*/

struct RadialBasis_Lighting_t{
  int num_comp; /*Number of radial bases*/
  r3_t* bcenter; /*centers of each radial basis coeficient*/
  double* bradius;/*radius of each radial basis coeficient*/
  double* coeficients; /*ditto*/
  r3_t view_dir;
};

typedef struct RadialBasis_Lighting_t radialbasis_lighting_t;

radialbasis_lighting_t* radialbasis_init_components(int resolution,double span,r2_t center,double raio, r2_t estica,r3_t pole);
/*Given Resolution, Span, center, radius,stretch and "north pole" direction, generate the terms of the radial basis */

double radialbasis_phi(int r, double* x, void* l_data);
int radialbasis_get_number_components(void* l_data);
void radialbasis_retrieve_components(double* C, int n,void* l_data);
void radialbasis_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma);
double radialbasis_shading(double* x,void* l_data);
void radialbasis_write_params(FILE* arq,void* l_data);
void* radialbasis_read_params(FILE* arq);
void* radialbasis_copy_lighting_data(void* l_data);
double radialbasis_compare_lightings(void* l_data1,void* l_data2);
void radialbasis_release_lighting_data(void* l_data);
double radialbasis_compute_pos_weights(double* x,void* l_data);

/*Glossy surface model*/

struct GlossyLike_lighting_t{
  double g0; /*Degree of glossiness, as much higher more spyke-like it is*/
  double g1; /*Degree of glossiness, as much higher more spyke-like it is*/
  r3_t f; /*Light source direction*/
  double Ed;/*Light source intensity*/
  double Eb;/*Back Plane light intensity*/
  double Ei;/*Isotropic term intensity*/
  bool_t BPL;/*Enables/disables BackPlane Ligthing model*/
  double albedo; /*Gauge albedo*/
  double gI0; /*gloss intensity*/
  double gI1; /*gloss intensity*/
  r3_t view_dir;
  
};

typedef struct GlossyLike_lighting_t glossylike_lighting_t;

glossylike_lighting_t* glossylike_init_components(double albedo,double g0, double g1,bool_t BPL ,r3_t view_dir);
/*Given a degree of glossiness and a view_dir generate the initial parameters */

double glossylike_phi(int r, double* x, void* l_data);
int glossylike_get_number_components(void* l_data);
void glossylike_retrieve_components(double* C, int n,void* l_data);
void glossylike_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma);
double glossylike_shading(double* x,void* l_data);
void glossylike_write_params(FILE* arq,void* l_data);
void* glossylike_read_params(FILE* arq);
void* glossylike_copy_lighting_data(void* l_data);
double glossylike_compare_lightings(void* l_data1,void* l_data2);
void glossylike_release_lighting_data(void* l_data);
double glossylike_compute_pos_weights(double* x,void* l_data);


/*Harmonic Stereographic Model Functions*/

struct HarmonicSG_Lighting_t{
  int num_comp; /*Number of tuples in the harmonic function*/
  double** tuples; /*Set of tuples of the harmonic function*/
  double* coeficients /*Value of each coeficient of the harmonic function*/;
  r3_t view_dir;
};

typedef struct HarmonicSG_Lighting_t harmonicSG_lighting_t;

harmonicSG_lighting_t* harmonicSG_init_components(int g,r3_t view_dir);
/*Determines the degree of the triplets of the harmonic model*/

double harmonicSG_phi(int r, double* x, void* l_data);
int harmonicSG_get_number_components(void* l_data);
void harmonicSG_retrieve_components(double* C, int n,void* l_data);
void harmonicSG_update_weights(void* l_data,double** X, double* F,double* weights, int n,double * sigma);
double harmonicSG_shading(double* x,void* l_data);
void harmonicSG_write_params(FILE* arq,void* l_data);
void* harmonicSG_read_params(FILE* arq);
void* harmonicSG_copy_lighting_data(void* l_data);
double harmonicSG_compare_lightings(void* l_data1,void* l_data2);
void harmonicSG_release_lighting_data(void* l_data);
double harmonicSG_compute_pos_weights(double* x,void* l_data);


  /*Validation functions*/

int NoValidation(double* c, int basis_size, bool_t* validity_array);
/*Dummy function that aways returns as valid a given array */
#endif
