#ifndef __LSSYSTEM_H__
#define __LSSYSTEM_H__

#include <stdio.h>
#include <bool.h>

/*This library provides a LS fitting function including support to weights.
It rellies in callback functions defined bellow, it is up to the library which
defines the desired model to initialize ALL the attributes of the ls_model_t structure.
Please reffer to lighting_models.h and lighting_models.c for a example */


/*Types of each function*/
typedef double phi_function(int i, double* x,void* l_data);
/*Given the coeficient number{i}, the argument vector {x} and given lighting model {l_data} computes the value of Phi{i}*/
typedef int number_components_function(void* l_data);
/*Give the number of the coeficients for the LS system*/
typedef void retrieve_function(double* C, int n,void* l_data);
/*Given the {n} computed LS coeficients {C}, returns the values of the lighting model ${l_data}*/
typedef void update_weight_function(void* l_data,double** X, double* F,double* weights, int n,double* sigma);
/*updates the weights {weights} and value of ${sigma} given the lighting model {l_data} values and
{X} and F(X)*/
typedef double evaluate_function(double* x,void* l_data);
/*Given a normal and the lighting model, returns its light intensity*/
typedef void write_function(FILE* arq,void* l_data);
/*writes {l_data} parameters into {arq} the parameters in humam-readable fashion*/
typedef void* read_function(FILE* arq);
/*reads {l_data} parameters from {arq} the parameters in humam-readable fashion written by write_function*/
typedef int validate_function(double* c, int basis_size, bool_t* validity_array) ;
/*Given a coeficient array and its size, fills in a validity array wich tells wich coeficients are valid*/
typedef void* copy_data_function(void* l_data);
/*Given a lighting_data structure creates a copy of it with same parameters*/
typedef double compare_function(void* l_data1,void* l_data2);
/*Given two lighting data structures , compute its difference*/
typedef void release_data_function(void* l_data);
/*Releases a given lighting data structure*/
typedef double compute_pos_weights_function(double*x, void* l_data);
/*Compute the weight for the LS system given a determined x value of the domain of F*/



/*LS generic model fitting data structure, contain all needed callback functions*/
struct LS_Model_t{
  /*Mandatory members -  without then the fitting procedures will crash ! */
  
  int type; /*This is meant to be used for debugging and enum types*/
//  void* ls_data; /*this contain the data structure which defines the modeled function F, stores initially the initial  guess and in the end the fitted function*/
  phi_function* phi; /*Phi function*/ 
  number_components_function* get_num_components; /*retrieve the number of phi funcions defined by the model*/
  retrieve_function* retrieve_components; /*Computes the model function from the solved LS system*/
  update_weight_function* update_weights; /*update the weights of the A matrix*/
  validate_function* validate_results; /*Discards the undesirable members of A*/
  copy_data_function* copy_data; /*Make a copy of ls_data, it is needed to "save" the results of an previous iteration*/
  compare_function* compare; /*compare the results of different iterations*/
  release_data_function* release_data; /*Release a ls_data pointer from memory, mandatory if you dont wish to have your memory taken by multiple iterations !*/
  compute_pos_weights_function* compute_pos_weights; /*compute the value of positional weights*/
  /*weights - are all initialized and filled by the fitting functions*/
  int num_weights; /*num of weights of each value of F used to perform the fitting*/
  double* weights; /*array which stores the weights of F values*/
  double* wpos /*position dependent weights, may be meaningless depending of your function*/;
  /*Optional members - they wont be called by the fitting functions and can be NULL*/
  evaluate_function* evaluate; /* compute the value of the fitting function, it is optional*/
  write_function*  write_param; /*write the params in human readable manner into a file*/
  read_function*  read_param; /*read the params from a file in human readable manner*/
  
};

typedef struct LS_Model_t ls_model_t;

void computeLeastSquaresMatrix(double* A,phi_function* phi, int basis_size, double** X,int n, double w[], double wpos[],void* l_data);
/*Computes the Least Square matrix A using the weights,n-dimensional X values and current values of the function.*/

void computeLeastSquaresRHSVector(double* b,phi_function* phi, int basis_size,double** X,double* F,int n , double w[], double wpos[],void* l_data);
/*Same above, but for RHS vector*/

void computeLSTerms(double* A,
	double* b ,
	double* c,
	phi_function* phi,
	int basis_size,
	validate_function* validateResults,
	double** X,
	double* F,
	int n,
	double* w,
	double* wpos,
	void* l_data
	);
	
/*Compute the LS terms for the given matrix {A}, RHS vector {b} and stores it in {c}, weights and current l*/

void fitModelToFunction(double** X, double* F, int n,ls_model_t* lm,void* l_data,int update_steps);
/*Given the set of points X, F(x) values, and ls_model  structure {lm} and initial guess ${l_data} and number of steps, fits the model 
using weighted LS , results are stored into {l_data}*/
#endif     